/*====================================+=====================================+
! File CFileList_Do.cpp               ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! The redisplays necessary after Siren's operations are done through the    !
! fswatcher.                                                                !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/msgdlg.h>
#include <wx/textdlg.h>
#include <wx/choicdlg.h>
#include <wx/filedlg.h>
#include "common/msw/sr_ctxmnu.h"
#include "CFileList.h"
#include "CDirTree.h"
#include "CFileAccess.h"
#include "CApp.h"
#include "CRunInfoMessage.h"
#include "CDlgManRen.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
void CFileList::sort_file_list( e_sfl sfl )
{
   /*----------------------------------------------------------------------*/
   if( m_dir.empty() ) { return ; }

   /*----------------------------------------------------------------------*/
   CFileListDisplayState display_state ;
   CRunInfoMessage       msg( _( "Sorting ..." ) ) ;
   /*----------------------------------------------------------------------*/
   if( sfl == SFL_DO_RESFRESH )
   {  extract_display_state( display_state ) ; }

   /*----------------------------------------------------------------------*/
   m_dir.sort( wxGetApp().M_boo_natural_sort.get(),
               wxGetApp().M_boo_list_dir_top.get(),
               wxGetApp().M_i_sort_subitem.get(),
               get_subitem_sort( wxGetApp().M_i_sort_subitem.get() ),
               wxGetApp().M_boo_sort_asc.get(),
               msg
             ) ;
   /*----------------------------------------------------------------------*/
   if( sfl == SFL_DO_RESFRESH )
   {
      /*-------------------------------------------------------------------*/
      RefreshItems( 0, m_dir.size() - 1 ) ;
      apply_display_state( display_state ) ;
      wxGetApp().m_frame->status_disp_text() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! After selection/recomputation some columns content may change (NEW NAME   !
! and SEL). If one of them is the sorting item, a new sort has to be done.  !
+--------------------------------------------------------------------------*/
void CFileList::sort_file_list_after_selection_change()
{
   /*----------------------------------------------------------------------*/
   switch( wxGetApp().M_i_sort_subitem.get() )
   {
      /*-------------------------------------------------------------------*/
      case COL_NONE_NEW_NAME :
      case COL_NONE_SEL_NUM :
         sort_file_list( SFL_DO_RESFRESH ) ;
         break ;
      /*-------------------------------------------------------------------*/
      default :
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--( "execution" of highlighted elements )--------------------------------*/
void CFileList::exec_hil()
{
   /*--( Looking for the first highlighted item )--------------------------*/
   long l_num = get_next_hil_item( -1 ) ;

   /*----------------------------------------------------------------------*/
   if( l_num < 0 ) { return ; }

   /*----------------------------------------------------------------------*/
   if( m_dir[ l_num ].is_dir() )
   {  /*--( wxFileName will deal with the last '\', if any )---------------*/
      wxGetApp().m_frame->change_directory(
                                   wxFileName( wxGetApp().M_s_dir.get(),
                                               m_dir[ l_num ].get_full_path()
                                             ).GetFullPath()
                                          ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*--( Exec on all selected files )-----------------------------------*/
      do
      {  /*----------------------------------------------------------------*/
         if( !m_dir[ l_num ].is_dir() )
         {  m_dir[ l_num ].exec_system_open() ; }
         /*----------------------------------------------------------------*/
      } while( ( l_num = get_next_hil_item( l_num ) ) != -1 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Deleting to trash is only available under MSW (for now)                   !
+--------------------------------------------------------------------------*/
void CFileList::del_hil()
{
   /*----------------------------------------------------------------------*/
   CLogGuiGrouped my_log ;
   long           l_num = -1 ;

   /*----------------------------------------------------------------------*/
   if(    wxGetApp().m_frame->m_loaddir.thread_running()
       || get_nb_hil() == 0
     )
   {  return ; }

#ifdef __WXMSW__
   /*----------------------------------------------------------------------*/
   wxString s_all_names ;
   /*----------------------------------------------------------------------*/
   while( ( l_num = get_next_hil_item( l_num ) ) != -1 )
   {
      /*-------------------------------------------------------------------*/
      if( !m_dir[ l_num ].is_selectable() ) { continue ; }
      /*--( Names are separated by '\0' )----------------------------------*/
      s_all_names.append( m_dir[ l_num ].get_abs_full_path().fn_str() ) ;
      s_all_names.append( 1, '\0' ) ;
      /*-------------------------------------------------------------------*/
   }
   /*--( The list end is marked by a '\0' )--------------------------------*/
   s_all_names.append( 1, '\0' ) ;

   /*----------------------------------------------------------------------*/
   SHFILEOPSTRUCT fop ;
   bool boo_trash = !wxGetKeyState( WXK_SHIFT ) ;
   /*----------------------------------------------------------------------*/
   SR_SET_0( fop ) ;
   /*----------------------------------------------------------------------*/
   fop.wFunc  = FO_DELETE ;
   fop.pFrom  = s_all_names.fn_str() ;
   fop.pTo    = NULL ;
   fop.fFlags = ( boo_trash ? FOF_ALLOWUNDO : 0 ) ;
   /*----------------------------------------------------------------------*/
   SHFileOperation( &fop ) ;
   /*----------------------------------------------------------------------*/
   if( fop.fAnyOperationsAborted )
   {  return ; }
   /*----------------------------------------------------------------------*/

#else // != __WXMSW__

   /*-----------------------------------------------------------------------+
   ! Confirmation asked emphasizing the fact that there is no "delete to    !
   ! trash" function.                                                       !
   +-----------------------------------------------------------------------*/
   if( wxMessageBox( wxString::Format( _( "Do you want to DEFINITIVELY "
                                          "remove %d file(s) ?"
                                        ),
                                       get_nb_hil()
                                     ) ,
                     _( "Delete confirmation" ), wxYES_NO
                   ) != wxYES
     )
   {  return ; }

   /*----------------------------------------------------------------------*/
   while( ( l_num = get_next_hil_item( l_num ) ) != -1 )
   {
      /*-------------------------------------------------------------------*/
      if( !m_dir[ l_num ].is_selectable() ) { continue ; }
      /*-------------------------------------------------------------------*/
      if( m_dir[ l_num ].do_delete() != 0 )
      {  m_dir[ l_num ].set_oper_err( my_log.get_last_err() ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Flush done before refresh to let user see the problems if any )---*/
   my_log.Flush() ;
   /*-----------------------------------------------------------------------+
   ! Keep the errors after the loading that may have occurred during the    !
   ! delete.                                                                !
   +-----------------------------------------------------------------------*/
   wxGetApp().m_frame->m_loaddir_param.set_keep_oper_err( true ) ;
   /*----------------------------------------------------------------------*/
#endif // __WXMSW__

   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::manual_rename( long l_item )
{
   /*----------------------------------------------------------------------*/
   CFile *p_file = m_dir.get_p_file( l_item ) ;
   /*--( Ok to edit ? )----------------------------------------------------*/
   if(    p_file == NULL
       || !p_file->is_selectable()
       || !wxGetApp().m_frame->get_fswatcher_active()
     )
   {  return ; }

   /*----------------------------------------------------------------------*/
   CDlgManRen dlg( this ) ;
   wxString   s_name = p_file->get_full_path() ;

   /*----------------------------------------------------------------------*/
   if( dlg.get_modal( s_name ) != wxID_OK || s_name.empty() )
   {  return ; }

   /*--( Some cleaning )---------------------------------------------------*/
   sr::replace_filename_forbidden_chars( s_name,
                                         wxGetApp().M_s_rep_char_inv.get()
                                       ) ;
   /*--( Something to do ? )-----------------------------------------------*/
   if( s_name.Trim().empty() ) { return ; }

   /*----------------------------------------------------------------------*/
   wxFileName filename( s_name ) ;
   rename_one_file( l_item, filename ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
!                                                                           !
! The renaming can occur during a directory loading because the "post       !
! loading" should be in same thread as this function.                       !
!                                                                           !
+--------------------------------------------------------------------------*/

int CFileList::rename_one_file( int i_num_item, wxFileName &new_name )
{
   /*----------------------------------------------------------------------*/
   if( !wxGetApp().m_frame->get_fswatcher_active() )
   {  return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   CFile *p_file = m_dir.get_p_file( i_num_item ) ;
   if( p_file == NULL ) { wxFAIL ; return( -1 ) ; }

   /*--( Keep track of the operation )-------------------------------------*/
   m_vec_otfile.push_back( OFILE_RENAME ) ;
   m_i_num_otfile = m_vec_otfile.size()   ;

   /*----------------------------------------------------------------------*/
   CLogFileOper my_log ;
   int          i_ret  ;

   /*----------------------------------------------------------------------*/
   i_ret = p_file->oper_do( m_i_num_otfile, OFILE_RENAME, new_name ) ;
   /*----------------------------------------------------------------------*/
   if( i_ret < 0 )
   {  /*-------------------------------------------------------------------*/
      p_file->set_oper_err( my_log.get_last_err() ) ;
      sr::error_message( my_log.get_last_err() ) ;
      RefreshItem( i_num_item ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( If not cleared then the message will appear during destruction )--*/
   my_log.Clear() ;

   /*-----------------------------------------------------------------------+
   ! If non zero then the renaming didn't happen. The operation has to be   !
   ! erased.                                                                !
   +-----------------------------------------------------------------------*/
   if( i_ret != 0 )
   {
      /*-------------------------------------------------------------------*/
      m_vec_otfile.pop_back() ;
      m_i_num_otfile = m_vec_otfile.size() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileList::oper_do( e_otfile otfile )
{
   /*----------------------------------------------------------------------*/
   int             i_sel_num ;
   int             i_num ;
   bool            boo_something_done = false ;
   CRunInfoMessage msg ;

   /*-----------------------------------------------------------------------+
   ! The monitoring must be up except during "command line" renaming : the  !
   ! initialisation is not done                                             !
   +-----------------------------------------------------------------------*/
   if(    m_dir.get_nb_sel() <= 0
       || wxGetApp().M_s_exp.empty()
       ||    wxGetApp().m_frame->get_fswatcher_active()
          != wxGetApp().get_init_done()
     )
   {  return( 0 ) ; }

   /*--( Initialisation and parameter control ... )------------------------*/
   switch( otfile )
   {
      /*-------------------------------------------------------------------*/
      case OFILE_RENAME :
         msg.set_message( _( "Renaming ..." ), _( "Renaming (%d) ..." ) ) ;
         break ;
      /*-------------------------------------------------------------------*/
      case OFILE_COPY :
         msg.set_message( _( "Copying ..." ), _( "Copying (%d) ..." ) ) ;
         break ;
      /*-------------------------------------------------------------------*/
      default : return( -1 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   msg.display() ;

   /*--( Keep track of the operation )-------------------------------------*/
   m_vec_otfile.push_back( otfile ) ;
   m_i_num_otfile = m_vec_otfile.size() ;

   /*----------------------------------------------------------------------*/
   CLogFileOper my_log ;
   int          i_ret  ;

   /*----------------------------------------------------------------------*/
   for( i_sel_num = 0 ; i_sel_num < m_dir.get_nb_sel() ; )
   {
      /*-------------------------------------------------------------------*/
      i_num = m_dir.get_sel_file_num( i_sel_num ) ;
      /*-------------------------------------------------------------------*/
      i_ret = m_dir[ i_num ].oper_do( m_i_num_otfile, otfile,
                                      m_dir[ i_num ].get_new_name()
                                    ) ;
      /*-------------------------------------------------------------------*/
      if( i_ret == 0 )
      {  boo_something_done = true ; }
      else
      if( i_ret < 0 )
      {  m_dir[ i_num ].set_oper_err( my_log.get_last_err() ) ;
         RefreshItem( i_num ) ;
      }
      /*-------------------------------------------------------------------*/
      msg.display( ++i_sel_num ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   my_log.Flush() ;

   /*----------------------------------------------------------------------*/
   if( !boo_something_done )
   {  m_vec_otfile.pop_back() ; m_i_num_otfile = m_vec_otfile.size() ;
      msg.reset_display() ;
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileList::oper_undo()
{
   /*--( Undo possible ? )-------------------------------------------------*/
   if( m_i_num_otfile <= 0 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   int             i_num ;
   bool            boo_something_done = false ;
   e_otfile        otfile    = get_otfile_num( m_i_num_otfile ) ;
   CRunInfoMessage msg( _( "Undoing ..." ), _( "Undoing (%d) ..." ) ) ;

   /*----------------------------------------------------------------------*/
   CLogFileOper my_log ;
   int          i_ret  ;

   /*----------------------------------------------------------------------*/
   msg.display() ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; )
   {
      /*-------------------------------------------------------------------*/
      i_ret = m_dir[ i_num ].oper_undo( m_i_num_otfile, otfile ) ;
      /*-------------------------------------------------------------------*/
      if( i_ret == 0 )
      {  boo_something_done = true ; }
      else
      if( i_ret < 0 )
      {  m_dir[ i_num ].set_oper_err( my_log.get_last_err() ) ;
         RefreshItem( i_num ) ;
      }
      /*-------------------------------------------------------------------*/
      msg.display( ++i_num ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !boo_something_done ) { msg.reset_display() ; }
   /*----------------------------------------------------------------------*/
   my_log.Flush() ;

   /*--( Next undo )-------------------------------------------------------*/
   --m_i_num_otfile ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileList::oper_redo()
{
   /*----------------------------------------------------------------------*/
   if( m_i_num_otfile < 0 || m_i_num_otfile > ( int )m_vec_otfile.size() )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   int             i_num ;
   bool            boo_something_done = false ;
   e_otfile        otfile ;
   CRunInfoMessage msg( _( "Redoing ..." ), _( "Redoing (%d) ..." ) ) ;

   /*----------------------------------------------------------------------*/
   CLogFileOper my_log ;
   int          i_ret  ;

   /*----------------------------------------------------------------------*/
   ++m_i_num_otfile ;
   otfile = get_otfile_num( m_i_num_otfile ) ;
   /*----------------------------------------------------------------------*/
   msg.display() ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; )
   {
      /*-------------------------------------------------------------------*/
      i_ret = m_dir[ i_num ].oper_redo( m_i_num_otfile, otfile ) ;
      /*--( An error occured ? )-------------------------------------------*/
      if( i_ret == 0 )
      {  boo_something_done = true ; }
      else
      if( i_ret < 0 )
      {  m_dir[ i_num ].set_oper_err( my_log.get_last_err() ) ;
         RefreshItem( i_num ) ;
      }
      /*-------------------------------------------------------------------*/
      msg.display( ++i_num ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !boo_something_done ) { msg.reset_display() ; }
   /*----------------------------------------------------------------------*/
   my_log.Flush() ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static const char *st_co_p_c_cbc_text
           = wxTRANSLATE( "%d characters have been copied to the clipboard\n"
                          "(%d file(s) / %d column(s))"
                        ) ;
static const char *st_co_p_c_open_ko
                = wxTRANSLATE( "Unable to open \"%s\"" ) ;

/*-------------------------------------------------------------------------*/
void CFileList::info_message_clipboard( const wxString &s_msg,
                                        const wxString &s_data
                                      )
{
   /*-----------------------------------------------------------------------+
   ! The dialog will contain a button to save the data to a file.           !
   ! Under Windows the "X" close button in the title bar is enabled if      !
   ! wxCANCEL is part of the style.                                         !
   +-----------------------------------------------------------------------*/
   wxMessageDialog dlg( this, s_msg, _( "Clipboard" ),
                          wxOK | wxCANCEL
                        | wxCANCEL_DEFAULT | wxICON_INFORMATION
                      ) ;
   /*----------------------------------------------------------------------*/
   dlg.SetOKCancelLabels( _( "&Save" ), wxID_OK ) ;
   /*----------------------------------------------------------------------*/
   if( dlg.ShowModal() == wxID_OK )
   {
      /*-------------------------------------------------------------------*/
      wxFileDialog file_dialog( this, _( "Save clipboard content to file" ),
                                wxEmptyString, wxEmptyString,
                                wxFileSelectorDefaultWildcardStr,
                                wxFD_SAVE | wxFD_OVERWRITE_PROMPT
                              ) ;
      /*-------------------------------------------------------------------*/
      if( file_dialog.ShowModal() != wxID_OK )
      {  return ; }
      /*-------------------------------------------------------------------*/
      wxLogNull   no_log ;
      wxString    s_dst_filename = file_dialog.GetPath() ;
      CFileAccess fa_dst ;
      /*-------------------------------------------------------------------*/
      if( fa_dst.open_to_write( s_dst_filename ) != 0 )
      {  sr::error_message( wxString::Format(
                                       wxGetTranslation( st_co_p_c_open_ko ),
                                       s_dst_filename
                                            )
                          ) ;
         return ;
      }
      /*-------------------------------------------------------------------*/
      if( fa_dst.write_string( s_data ) != 0 )
      {  sr::error_message( wxString::Format( _( "Error writing to \"%s\"" ),
                                              s_dst_filename
                                            )
                          ) ;
         return ;
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::selected_list_to_clipboard()
{
   /*----------------------------------------------------------------------*/
   const int co_i_nb_col = get_nb_col() ;
   int       i_num_sel                  ;
   int       i_num                      ;
   int       i_num_col                  ;
   wxString  s_res                      ;
   wxUniChar c_sep                      ;
   wxUniChar c_repl_sep                 ;
   wxString  s_val                      ;

   /*--( Determination of the column separation character ? )--------------*/
   if(    wxGetApp().M_boo_cbc_sep_tab.get()
       || wxGetApp().M_s_cbc_sep_char.empty()
     )
   {  c_sep = '\t' ; }
   else
   {  c_sep = wxGetApp().M_s_cbc_sep_char.get()[ 0 ] ; }

   /*-----------------------------------------------------------------------+
   ! Determination of the replacement character of the separation character !
   ! if it appears in the data.                                             !
   +-----------------------------------------------------------------------*/
   if( wxGetApp().M_s_cbc_repl_sep_char.empty() )
   {  c_repl_sep = c_sep ; }
   else
   {  c_repl_sep = wxGetApp().M_s_cbc_repl_sep_char.get()[ 0 ] ; }

   /*--( Add the column headers )------------------------------------------*/
   for( i_num_col = COL_NONE_NAME ; i_num_col < COL_NB ; ++i_num_col )
   {
      /*-------------------------------------------------------------------*/
      if( !m_tb_col[ i_num_col ].displayed() ) { continue ; }
      /*-------------------------------------------------------------------*/
      if( i_num_col > COL_NONE_NAME ) { s_res += c_sep ; }
      /*-------------------------------------------------------------------*/
      s_val = m_tb_col[ i_num_col ].m_s_name ;
      sr::replace_chars( s_val, c_sep, c_repl_sep ) ;
      s_res += s_val ;
      /*-------------------------------------------------------------------*/
   }
   /*--( Record separator )------------------------------------------------*/
   s_res += SR_EOL  ;

   /*-----------------------------------------------------------------------+
   ! The info is extracted following the selection order, not the display   !
   ! order.                                                                 !
   +-----------------------------------------------------------------------*/
   for( i_num_sel = 0 ; i_num_sel < m_dir.get_nb_sel() ; ++i_num_sel )
   {
      /*-------------------------------------------------------------------*/
      i_num = m_dir.get_sel_file_num( i_num_sel ) ;

      /*--( For each file, the info are extracted one col after another )--*/
      for( i_num_col = COL_NONE_NAME ; i_num_col < co_i_nb_col ; ++i_num_col)
      {
         /*----------------------------------------------------------------*/
         if( i_num_col > COL_NONE_NAME ) { s_res += c_sep ; }
         /*--( The extracted information is the displayed one )------------*/
         s_val = OnGetItemText( i_num, i_num_col ) ;
         if( !s_val.empty() )
         {
            /*-------------------------------------------------------------*/
            sr::replace_chars( s_val, c_sep, c_repl_sep ) ;
            s_res += s_val ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*--( Record separator )---------------------------------------------*/
      s_res += SR_EOL  ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   sr::set_clipboard( s_res ) ;
   /*----------------------------------------------------------------------*/
   info_message_clipboard(
                    wxString::Format( wxGetTranslation( st_co_p_c_cbc_text ),
                                      ( int )s_res.size(),
                                      m_dir.get_nb_sel(),
                                      ( int )m_vec_i_col_subitem.size()
                                    ),
                    s_res
                         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( One column data of the selected files )------------------------------*/
wxString CFileList::col_selected_list_to_string( int i_si )
{
   /*----------------------------------------------------------------------*/
   int      i_num_sel ;
   int      i_num     ;
   wxString s_res     ;

   /*-----------------------------------------------------------------------+
   ! The info is extracted following the selection order, not display one   !
   +-----------------------------------------------------------------------*/
   for( i_num_sel = 0 ; i_num_sel < m_dir.get_nb_sel() ; ++i_num_sel )
   {
      /*-------------------------------------------------------------------*/
      i_num = m_dir.get_sel_file_num( i_num_sel ) ;
      /*--( Record separator )---------------------------------------------*/
      s_res += m_dir[ i_num ].get_s_val( i_si ) + SR_EOL ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*--( One column data of the selected files )------------------------------*/
void CFileList::col_selected_list_to_clipboard( int i_si )
{
   /*----------------------------------------------------------------------*/
   wxString s_res = col_selected_list_to_string( i_si ) ;
   /*----------------------------------------------------------------------*/
   sr::set_clipboard( s_res ) ;
   /*----------------------------------------------------------------------*/
   info_message_clipboard(
                    wxString::Format( wxGetTranslation( st_co_p_c_cbc_text ),
                                      ( int )s_res.size(),
                                      m_dir.get_nb_sel(), 1
                                    ),
                    s_res
                         ) ;
   /*----------------------------------------------------------------------*/
}

#ifdef __WXMSW__
/*--( All the selected files have to be in the same directory )------------*/
int CFileList::shell_contextual_menu()
{
   /*----------------------------------------------------------------------*/
   wxArrayString as_hil ;
   int           i_num  ;

   /*----------------------------------------------------------------------*/
   for( i_num = -1 ; ( i_num = get_next_hil_item( i_num ) ) >= 0 ; )
   {  as_hil.Add( m_dir[ i_num ].get_full_path() ) ; }

   /*--( No highlight => menu for current directory )----------------------*/
   if( as_hil.GetCount() == 0 )
   {  return( sr::shell_contextual_menu( wxGetApp().M_s_dir.get() ) ) ; }
   /*----------------------------------------------------------------------*/
   return( sr::shell_contextual_menu( wxGetApp().M_s_dir.get(), as_hil ) ) ;
   /*----------------------------------------------------------------------*/
}
#endif // __WXMSW__

/*--( Extraction of file start/end to disk. Normally for debug purpose )---*/
int CFileList::file_content_extraction()
{
   /*----------------------------------------------------------------------*/
   CFile *p_file ;
   const char *tb_tb_co_c_choice[]
       = { _( "First 100 KiB" ), _( "First 512 KiB" ), _( "First 1024 KiB" ),
           _( "Last 1024 KiB" ), _( "Last 512 KiB" ), _( "Last 100 KiB" )
         } ;
   int   i_choice ;
   int   i_nb_k   ;

   /*----------------------------------------------------------------------*/
   p_file = get_p_file_focus() ;
   if( p_file == NULL || p_file->is_dir() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
     i_choice
   = wxGetSingleChoiceIndex( _( "Origin and size of the extraction" ),
                             _( "File part extraction" ),
                             wxArrayString( WXSIZEOF( tb_tb_co_c_choice ),
                                            tb_tb_co_c_choice
                                          )
                           ) ;

   /*----------------------------------------------------------------------*/
   switch( i_choice )
   {  /*-------------------------------------------------------------------*/
      case 0 : i_nb_k =  +100 ; break ;
      case 1 : i_nb_k =  +512 ; break ;
      case 2 : i_nb_k = +1024 ; break ;
      case 3 : i_nb_k = -1024 ; break ;
      case 4 : i_nb_k =  -512 ; break ;
      case 5 : i_nb_k =  -100 ; break ;
      /*-------------------------------------------------------------------*/
      default : return( -2 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   int i_nb_byte = abs( i_nb_k ) * 1024 ;
   wxString s_dst_filename( wxString::Format( _( "%s_%dKiB.%s" ),
                            p_file->get_comp_basename( p_file->get_name() ),
                            i_nb_k,
                            p_file->get_comp_extension( p_file->get_name() )
                                            )
                          ) ;
   /*----------------------------------------------------------------------*/
   wxFileDialog file_dialog( this,
                  wxString::Format( _( "Extraction of the %s %d KiB of %s" ),
                                    i_nb_k > 0 ? _( "first" ) : _( "last" ),
                                    abs( i_nb_k ), p_file->get_full_path()
                                  ),
                             wxEmptyString, s_dst_filename,
                             wxFileSelectorDefaultWildcardStr,
                             wxFD_SAVE | wxFD_OVERWRITE_PROMPT
                           ) ;

   /*----------------------------------------------------------------------*/
   if( file_dialog.ShowModal() != wxID_OK )
   {  return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   wxLogNull   no_log ;
   CFileAccess fa_src ;
   CFileAccess fa_dst ;
   char tb_b_buffer[ 1024 * 10 ] ;
   bool boo_end ;
   /*----------------------------------------------------------------------*/
   if( fa_src.open_to_read( p_file->get_full_path() ) != 0 )
   {  sr::error_message( wxString::Format(
                                       wxGetTranslation( st_co_p_c_open_ko ),
                                       p_file->get_full_path()
                                         )
                       ) ;
      return( -4 ) ;
   }
   /*--( Extraction from the end ? )---------------------------------------*/
   if( i_nb_k < 0 && fa_src.Length() > i_nb_byte )
   {  /*-------------------------------------------------------------------*/
      if( fa_src.set_offset_from_end( i_nb_byte ) != 0 )
      {  return( -5 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   s_dst_filename = file_dialog.GetPath() ;
   if( fa_dst.open_to_write( s_dst_filename ) != 0 )
   {  sr::error_message( wxString::Format(
                                       wxGetTranslation( st_co_p_c_open_ko ),
                                       s_dst_filename
                                         )
                       ) ;
      return( -6 ) ;
   }
   /*----------------------------------------------------------------------*/
   for( boo_end = false ; !boo_end ; )
   {
      /*-------------------------------------------------------------------*/
      size_t sz_len = wxMin( i_nb_byte, ( int )sizeof( tb_b_buffer ) ) ;
      /*-------------------------------------------------------------------*/
      if( fa_src.read_buffer_max( sz_len, tb_b_buffer ) != 0 ) { break ; }
      /*-------------------------------------------------------------------*/
      i_nb_byte -= sz_len ;
      boo_end    = ( sz_len < sizeof( tb_b_buffer ) || i_nb_byte <= 0 ) ;
      /*-------------------------------------------------------------------*/
      if( fa_dst.write_buffer( tb_b_buffer, sz_len ) != 0 ) {  break ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   fa_src.close() ;
   fa_dst.close() ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                     End of file CFileList_Do.cpp                          !
+==========================================================================*/
