/*====================================+=====================================+
! File CFileList_Base.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifdef __WXMSW__
#include <wx/wx.h>
#include <windef.h>
#include "wx/chartype.h"
#include "wx/msw/wrapwin.h"
#include "wx/msw/wrapcctl.h"
#endif // __WXMSW__
#include <algorithm>
#include "common/sr_lib.h"
#include "CApp.h"
#include "CFileList.h"
#include "CRunInfoMessage.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CFileList::CFileList( wxWindow *parent, wxWindowID id )
         : CListCtrlBase( parent, id, wxDefaultPosition, wxDefaultSize,
                          wxLC_REPORT | wxLC_VIRTUAL
                        )
{  init() ; }

/*-------------------------------------------------------------------------*/
CFileList::~CFileList()
{  delete m_p_image_list ; }

/*-------------------------------------------------------------------------*/
CColumn *CFileList::get_column_with_subitem( int i_subitem, int i_from )
{
   /*----------------------------------------------------------------------*/
   CColumn *p_col
         = std::find( m_tb_col + i_from, m_tb_col + COL_NB, i_subitem ) ;
   /*----------------------------------------------------------------------*/
   if( p_col >= m_tb_col + COL_NB ) { wxFAIL ; return( NULL ) ; }
   /*----------------------------------------------------------------------*/
   return( p_col ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::set_column_image( int i_col_disp, int i_ind_img )
{
   /*----------------------------------------------------------------------*/
   wxListItem item ;
   /*----------------------------------------------------------------------*/
   item.SetImage( i_ind_img ) ;
   /*----------------------------------------------------------------------*/
   SetColumn( i_col_disp, item ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::set_column_sort_image( int i_col_disp, e_icfl icfl )
{
   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   /*----------------------------------------------------------------------*/
   HWND     hHeader = ListView_GetHeader( GetHandle() ) ;
   wxHDITEM hdi ;
   /*----------------------------------------------------------------------*/
   hdi.mask = HDI_FORMAT ;
   if( Header_GetItem( hHeader, i_col_disp, &hdi ) == FALSE )
   {  wxFAIL ; return ; }
   /*----------------------------------------------------------------------*/
   hdi.fmt &= ~( HDF_SORTUP | HDF_SORTDOWN ) ;
   /*----------------------------------------------------------------------*/
   switch( icfl )
   {  /*-------------------------------------------------------------------*/
      case ICFL_SORT_ASC  : hdi.fmt |= HDF_SORTUP   ; break ;
      case ICFL_SORT_DESC : hdi.fmt |= HDF_SORTDOWN ; break ;
      /*--( To avoid warning under gcc )-----------------------------------*/
      default : ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( Header_SetItem( hHeader, i_col_disp, &hdi ) == FALSE )
   {  wxFAIL ; return ; }
   /*----------------------------------------------------------------------*/
#else
   /*----------------------------------------------------------------------*/
   if( icfl == ICFL_SORT_ASC )
   {  set_column_image( i_col_disp, m_p_image_list->get_asc() ) ; }
   else
   if( icfl == ICFL_SORT_DESC )
   {  set_column_image( i_col_disp, m_p_image_list->get_desc() ) ; }
   else
   {  set_column_image( i_col_disp, -1 ) ; }
   /*----------------------------------------------------------------------*/
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::set_column_sort_image( int i_col_disp )
{
   /*--( Use current value ? It may not be displayed )---------------------*/
   if( i_col_disp < 0 )
   {    i_col_disp
      = get_col_num_with_subitem( wxGetApp().M_i_sort_subitem.get() ) ;
   }
   /*----------------------------------------------------------------------*/
   set_column_sort_image( i_col_disp,
                          wxGetApp().M_boo_sort_asc.get()
                          ? ICFL_SORT_ASC : ICFL_SORT_DESC
                        ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::remove_column_sort_image( int i_col_disp )
{  set_column_sort_image( i_col_disp, ICFL_NO_IMAGE ) ; }

/*--------------------------------------------------------------------------+
! Change sort order to the chosen displayed column.                         !
! Change column header display and internal data.                           !
+--------------------------------------------------------------------------*/
void CFileList::change_column_sort_image( int i_col_disp )
{
   /*----------------------------------------------------------------------*/
   int i_si_prv = wxGetApp().M_i_sort_subitem.get() ;
   int i_si_new = get_subitem_in_col( i_col_disp )  ;
   /*--( On the same column: just change the orientation )-----------------*/
   if( i_si_prv == i_si_new )
   {  /*-------------------------------------------------------------------*/
      wxGetApp().M_boo_sort_asc.set( !wxGetApp().M_boo_sort_asc.get() ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*--( Clear image from previous one )--------------------------------*/
      remove_column_sort_image( get_col_num_with_subitem( i_si_prv ) ) ;
      /*--( And set it on the new one )------------------------------------*/
      wxGetApp().M_boo_sort_asc.set( true ) ;
      wxGetApp().M_i_sort_subitem.set( i_si_new ) ;
      /*-------------------------------------------------------------------*/
   }
   /*--( Set the new image )-----------------------------------------------*/
   set_column_sort_image( i_col_disp ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileList::get_col_num_with_subitem( int i_si ) const
{
   /*----------------------------------------------------------------------*/
   sr::t_vec_int_cit cit = std::find( m_vec_i_col_subitem.begin(),
                                      m_vec_i_col_subitem.end(), i_si
                                    ) ;
   /*----------------------------------------------------------------------*/
   return( cit == m_vec_i_col_subitem.end()
           ? -1 : cit - m_vec_i_col_subitem.begin()
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::save_all_col_width()
{
   /*----------------------------------------------------------------------*/
   int     i_col  ;
   CColumn *p_col ;
   /*----------------------------------------------------------------------*/
   for( i_col = 0 ; i_col < ( int )m_vec_i_col_subitem.size() ; ++i_col )
   {
      /*-------------------------------------------------------------------*/
      p_col = &m_tb_col[ get_ind_col_disp_col( i_col ) ] ;
      /*-------------------------------------------------------------------*/
      if( p_col->displayed() && p_col->m_boo_width_modifiable )
      {  p_col->m_i_width = GetColumnWidth( p_col->m_i_col_disp ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::init_tb_subitem_col()
{
   /*----------------------------------------------------------------------*/
   int i_col ;
   /*----------------------------------------------------------------------*/
   for( i_col = 0 ; i_col < COL_NB ; ++i_col )
   {  m_tb_subitem_in_tb_col[ m_tb_col[ i_col ].m_i_subitem ] = i_col ; }
   /*----------------------------------------------------------------------*/
}

/*--( Apply column config information to internal data )-------------------*/
void CFileList::load_col_info()
{
   /*----------------------------------------------------------------------*/
   int      i_num ;
   wxString s_val ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < COL_NB ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      CColumn       col   ;
      CConfigString *p_cs ;

      /*--( As the array has been initialized the entry should be present )*/
      p_cs = wxGetApp().M_tb_col.get_pval( i_num ) ;
      /*-------------------------------------------------------------------*/
      if( p_cs == NULL ) { wxFAIL ; continue ; }

      /*--( If not defined in config then set it to defaults )-------------*/
      if( p_cs->empty() )
      {
         /*----------------------------------------------------------------*/
         p_cs->set( m_tb_col[ i_num ].datas_to_string() ) ;
         p_cs->set_to_save( false ) ;
         continue ;
         /*----------------------------------------------------------------*/
      }

      /*--( Transfer/convert config data to check and analyse )------------*/
      col.string_to_datas( p_cs->get() ) ;

      /*--------------------------------------------------------------------+
      ! Minimal control and check that the "forced displayed" columns can't !
      ! are "at their place" and displayed.                                 !
      +--------------------------------------------------------------------*/
      if(    (    m_tb_col[ i_num ].m_boo_on_forced
               && ( col.m_i_subitem != i_num || !col.m_boo_on )
             )
          || col.m_i_subitem <  0
          || col.m_i_subitem >= COL_NB
          || col.m_i_width   <= 0
          || col.m_i_width   >  10000
        )
      {  p_cs->del() ; continue ; }

      /*--( A subitem can't appear twice )---------------------------------*/
      if( i_num > 0 )
      {
         /*--( A specific "==" operator has been created )-----------------*/
         CColumn *p_col
               = std::find( m_tb_col, m_tb_col + i_num, col.m_i_subitem ) ;
         /*----------------------------------------------------------------*/
         if( p_col < m_tb_col + i_num - 1 )
         {  p_cs->del() ; continue ; }
         /*----------------------------------------------------------------*/
      }

      /*--( The column array follows the displayed one )-------------------*/
      if( m_tb_col[ i_num ].m_i_subitem != col.m_i_subitem )
      {
         /*----------------------------------------------------------------*/
         CColumn *p_col = get_column_with_subitem( col.m_i_subitem, i_num ) ;
         if( p_col == NULL )
         {  p_cs->del() ; continue ; }
         /*----------------------------------------------------------------*/
         std::swap( m_tb_col[ i_num ], *p_col ) ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      m_tb_col[ i_num ].string_to_datas( p_cs->get() ) ; ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   init_tb_subitem_col() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::transfert_col_info_to_cfg()
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < COL_NB ; ++i_num )
   {  wxGetApp().M_tb_col.set( i_num, m_tb_col[ i_num ].datas_to_string() );}
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::init_col_and_grp_used()
{
   /*----------------------------------------------------------------------*/
   int                i_num_file ;
   CFile::t_s_val_cit it_val     ;
   int                i_col      ;
   int                i_grp      ;

   /*--( Reset column "used" )---------------------------------------------*/
   for( i_col = 0 ; i_col < COL_NB ; ++i_col )
   {  m_tb_col[ i_col ].m_boo_used = false ; }
   /*--( Set column "used" )-----------------------------------------------*/
   for( i_num_file = 0 ; i_num_file < m_dir.size() ; ++i_num_file )
   {
      /*-------------------------------------------------------------------*/
      for( it_val  = m_dir[ i_num_file ].get_map_s_val().begin() ;
           it_val != m_dir[ i_num_file ].get_map_s_val().end() ;
           ++it_val
         )
      {  /*----------------------------------------------------------------*/
         wxASSERT( it_val->first >= 0 && it_val->first < COL_NB ) ;
         /*----------------------------------------------------------------*/
           m_tb_col[ m_tb_subitem_in_tb_col[ it_val->first ] ].m_boo_used
         = true ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Reset group "used" )----------------------------------------------*/
   for( i_grp = 0 ; i_grp < GRP_NB ; ++i_grp )
   {  wxGetApp().m_colgroup.set_used( i_grp, false ) ; }
   /*--( Set group "used" based on previous column results )---------------*/
   for( i_col = 0 ; i_col < COL_NB ; ++i_col )
   {  /*-------------------------------------------------------------------*/
      if( m_tb_col[ i_col ].m_boo_used )
      {  wxGetApp().m_colgroup.set_used( m_tb_col[ i_col ].m_grp, true ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Show group display status )---------------------------------------*/
   if( wxGetApp().M_boo_grp_auto.get() )
   {  wxGetApp().m_colgroup.synchro_used_disp_button() ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileList::delete_all_col()
{
   /*----------------------------------------------------------------------*/
   int i_col ;
   /*----------------------------------------------------------------------*/
   for( i_col = m_vec_i_col_subitem.size() - 1 ; i_col >= 0 ; --i_col )
   {
      /*-------------------------------------------------------------------*/
      if( DeleteColumn( i_col ) == false )
      {  wxFAIL ; return( -1 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_vec_i_col_subitem.clear() ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool CFileList::create_all_col( e_cac cac_force )
{
   /*----------------------------------------------------------------------*/
   int i_num ;

   /*-----------------------------------------------------------------------+
   ! Do it ?                                                                !
   ! If there is not change in the displayed/hidden column state, it is not !
   ! necessary to save their widths and recreate them                       !
   +-----------------------------------------------------------------------*/
   if( cac_force == CAC_DONT_FORCE_CREATION )
   {
      /*-------------------------------------------------------------------*/
      for( i_num = 0 ;
              i_num < COL_NB
           &&    m_tb_col[ i_num ].displayed()
              == m_tb_col[ i_num ].to_display() ;
           ++i_num
         )
      {  ; }
      /*-------------------------------------------------------------------*/
      if( i_num >= COL_NB ) { return( false ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   save_all_col_width() ;

   /*-----------------------------------------------------------------------+
   ! Freeze/Thaw is necessary at least under W7 because else the horizontal !
   ! scrollbar can disappear during the process.                            !
   +-----------------------------------------------------------------------*/
   Freeze() ;
   delete_all_col() ;

   /*-----------------------------------------------------------------------+
   ! Which columns have to be displayed ?                                   !
   +-----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < COL_NB ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if( !m_tb_col[ i_num ].to_display() )
      {  m_tb_col[ i_num ].reset_col_disp() ; }
      else
      {  /*----------------------------------------------------------------*/
         m_tb_col[ i_num ].m_i_col_disp = get_nb_col() ;
         /*--( Keep track of the displayed subitem )-----------------------*/
         m_vec_i_col_subitem.push_back( m_tb_col[ i_num ].m_i_subitem ) ;
         /*----------------------------------------------------------------*/
         if( InsertColumn( m_vec_i_col_subitem.size() - 1,
                           m_tb_col[ i_num ].m_s_name,
                           m_tb_col[ i_num ].m_i_fmt,
                           m_tb_col[ i_num ].m_i_width
                         ) == -1
           )
         {  wxFAIL ; return( false ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Use defaults if the sorting column is not present or if the columns    !
   ! have not been created yet.                                             !
   +-----------------------------------------------------------------------*/
   if( get_col_num_with_subitem( wxGetApp().M_i_sort_subitem.get() ) < 0 )
   {  /*-------------------------------------------------------------------*/
      wxGetApp().M_i_sort_subitem.set( COL_NONE_NAME ) ;
      wxGetApp().M_boo_sort_asc.set( true ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   set_column_sort_image() ;
   /*----------------------------------------------------------------------*/
   Thaw() ;
   /*----------------------------------------------------------------------*/
   return( true ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::turn_all_grp_col_off_except( int i_col )
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < COL_NB ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if(    i_num != i_col
          && m_tb_col[ i_num ].m_grp == m_tb_col[ i_col ].m_grp
        )
      {  m_tb_col[ i_num ].m_boo_on = false ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   create_all_col() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::turn_all_grp_col_on_off( e_grp grp, bool boo_on )
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < COL_NB ; ++i_num )
   {
       /*------------------------------------------------------------------*/
       if( m_tb_col[ i_num ].m_grp == grp )
       {  m_tb_col[ i_num ].m_boo_on = boo_on ; }
       /*------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   create_all_col() ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function exists because                                              !
! "wxListCtrl.SetColumnWidth( i, wxLIST_AUTOSIZE ) doesn't seem to work     !
! under GTK with virtual lists.                                             !
+--------------------------------------------------------------------------*/
void CFileList::col_auto_size( int i_col )
{
   /*----------------------------------------------------------------------*/
   const int co_i_si = m_tb_col[ i_col ].m_i_subitem ;
   int       i_width_max = 0 ;
   int       i_width         ;
   int       i_num           ;

   /*--( For some columns the width can't be changed -icon, check box ... )*/
   if( !m_tb_col[ i_col ].m_boo_width_modifiable )
   {  return ; }

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      i_width = col_width_for_text( m_dir[ i_num ].get_s_val( co_i_si ) ) ;
      /*-------------------------------------------------------------------*/
      if( i_width > i_width_max ) { i_width_max = i_width ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   SetColumnWidth( m_tb_col[ i_col ].m_i_col_disp, i_width_max ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::extract_display_state( CFileListDisplayState &state )
{
   /*----------------------------------------------------------------------*/
   long l_num ;
   /*----------------------------------------------------------------------*/
   if( ( l_num = get_item_focus() ) != -1 )
   {  state.m_s_focus = m_dir[ l_num ].get_full_path() ; }
   else
   {  state.m_s_focus.clear() ; }
   /*----------------------------------------------------------------------*/
   state.m_as_hil.clear() ;
   /*----------------------------------------------------------------------*/
   for( l_num = -1 ; ( l_num = get_next_hil_item( l_num ) ) != -1 ; )
   {  state.m_as_hil.Add( m_dir[ l_num ].get_full_path() ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::apply_display_state( const CFileListDisplayState &state )
{
   /*----------------------------------------------------------------------*/
   long l_num_focus = 0       ; // By default the focus is on the first file
   long l_nb_hil              ; // Number of highlighted files
   long l_num_file            ;
   bool m_boo_prev_disp_image ;

   /*--( Avoiding image display while changing focus )---------------------*/
   m_boo_prev_disp_image = wxGetApp().m_frame->get_disp_image() ;
   wxGetApp().m_frame->set_disp_image( false ) ;

   /*----------------------------------------------------------------------*/
   set_item_hil( -1, false ) ;
   l_nb_hil = 0 ;

   /*----------------------------------------------------------------------*/
   if( !state.m_s_focus.empty() )
   {
      /*--( The focus is not forced "visible" ... It can be annoying )-----*/
      if( ( l_num_focus = m_dir.file_with_name( state.m_s_focus ) ) != -1 )
      {  /*----------------------------------------------------------------*/
         set_item_focus( l_num_focus ) ;
         /*----------------------------------------------------------------*/
      }
      else
         /*-----------------------------------------------------------------+
         ! If the previously focused file is not present any more the new   !
         ! one will be highlighted.                                         !
         +-----------------------------------------------------------------*/
      {  l_num_focus = get_item_focus() ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( !state.m_as_hil.IsEmpty() )
   {
      /*-------------------------------------------------------------------*/
      long l_num_hil ;
      /*-------------------------------------------------------------------*/
      for( l_num_hil = 0 ;
           l_num_hil < ( long )state.m_as_hil.GetCount() ;
           ++l_num_hil
         )
      {  /*----------------------------------------------------------------*/
         l_num_file = m_dir.file_with_name( state.m_as_hil[ l_num_hil ] ) ;
         if( l_num_file != -1 )
         {  set_item_hil( l_num_file, true ) ;
            ++l_nb_hil ;
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( By default the focus is highlighted )-----------------------------*/
   if( l_nb_hil == 0 )
   {
      /*--( Focus defined ? )----------------------------------------------*/
      if( l_num_focus >= 0 && l_num_focus < m_dir.size() )
      {  set_item_hil( l_num_focus, true ) ; }
      else /*--( Else ... the focus is set on the last file ... )----------*/
      if( ( l_num_file = m_dir.size() - 1 ) >= 0 )
      {  set_item_active( l_num_file ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( Back to previous "image display" )--------------------------------*/
   wxGetApp().m_frame->set_disp_image( m_boo_prev_disp_image ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::launch_file_explorer()
{
   /*----------------------------------------------------------------------*/
   const CFile *p_file = get_p_file_focus() ;
   wxString s_param ;

   /*----------------------------------------------------------------------*/
   if( p_file == NULL )
   {  s_param = "." ; }
   else
   {
      /*-------------------------------------------------------------------*/
#ifdef __WXMSW__
      s_param = "/e,/select," + p_file->get_abs_full_path() ;
#else
      s_param = p_file->get_abs_path() ;
#endif // __WXMSW__
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   wxExecute( "explorer.exe " + s_param ) ;
#else
   /*--( Default application for a directory should be the file manager )--*/
   wxLaunchDefaultApplication( s_param ) ;
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileList::post_loading( CLoadDir &ld )
{
   /*----------------------------------------------------------------------*/
   bool boo_prev_aff_image ;
   bool boo_keep_oper_err  ;
   int  i_file             ;
   int  i_ret_code = 0     ;

   /*--( This is needed for the "is_selectable" that follows )-------------*/
   m_dir.set_recurse( ld.m_dir.get_recurse() ) ;
   /*----------------------------------------------------------------------*/
   boo_keep_oper_err = ld.m_param.get_keep_oper_err() ;

   /*-----------------------------------------------------------------------+
   ! If the directory has changed (not a "reload") then the current path is !
   ! set to it.                                                             !
   +-----------------------------------------------------------------------*/
   if( !ld.get_same_dir() )
   {
      /*-------------------------------------------------------------------*/
      wxString s_new_dir( ld.get_path() ) ;
      /*-------------------------------------------------------------------*/
      if( !s_new_dir.empty() && !wxFileName::SetCwd( s_new_dir ) )
      {  i_ret_code = -1 ; goto func_end ; }

      /*--( Synchronize the config variable )------------------------------*/
#ifdef __WXMSW__
      /*--------------------------------------------------------------------+
      ! This is done to get/display the "real case" of the path.            !
      ! The user may have typed it in the combo not using the "registered"  !
      ! case of the path or maybe typed a 8.3 dos path.                     !
      +--------------------------------------------------------------------*/
      wxFileName fn_path ;
      /*-------------------------------------------------------------------*/
      fn_path.Assign( wxFileName::GetCwd() ) ;
      fn_path.Assign( fn_path.GetShortPath() ) ;
      /*--( The volume is forced "uppercase" because it is "standard" )----*/
      fn_path.SetVolume( fn_path.GetVolume().Upper() ) ;
      /*-------------------------------------------------------------------*/
      wxGetApp().M_s_dir.set( fn_path.GetLongPath() ) ;
      /*-------------------------------------------------------------------*/
#else
      wxGetApp().M_s_dir.set( wxFileName::GetCwd() ) ;
#endif // __WXMSW__

      /*--( Zero the undo/redo info )--------------------------------------*/
      m_vec_otfile.clear() ;
      m_i_num_otfile = 0   ;
      /*-------------------------------------------------------------------*/
   }
   else /*------------------------------------------------------------------+
        ! It is a "reload". The following work has to be done :             !
        ! - Keep (re-affect) some information                               !
        ! - Recompute the new names ...                                     !
        ! The icon index is kept only if something else has to be kept too  !
        +------------------------------------------------------------------*/
   {
      /*--------------------------------------------------------------------+
      ! During the load of the directory the list can be cleared. Before    !
      ! that its display state has been saved.                              !
      +--------------------------------------------------------------------*/
      if( ld.m_fl_display_state.empty() )
      {  extract_display_state( ld.m_fl_display_state ) ; }

      /*-------------------------------------------------------------------*/
      int             i_num_old ; // File number in old/current m_dir
      int             i_num_new ; // File number in the new loaded one
      int             i_num_sel ;
      CRunInfoMessage msg( _( "Restoring data ..." ),
                           _( "Restoring data (%d) ..." )
                         ) ;

      /*--------------------------------------------------------------------+
      ! First: restore the data. The "selection" restoration is done later  !
      +--------------------------------------------------------------------*/
      for( i_num_old = 0 ; i_num_old < m_dir.size() ; ++i_num_old )
      {
         /*--( Some data to keep for this file ? )-------------------------*/
         if(    m_dir[ i_num_old ].get_nb_undo() == 0
             && m_dir[ i_num_old ].checksum_info_empty()
           )
         {  continue ; }

         /*----------------------------------------------------------------*/
         msg.display( i_num_old ) ;

         /*--( This file exists in the new list ? )------------------------*/
           i_num_new
         = ld.m_dir.file_with_name( m_dir[ i_num_old ].get_name() ) ;
         if( i_num_new < 0 ) { continue ; }

         /*--( Keeping icon is only done if something else has to be kept )*/
         ld.m_dir[ i_num_new ].set_i_ico( m_dir[ i_num_old ].get_i_ico() ) ;
         /*----------------------------------------------------------------*/
         if( m_dir[ i_num_old ].get_nb_undo() > 0 )
         {  ld.m_dir[ i_num_new ].swap_undo( m_dir[ i_num_old ] ) ; }
         /*----------------------------------------------------------------*/
         if( boo_keep_oper_err )
         {  ld.m_dir[ i_num_new ].set_oper_err(
                                            m_dir[ i_num_old ].get_oper_err()
                                              ) ;
         }
         /*----------------------------------------------------------------*/
         ld.m_dir[ i_num_new ].checksum_info_copy( m_dir[ i_num_old ] ) ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! Reselect what already was !                                         !
      ! The selection is done in a separate loop to keep it as close as     !
      ! possible to the previous one                                        !
      ! The selection is done after the data restoration because some data  !
      ! may need to be recomputed (md5 ...)                                 !
      +--------------------------------------------------------------------*/
      for( i_num_sel = 0 ; i_num_sel < m_dir.get_nb_sel() ; ++i_num_sel )
      {
         /*----------------------------------------------------------------*/
         msg.display( i_num_sel ) ;

         /*--( The index of this selected file ? )-------------------------*/
         i_num_old = m_dir.get_sel_file_num( i_num_sel ) ;
         /*--( Is it in the new list ? )-----------------------------------*/
           i_num_new
         = ld.m_dir.file_with_name( m_dir[ i_num_old ].get_name() ) ;
         if( i_num_new < 0 ) { continue ; }

         /*--( Keep icon )-------------------------------------------------*/
         ld.m_dir[ i_num_new ].set_i_ico( m_dir[ i_num_old ].get_i_ico() ) ;

         /*--( And select it if possible )---------------------------------*/
         if( ld.m_dir[ i_num_new ].is_selectable() )
         {  ld.m_dir.sel( i_num_new ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   Freeze() ;

   /*--( The new list becomes the current )--------------------------------*/
   m_dir.swap( ld.m_dir ) ;
   SetItemCount( m_dir.size() ) ;

   /*-----------------------------------------------------------------------+
   ! The selection numbers and the data may have change, so a global        !
   ! recomputation is done after the swap.                                  !
   +-----------------------------------------------------------------------*/
   compute_new_names() ;

   /*-----------------------------------------------------------------------+
   ! Setting the focus may activate the display of the image so it is       !
   ! temporarily suppressed.                                                !
   +-----------------------------------------------------------------------*/
   boo_prev_aff_image = wxGetApp().m_frame->get_disp_image() ;
   wxGetApp().m_frame->set_disp_image( false ) ;

   /*--( List has changed so the "use" needs to be reinitialized )---------*/
   init_col_and_grp_used() ;
   /*----------------------------------------------------------------------*/
   create_all_col() ;
   sort_file_list( SFL_DONT_RESFRESH ) ;

   /*-----------------------------------------------------------------------+
   ! A selection filter to apply ?                                          !
   ! The first selected file will have the focus and will be visible        !
   +-----------------------------------------------------------------------*/
   i_file = -1 ;
   if( !ld.m_param.get_sel_filter().empty() )
   {  i_file = sel_fil( ld.m_param.get_sel_filter(),
                        SEARCH_FILTER_MATCH | SEARCH_FILTER_DIR
                      ) ;
   }

   /*-----------------------------------------------------------------------+
   ! Set the focus on a specific file ?                                     !
   ! If it has not been found, the first one will be the active one         !
   +-----------------------------------------------------------------------*/
   if(    !ld.m_param.get_repos_name().empty()
       && ( i_file = m_dir.file_with_name( ld.m_param.get_repos_name() )
          ) >= 0
     )
   {  set_item_active( i_file ) ; }

   /*-----------------------------------------------------------------------+
   ! Set back the state after repositioning the "active item".              !
   ! Note that when it had been set all highlighting has been suppressed.   !
   +-----------------------------------------------------------------------*/
   if( ld.get_same_dir() )
   {  apply_display_state( ld.m_fl_display_state ) ;
      ld.m_fl_display_state.clear() ;
   }
   else if( i_file < 0 )
   {  set_item_active( 0 ) ; }

   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->set_disp_image( boo_prev_aff_image ) ;
   /*----------------------------------------------------------------------*/
   Thaw() ;
   Refresh() ;
   /*----------------------------------------------------------------------*/
func_end :
   /*--( The loaded directory data are useless now )-----------------------*/
   ld.m_dir.zero() ;
   /*----------------------------------------------------------------------*/
   return( i_ret_code ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                     End of file CFileList_Base.cpp                        !
+==========================================================================*/
