/*====================================+=====================================+
! File CFileInit_ttf.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                      Font files ".ttf" and ".otf"                         !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Info: http://www.microsoft.com/typography/otspec/name.htm                 !
!       http://www.microsoft.com/typography/otspec/otff.htm                 !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_ttf_header ;
struct str_ttf_table_entry ;
struct str_ttf_name_header ;
struct str_ttf_name_record ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_ttf : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_ttf( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_be_data( str_ttf_table_entry &tte ) ;
      int read_be_data( str_ttf_header &th ) ;
      int read_be_data( str_ttf_name_header &tnh ) ;
      int read_be_data( str_ttf_name_record &tnr ) ;
      int read_val( const wxFileOffset &fo_offset,
                    size_t sz_size, wxString &s_val
                  ) ;
      int ttf_read() ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_ttf_header
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_sfnt_version_major ;
   wxUint16 w_sfnt_version_minor ;
   wxUint16 w_numTables          ; // Number of tables
   wxUint16 ___w_searchRange     ;
   wxUint16 ___w_entrySelector   ;
   wxUint16 ___w_rangeShift      ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ttf_table_entry
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_tag         ; // 4 -byte identifier
   wxUint32 ___dw_checkSum ; // CheckSum for this table
   wxUint32 dw_offset      ; // Offset from beginning of TrueType font file
   wxUint32 dw_length      ; // Length of this table
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ttf_name_header
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_format       ; // Format selector (=0).
   wxUint16 w_count        ; // Number of name records.
   wxUint16 w_stringOffset ; // Offset to start of string storage
                             // (from start of table).
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ttf_name_record
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_platformID ; // Platform ID
   wxUint16 w_encodingID ; // Platform-specific encoding ID
   wxUint16 w_languageID ; // Language ID
   wxUint16 w_nameID     ; // Name ID
   wxUint16 w_length     ; // String length (in bytes)
   wxUint16 w_offset     ; // String offset from start of storage area
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_ttf::read_be_data( str_ttf_header &th )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( th ), &th ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   th.w_sfnt_version_major = wxUINT16_SWAP_ALWAYS( th.w_sfnt_version_major );
   th.w_sfnt_version_minor = wxUINT16_SWAP_ALWAYS( th.w_sfnt_version_minor );
   th.w_numTables          = wxUINT16_SWAP_ALWAYS( th.w_numTables ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ttf::read_be_data( str_ttf_table_entry &tte )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tte ), &tte ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   //tte.dw_tag = wxUINT32_SWAP_ALWAYS( tte.dw_tag ) ;
   tte.dw_offset = wxUINT32_SWAP_ALWAYS( tte.dw_offset ) ;
   tte.dw_length = wxUINT32_SWAP_ALWAYS( tte.dw_length ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ttf::read_be_data( str_ttf_name_header &tnh )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tnh ), &tnh ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   tnh.w_format       = wxUINT16_SWAP_ALWAYS( tnh.w_format ) ;
   tnh.w_count        = wxUINT16_SWAP_ALWAYS( tnh.w_count ) ;
   tnh.w_stringOffset = wxUINT16_SWAP_ALWAYS( tnh.w_stringOffset ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ttf::read_be_data( str_ttf_name_record &tnr )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tnr ), &tnr ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   tnr.w_platformID = wxUINT16_SWAP_ALWAYS( tnr.w_platformID ) ;
   tnr.w_encodingID = wxUINT16_SWAP_ALWAYS( tnr.w_encodingID ) ;
   tnr.w_languageID = wxUINT16_SWAP_ALWAYS( tnr.w_languageID ) ;
   tnr.w_nameID     = wxUINT16_SWAP_ALWAYS( tnr.w_nameID ) ;
   tnr.w_length     = wxUINT16_SWAP_ALWAYS( tnr.w_length ) ;
   tnr.w_offset     = wxUINT16_SWAP_ALWAYS( tnr.w_offset ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ttf::read_val( const wxFileOffset &fo_offset,
                             size_t sz_size, wxString &s_val
                           )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset fo_offset_sav = m_fa.get_offset() ;
   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( fo_offset ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.file_read_tb_c( sz_size, s_val ) != 0 ) { return( -2 ) ; }
   /*--( Go back to source )-----------------------------------------------*/
   if( m_fa.set_offset( fo_offset_sav ) != 0 ) { return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ttf::ttf_read()
{
   /*----------------------------------------------------------------------*/
   str_ttf_header      header              ;
   str_ttf_table_entry table_entry = { 0 } ; // Init for compilation warning
   str_ttf_name_header name_header         ;
   str_ttf_name_record name_record         ;
   int                 i_col               ;
   wxString            s_val               ;
   int                 i_num               ;

   /*----------------------------------------------------------------------*/
   if( read_be_data( header ) != 0 ) { return( -1 ) ; }
   /*--( TTF / OTF ? )-----------------------------------------------------*/
   if(    (    header.w_sfnt_version_major != 1
            || header.w_sfnt_version_minor != 0
          )
       && (    header.w_sfnt_version_major != SR_TWOCC( 'T','O' )
            || header.w_sfnt_version_minor != SR_TWOCC( 'O','T' )
          )
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ;
           i_num < header.w_numTables
        && read_be_data( table_entry ) == 0
        && table_entry.dw_tag != SR_FOURCC( 'n','a','m','e' ) ;
        ++i_num
      )
   {  ; }

   /*----------------------------------------------------------------------*/
   if( table_entry.dw_tag != SR_FOURCC( 'n','a','m','e' ) )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( table_entry.dw_offset ) != 0 )
   {  return( -4 ) ; }

   /*--( Start of the "name" table )---------------------------------------*/
   if( read_be_data( name_header ) != 0 )
   {  return( -5 ) ; }
   /*--( Format 0 should be found first )----------------------------------*/
   if( name_header.w_format != 0 )
   {  return( -6 ) ; }

   /*----------------------------------------------------------------------*/
   m_fi.set_conv_from_utf16be() ;

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < name_header.w_count ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if( read_be_data( name_record ) != 0 ) { return( -7 ) ; }

      /*--( Only read English text encoded in UTF16 BE )-------------------*/
      if(    name_record.w_languageID != 1033
          || name_record.w_platformID != 3
        )
      {  continue ; }

      /*-------------------------------------------------------------------*/
      switch( name_record.w_nameID )
      {
         /*----------------------------------------------------------------*/
         case  0 : // Copyright notice
            i_col = COL_DOC_COPYRIGHT ; break ;
         case  1 : // Font name
            i_col = COL_DOC_TITLE ; break ;
         case  3 : // Unique font identifier
            i_col = COL_DOC_ID ; break ;
         case  4 : // Font full name
            i_col = COL_DOC_SUBJECT ; break ;
         case  5 : // Version string
            i_col = COL_DOC_VERSION ; break ;
         case  7 : // Trademark
            i_col = COL_DOC_TRADEMARKS ; break ;
         case  8 : // Manufacturer Name
            i_col = COL_DOC_COMPANY ; break ;
         case  9 : // Designer
            i_col = COL_DOC_AUTHOR ; break ;
         case 10 : // Description
            i_col = COL_DOC_COMMENT ; break ;
         case 11 : // URL Vendor
            i_col = COL_DOC_URL ; break ;
         case 13 : // License Description
            i_col = COL_DOC_INFO ; break ;
         /*----------------------------------------------------------------*/
         default : continue ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      if( !m_fi.reserve_col( i_col ) ) { continue ; }
      /*-------------------------------------------------------------------*/
      if( read_val(   table_entry.dw_offset
                    + name_header.w_stringOffset
                    + name_record.w_offset,
                    name_record.w_length,
                    s_val
                  ) != 0
         )
      {  continue ; }
      /*-------------------------------------------------------------------*/
      if( !s_val.empty() )
      {  m_f.val_s( i_col ) = s_val ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_ttf()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "ttf" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_ttf( *this ).ttf_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_ttf.cpp                       !
+==========================================================================*/
