/*====================================+=====================================+
! File CFileInit_tga.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                          Image files ".tga"                               !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Even if they are not really EXIF, the metadata are stored in this group   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_tga_header ;
struct str_tga_new_footer ;
struct str_tga_extension_area ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_tga : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_tga( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_le_data( str_tga_header &th ) ;
      int read_le_data( str_tga_new_footer &tnf ) ;
      int read_le_data( str_tga_extension_area &tea ) ;
      int tga_read( wxFileOffset WXUNUSED( fo_offset_image ),
                    wxULongLong  ull_image_size
                  ) ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_tga_header
{
   /*----------------------------------------------------------------------*/
   wxUint8  b_id_length                          ; // Len of image_id (below)
   wxUint8  ___b_color_map_type                  ;
   wxUint8  b_image_type                         ;
   wxUint8  ___tb_b_color_map_specification[ 5 ] ;
   wxUint16 ___w_x_origin                        ;
   wxUint16 ___w_y_origin                        ;
   wxUint16 w_width                              ;
   wxUint16 w_height                             ;
   wxUint8  b_bpp                                ;
   wxUint8  b_image_descriptor                   ;
   char     tb_c_image_id[ 255 ]                 ; // 255 is the Max size
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_tga_new_footer
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_extension_area_offset          ;
   wxUint32 ___dw_developper_directory_offset ;
   char     tb_c_signature[ 16 ]              ;
   char     b_dot_character                   ;
   char     b_zero_binary                     ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_tga_extension_area
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_size                       ;
   char     tb_c_author_name    [  41 ]  ;
   char     tb_c_author_comments[ 324 ]  ;
   wxUint16 w_dt_stamp_month             ;
   wxUint16 w_dt_stamp_day               ;
   wxUint16 w_dt_stamp_year              ;
   wxUint16 w_dt_stamp_hour              ;
   wxUint16 w_dt_stamp_minute            ;
   wxUint16 w_dt_stamp_second            ;
   char     tb_c_job_name_id     [ 41 ]  ;
   wxUint16 w_job_time_hours             ;
   wxUint16 w_job_time_minutes           ;
   wxUint16 w_job_time_seconds           ;
   char     tb_c_software_id     [ 41 ]  ;
   wxUint16 w_soft_version_number100     ;
   char     c_soft_version_letter        ;
/*
   and many others ...
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_tga::read_le_data( str_tga_header &th )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( th ), &th ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   th.w_width  = wxUINT16_SWAP_ALWAYS( th.w_width  ) ;
   th.w_height = wxUINT16_SWAP_ALWAYS( th.w_height ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_tga::read_le_data( str_tga_new_footer &tnf )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tnf ), &tnf ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     tnf.dw_extension_area_offset
   = wxUINT32_SWAP_ALWAYS( tnf.dw_extension_area_offset ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_tga::read_le_data( str_tga_extension_area &tea )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tea ), &tea ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     tea.w_size
   = wxUINT16_SWAP_ALWAYS( tea.w_size ) ;
     tea.w_dt_stamp_month
   = wxUINT16_SWAP_ALWAYS( tea.w_dt_stamp_month ) ;
     tea.w_dt_stamp_day
   = wxUINT16_SWAP_ALWAYS( tea.w_dt_stamp_day ) ;
     tea.w_dt_stamp_year
   = wxUINT16_SWAP_ALWAYS( tea.w_dt_stamp_year ) ;
     tea.w_dt_stamp_hour
   = wxUINT16_SWAP_ALWAYS( tea.w_dt_stamp_hour ) ;
     tea.w_dt_stamp_minute
   = wxUINT16_SWAP_ALWAYS( tea.w_dt_stamp_minute ) ;
     tea.w_dt_stamp_second
   = wxUINT16_SWAP_ALWAYS( tea.w_dt_stamp_second ) ;
     tea.w_job_time_hours
   = wxUINT16_SWAP_ALWAYS( tea.w_job_time_hours ) ;
     tea.w_job_time_minutes
   = wxUINT16_SWAP_ALWAYS( tea.w_job_time_minutes ) ;
     tea.w_job_time_seconds
   = wxUINT16_SWAP_ALWAYS( tea.w_job_time_seconds ) ;
     tea.w_soft_version_number100
   = wxUINT16_SWAP_ALWAYS( tea.w_soft_version_number100 ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_tga::tga_read( wxFileOffset WXUNUSED( fo_offset_image ),
                             wxULongLong  ull_image_size
                           )
{
   /*----------------------------------------------------------------------*/
   str_tga_header         header   ;
   str_tga_new_footer     footer   ;
   str_tga_extension_area ext_area ;
   const char             *p_c_tmp ;
   wxString               s_val    ;
   wxDateTime             dt_val   ;

   /*----------------------------------------------------------------------*/
   if( ull_image_size <= ( int )sizeof( header ) )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( read_le_data( header ) != 0 )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   switch( header.b_image_type )
   {
      /*-------------------------------------------------------------------*/
      case  0 :
         p_c_tmp = "No Image Data Included" ;
         break ;
      case  1 :
         p_c_tmp = "Uncompressed, Color-mapped" ;
         break ;
      case  2 :
         p_c_tmp = "Uncompressed, True-color" ;
         break ;
      case  3 :
         p_c_tmp = "Uncompressed, Black-and-white" ;
         break ;
      case  9 :
         p_c_tmp = "Run-length encoded, Color-mapped" ;
         break ;
      case 10 :
         p_c_tmp = "Run-length encoded, True-color" ;
         break ;
      case 11 :
         p_c_tmp = "Run-length encoded, Black-and-white" ;
         break ;
      /*-------------------------------------------------------------------*/
      default : p_c_tmp = "Unknown" ; break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_f.val_s( COL_IMG_INFO ).Printf( "Type %d (%s)",
                                     header.b_image_type, p_c_tmp
                                   ) ;

   /*----------------------------------------------------------------------*/
   m_fi.init_img_x_y( header.w_width, header.w_height ) ;
   m_fi.init_img_bpp( header.b_bpp ) ;

   /*----------------------------------------------------------------------*/
   switch( ( header.b_image_descriptor & 0x30 ) >> 4 )
   {
      /*-------------------------------------------------------------------*/
      case 0  : m_fi.init_exif_rotation( IV_ROTATION_180_HM ) ; break ;
      case 1  : m_fi.init_exif_rotation( IV_ROTATION_180 )    ; break ;
      case 2  : m_fi.init_exif_rotation( IV_ROTATION_0 )      ; break ;
      case 3  : m_fi.init_exif_rotation( IV_ROTATION_0_HM )   ; break ;
      /*-------------------------------------------------------------------*/
      default : m_fi.init_exif_rotation( IV_ROTATION_NB )     ; break ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if(    header.b_id_length > 0
       && m_fi.prepare_string( header.tb_c_image_id,
                               sizeof( header.tb_c_image_id ), s_val
                             ) > 0
     )
   {  m_f.val_s( COL_EXIF_IMAGEUNIQUEID ) = s_val ; }

   /*-----------------------------------------------------------------------+
   ! This file is a "new tga file" ? If so, it has a specific footer        !
   ! that may point to the "extension area" which may contain some info.    !
   +-----------------------------------------------------------------------*/
   if(    m_fa.set_offset_from_end( sizeof( footer ) ) != 0
       || read_le_data( footer ) != 0
       || memcmp( footer.tb_c_signature, "TRUEVISION-XFILE",
                  sizeof( footer.tb_c_signature )
                ) != 0
       || footer.b_dot_character != '.'
       || footer.b_zero_binary != '\0'
     )
   {  return( 0 ) ; }

   /*--( Offset and read ok ? )--------------------------------------------*/
   if(    footer.dw_extension_area_offset == 0
       || m_fa.set_offset( footer.dw_extension_area_offset ) != 0
       || read_le_data( ext_area ) != 0
     )
   {  return( 0 ) ; }

   /*----------------------------------------------------------------------*/
   if( ext_area.w_size != 495 )
   {  return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( ext_area.tb_c_author_name,
                            sizeof( ext_area.tb_c_author_name ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_EXIF_ARTIST ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( ext_area.tb_c_author_comments,
                            sizeof( ext_area.tb_c_author_comments ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_EXIF_USERCOMMENT ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( sr::set_datetime( ext_area.w_dt_stamp_year, ext_area.w_dt_stamp_month,
                         ext_area.w_dt_stamp_day , ext_area.w_dt_stamp_hour,
                         ext_area.w_dt_stamp_minute,
                         ext_area.w_dt_stamp_second, 0, dt_val
                       ) == 0
     )
   {  m_fi.init_date( COL_EXIF_DATETIMEORIGINAL, dt_val ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( ext_area.tb_c_job_name_id,
                            sizeof( ext_area.tb_c_job_name_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_EXIF_COPYRIGHT ) = s_val ; }
   /*----------------------------------------------------------------------*/
   m_f.val_s( COL_EXIF_FILESOURCE ).Printf( "%dh%02dm%02ds",
                                            ext_area.w_job_time_hours,
                                            ext_area.w_job_time_minutes,
                                            ext_area.w_job_time_seconds
                                          ) ;

   /*--( Store software and version in the same Exif col )-----------------*/
   if( m_fi.prepare_string( ext_area.tb_c_software_id,
                            sizeof( ext_area.tb_c_software_id ), s_val
                          ) > 0
     )
   {  s_val.append( ' ' ) ; }
   /*----------------------------------------------------------------------*/
     m_f.val_s( COL_EXIF_SOFTWARE )
   = s_val + wxString::Format( "%d.%02d%c",
                               ext_area.w_soft_version_number100 / 100,
                               ext_area.w_soft_version_number100 % 100,
                               ext_area.c_soft_version_letter
                             ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_tga()
{
   /*----------------------------------------------------------------------*/
   int i_ret ;
   /*----------------------------------------------------------------------*/
   m_s_type_det = "tga" ;
   /*----------------------------------------------------------------------*/
   if( ( i_ret = tga_read( 0, m_f.get_size() ) ) == 0 )
   {  m_f.set_image_offset( 0 ) ; }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
!                                                                           !
! This function is directly used by other init modules ...                  !
!                                                                           !
+--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
int CFileInit::tga_read( wxFileOffset fo_offset, wxULongLong ull_size )
{  return( CFileInit_tga( *this ).tga_read( fo_offset, ull_size ) ) ; }

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_tga.cpp                       !
+==========================================================================*/
