/*====================================+=====================================+
! File CFileInit_pdf.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!               Document file: ".pdf": Document Acrobat Reader              !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Compressed xref/streams are not supported.                                !
!                                                                           !
! The principle is:                                                         !
! - first load the "references" of the xref tables: offset, starting ref    !
!   and number of entries.                                                  !
! - Then read of the "dictionaries" entries based on the xref information   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/wfstream.h>
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
#include "common/CChecksum.h"
#include "common/CCrypt.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
const size_t co_sz_raw_val_max_len = 512 ;

/*--( Array of data values )-----------------------------------------------*/
typedef std::vector< wxMemoryBuffer > t_vec_mb ;

/*-------------------------------------------------------------------------*/
enum e_pdf_dict_data_mask
{
   /*----------------------------------------------------------------------*/
   PDF_DDM_XREF    = 1 << 0,
   PDF_DDM_ROOT    = 1 << 1,
   PDF_DDM_PAGES   = 1 << 2,
   PDF_DDM_ENCRYPT = 1 << 3,
   PDF_DDM_INFO    = 1 << 4
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*--( Cross reference line structure )-------------------------------------*/
struct str_pdf_xref_line
{
   /*----------------------------------------------------------------------*/
   char tb_c_offset[ 10 ]    ;
   char c_separator_1        ;
   // cppcheck-suppress unusedStructMember
   char tb_c_generation[ 5 ] ;
   // cppcheck-suppress unusedStructMember
   char c_separator_2        ;
   char c_entry_usage        ; // n = in use, f = free
   // cppcheck-suppress unusedStructMember
   char c_cariage_return     ;
   // cppcheck-suppress unusedStructMember
   char c_line_feed          ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
struct str_pdf_encrypt
{
   /*----------------------------------------------------------------------*/
   str_pdf_encrypt() : m_boo_encrypted( false ),
                       m_i_R( -1 ), // for cppcheck
                       m_boo_R_present( false ),
                       m_boo_O_present( false ),
                       m_boo_U_present( false ),
                       m_i_P( -1 ), // for cppcheck
                       m_boo_P_present( false ),
                       m_boo_EncryptMetadata( true ),
                       m_i_Length( 40 ),
                       m_i_V( 0 ),
                       m_i_ID_len( -1 ), // for cppcheck
                       m_boo_ID_present( false ),
                       m_i_obj_num_Info( -1 ), // for cppcheck
                       m_i_gen_num_Info( -1 ), // for cppcheck
                       m_i_key_len( -1 )       // for cppcheck
   {  ; }
   /*----------------------------------------------------------------------*/
   int init_key( const char *p_c_password ) ;

   /*----------------------------------------------------------------------*/
   bool     m_boo_encrypted       ;

   /*--( Fields of the Encrypt object )------------------------------------*/
   int      m_i_R                 ;
   bool     m_boo_R_present       ;
   wxUint8  m_tb_b_O[ 32 ]        ;
   bool     m_boo_O_present       ;
   wxUint8  m_tb_b_U[ 32 ]        ;
   bool     m_boo_U_present       ;
   int      m_i_P                 ;
   bool     m_boo_P_present       ;
   bool     m_boo_EncryptMetadata ;
   wxString m_s_Filter            ;
   int      m_i_Length            ;
   int      m_i_V                 ;
   /*--( First element of the pdf's ID element )---------------------------*/
   wxUint8  m_tb_b_ID[ co_sz_raw_val_max_len ] ;
   int      m_i_ID_len            ;
   bool     m_boo_ID_present      ;
   /*-----------------------------------------------------------------------+
   ! All the data that have to be decrypted belong to the "Info" dictionary !
   ! To create the RC4 key the INFO ref object has to be kept               !
   +-----------------------------------------------------------------------*/
   int      m_i_obj_num_Info      ;
   int      m_i_gen_num_Info      ;

   /*--( Computed key used to crypt/decrypt )------------------------------*/
   wxUint8  m_tb_b_key[ 16 ]      ;
   int      m_i_key_len           ;
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_pdf : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_pdf( CFileInit &parent )
                   : CFileInit_type_base( parent ),
                     m_finput( m_fa ),
                     m_fo_offset_prev_xref( -1 ),
                     m_i_obj_num_encrypt( -1 ),
                     m_i_obj_num_root( -1 ),
                     m_i_obj_num_pages( -1 ),
                     m_i_obj_num_info( -1 ),
                     m_i_obj_num_page_count( -1 )
      {  ; }

      /*--( Map: object number - file offset )-----------------------------*/
      typedef std::map< int, wxFileOffset > t_map_int_fo     ;
      typedef t_map_int_fo::const_iterator  t_map_int_fo_cit ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      t_map_int_fo m_map_xref ;
      /*-------------------------------------------------------------------*/
      wxFileInputStream m_finput ;
      /*--( Raw data read from pdf: dictionary or metadata )---------------*/
      t_vec_mb m_vec_mb_raw_data ;

      /*--( The Cross References are linked in "back" order )--------------*/
      wxFileOffset m_fo_offset_prev_xref ;
      /*--( The object numbers found. The ones that will be used )---------*/
      int m_i_obj_num_encrypt    ;
      int m_i_obj_num_root       ;
      int m_i_obj_num_pages      ;
      int m_i_obj_num_info       ;
      int m_i_obj_num_page_count ;
      int m_i_obj_num_meta       ;

      /*-------------------------------------------------------------------*/
      str_pdf_encrypt m_encrypt ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int read_line( wxString &s ) ;
      int skip_spaces() ;
      int read_data( bool boo_keep_data = true ) ;
      int goto_indirect_ref( int i_num ) ;
      int read_var_val( wxString &s_var, wxString &s_val,
                        bool boo_info_val
                      ) ;
      int read_full_obj( int i_obj_num, wxString &s_val ) ;
      int read_dict( wxUint32 ui_mask, int i_dict_obj_num = -1 ) ;
      int read_xref( wxFileOffset fo_offset ) ;
      int pdf_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
int CFileInit_pdf::read_line( wxString &s )
{
   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   while(    m_finput.Peek() != '\r'
          && m_finput.Peek() != '\n'
          && m_finput.IsOk()
        )
   {  s += m_finput.GetC() ; }
   /*----------------------------------------------------------------------*/
   if( !m_finput.IsOk() ) { return( -1 ) ; }
   /*--( Skipping a LF after a CR )----------------------------------------*/
   if( m_finput.Peek() == '\r' ) { m_finput.GetC() ; }
   if( m_finput.Peek() == '\n' ) { m_finput.GetC() ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_pdf::skip_spaces()
{
   /*----------------------------------------------------------------------*/
   while( isspace( m_finput.Peek() ) && !m_finput.Eof() )
   {  m_finput.GetC() ; }
   if( m_finput.Eof() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Recursive function )-------------------------------------------------*/
int CFileInit_pdf::read_data( bool boo_keep_data )
{
   /*----------------------------------------------------------------------*/
   wxMemoryBuffer mb_data ;
   wxUint8        b_in    ;
   wxUint8        b_val   ;

   /*----------------------------------------------------------------------*/
   if( skip_spaces() != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   b_in = m_finput.Peek() ;

   /*--( Name ? )----------------------------------------------------------*/
   if( b_in == '/' )
   {
      /*-------------------------------------------------------------------*/
      m_finput.GetC() ;
      /*-------------------------------------------------------------------*/
      while(    (    isalnum( b_in = m_finput.Peek() )
                  || b_in == '#' || b_in == '_' || b_in == '-' || b_in == '.'
                )
             && !m_finput.Eof()
           )
      {
         /*----------------------------------------------------------------*/
         m_finput.GetC() ;
         /*--( A character can be specified in hexadecimal: #XX )----------*/
         if( b_in == '#' )
         {  /*-------------------------------------------------------------*/
            wxUint8 b_xnum1 ;
            wxUint8 b_xnum2 ;
            /*-------------------------------------------------------------*/
            if(  !isxdigit( b_xnum1 = m_finput.GetC() ) || m_finput.Eof()
              || !isxdigit( b_xnum2 = m_finput.GetC() ) || m_finput.Eof()
              )
            {  return( -2 ) ; }
            /*-------------------------------------------------------------*/
            b_in = (  ( sr::xchar_value( b_xnum1 ) << 4 )
                      | sr::xchar_value( b_xnum2 )
                   ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         mb_data.AppendByte( b_in ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   else /*--( String ? )---------------------------------------------------*/
   if( b_in == '(' )
   {
      /*-------------------------------------------------------------------*/
      int  i_num_embed_par = 1 ;
      bool boo_char_to_add ;
      /*-------------------------------------------------------------------*/
      b_in = m_finput.GetC() ;
      if( m_finput.Eof() ) { return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      while( ( b_val = m_finput.GetC() ) != ')' || i_num_embed_par != 1 )
      {
         /*----------------------------------------------------------------*/
         boo_char_to_add = true ;
         /*----------------------------------------------------------------*/
         if( b_val == '\\' )
         {
            /*-------------------------------------------------------------*/
            b_in = m_finput.GetC() ;
            if( m_finput.Eof() ) { return( -4 ) ; }
            /*--( There can be an octal char code on 3 positions max )-----*/
            if( sr::bool_isodigit( b_in ) )
            {
               /*----------------------------------------------------------*/
               int i_num = 1 ;
               for( b_val = sr::char_value( b_in ) ;
                    i_num < 3 && sr::bool_isodigit( m_finput.Peek() ) ;
                    ++i_num
                  )
               {  b_val = b_val * 8 + sr::char_value( m_finput.GetC() ) ; }
               /*----------------------------------------------------------*/
            }
            else
            {
               /*----------------------------------------------------------*/
               switch( b_in )
               {
                  /*-------------------------------------------------------*/
                  case 'n'  : b_val = '\n' ; break ;
                  case 'r'  : b_val = '\r' ; break ;
                  case 't'  : b_val = '\t' ; break ;
                  case 'b'  : b_val = '\b' ; break ;
                  case 'f'  : b_val = '\f' ; break ;
                  /*--------------------------------------------------------+
                  ! A "\" at the end of a line indicates that the text      !
                  ! continues on the next one                               !
                  +--------------------------------------------------------*/
                  case '\r' : // Must be followed by a LF that we skip
                     if( m_finput.Peek() == '\n' ) { m_finput.GetC() ; }
                  case '\n' :
                     boo_char_to_add = false ;
                     break ;
                  /*-------------------------------------------------------*/
                  default :
                     b_val = b_in ;
                     break ;
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         else /*-----------------------------------------------------------*/
         if( b_val != ')' || i_num_embed_par > 1 )
         {
            /*--( The parentheses can be embedded ... )--------------------*/
            if( b_val == '(' ) { ++i_num_embed_par ; }
            else
            if( b_val == ')' ) { --i_num_embed_par ; }
            /*-------------------------------------------------------------*/
         }

         /*--( Store standard read or "computed" character )---------------*/
         if( boo_char_to_add && mb_data.GetDataLen() < co_sz_raw_val_max_len)
         {  mb_data.AppendByte( b_val ) ; }

         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   else
   if( b_in == '<' )
   {
      /*-------------------------------------------------------------------*/
      m_finput.GetC() ;
      if( m_finput.Eof() ) { return( -5 ) ; }
      /*--( "Sub-object" delimited by << >> )------------------------------*/
      if( m_finput.Peek() == '<' )
      {
         /*--( First sub level )-------------------------------------------*/
         int i_embed_level = 1 ;

         /*--( Embedded data won't be kept but have to be skipped )--------*/
         boo_keep_data = false ;
         /*--( Read the '<' character and skip the junk )------------------*/
         m_finput.GetC() ;
         if( skip_spaces() != 0 ) { return( -6 ) ; }
         /*----------------------------------------------------------------*/
         do
         {
            /*-------------------------------------------------------------*/
            b_in = m_finput.Peek() ;
            /*-------------------------------------------------------------*/
            if( b_in == '<' || b_in == '>' )
            {
               /*--( Skip the '<' or '>' to check the next one )-----------*/
               m_finput.GetC() ;
               /*--( The '<' or '>' has to be doubled to open or close )---*/
               if( m_finput.Peek() == b_in )
               {  /*-------------------------------------------------------*/
                  i_embed_level += ( b_in == '<' ? +1 : -1 ) ;
                  m_finput.GetC() ;
                  continue ;
                  /*-------------------------------------------------------*/
               }

               /*--( No, the character is finally a standard one ! )-------*/
               m_finput.Ungetch( b_in ) ;
               /*----------------------------------------------------------*/
            }

            /*--------------------------------------------------------------+
            ! Only the first level data is kept.                            !
            ! The sub-dictionary is composed of elementary data.            !
            +--------------------------------------------------------------*/
            if( read_data( i_embed_level == 1 ) != 0 )
            {  return( -7 ) ; }
            /*-------------------------------------------------------------*/
         } while( i_embed_level > 0 && !m_finput.Eof() ) ;

         /*----------------------------------------------------------------*/

      }
      else /*--( Hexadecimal string between <> )---------------------------*/
      {
         /*--( The hexadecimal digits can be separated )-------------------*/
         while(    !m_finput.Eof()
                && skip_spaces() == 0 && m_finput.Peek() != '>'
              )
         {
            /*-------------------------------------------------------------*/
            if( !isxdigit( m_finput.Peek() ) )
            {  m_finput.GetC() ; }
            else
            {  /*----------------------------------------------------------*/
               b_val = ( wxUint8 )sr::xchar_value( m_finput.GetC() ) << 4 ;
               if( isxdigit( m_finput.Peek() ) )
               {  b_val |= ( wxUint8 )sr::xchar_value( m_finput.GetC() ) ; }
               /*--( Store the character )---------------------------------*/
               if( mb_data.GetDataLen() < co_sz_raw_val_max_len )
               {  mb_data.AppendByte( b_val ) ; }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         if( m_finput.Peek() == '>' ) { m_finput.GetC() ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   else /*--( Array ? )----------------------------------------------------*/
   if( b_in == '[' )
   {
      /*--( First sub level )----------------------------------------------*/
      int i_embed_level = 1 ;

      /*--( Embedded data won't be kept but have to be skipped )-----------*/
      boo_keep_data = false ;
      /*--( Read the bracket and skip the junk )---------------------------*/
      m_finput.GetC() ;
      if( skip_spaces() != 0 ) { return( -8 ) ; }
      /*-------------------------------------------------------------------*/
      do
      {
         /*----------------------------------------------------------------*/
         b_in = m_finput.Peek() ;
         /*----------------------------------------------------------------*/
         if( b_in == '[' )
         {  ++i_embed_level ; m_finput.GetC() ; }
         else
         if( b_in == ']' )
         {  --i_embed_level ; m_finput.GetC() ; }
         else
         {  /*--------------------------------------------------------------+
            ! Only the first level data is kept.                            !
            ! The array is composed of elementary data.                     !
            +--------------------------------------------------------------*/
            if( read_data( i_embed_level == 1 ) != 0 )
            {  return( -9 ) ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      } while( i_embed_level > 0 && !m_finput.Eof() ) ;
      /*-------------------------------------------------------------------*/
   }
   else /*--( Get the standard values piece after piece )------------------*/
   {
      /*-------------------------------------------------------------------*/
      while(    (     isalnum( b_in ) || isspace( b_in )
                   || b_in == '+' || b_in == '-' || b_in == '.'
                )
             && !m_finput.Eof()
           )
      {  /*----------------------------------------------------------------*/
         if( mb_data.GetDataLen() < co_sz_raw_val_max_len )
         {  /*--( Replace \r, \n ... with space )--------------------------*/
            mb_data.AppendByte( isspace( b_in ) ? ' ' : b_in ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         m_finput.GetC() ;
         b_in = m_finput.Peek() ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( boo_keep_data )
   {  m_vec_mb_raw_data.push_back( sr::memorybuffer_copy( mb_data ) ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static const wxUniChar st_co_tb_c_pdfDocEncoding[ 256 ] =
{
   /*----------------------------------------------------------------------*/
   0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 00
   0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
   0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 10
   0x02d8, 0x02c7, 0x02c6, 0x02d9, 0x02dd, 0x02db, 0x02da, 0x02dc,
   0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, // 20
   0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
   0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, // 30
   0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
   0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, // 40
   0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
   0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, // 50
   0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
   0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, // 60
   0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
   0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, // 70
   0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x0000,
   0x2022, 0x2020, 0x2021, 0x2026, 0x2014, 0x2013, 0x0192, 0x2044, // 80
   0x2039, 0x203a, 0x2212, 0x2030, 0x201e, 0x201c, 0x201d, 0x2018,
   0x2019, 0x201a, 0x2122, 0xfb01, 0xfb02, 0x0141, 0x0152, 0x0160, // 90
   0x0178, 0x017d, 0x0131, 0x0142, 0x0153, 0x0161, 0x017e, 0x0000,
   0x20ac, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7, // a0
   0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x0000, 0x00ae, 0x00af,
   0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7, // b0
   0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
   0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7, // c0
   0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
   0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7, // d0
   0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
   0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7, // e0
   0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
   0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7, // f0
   0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
static int st_pdf_conv_tag_col( const wxString &s_var )
{
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Title" ) == 0 )
   {  return( COL_DOC_TITLE ) ; }
   if( s_var.CmpNoCase( "Author" ) == 0 )
   {  return( COL_DOC_AUTHOR   ) ; }
   if( s_var.CmpNoCase( "Subject" ) == 0 )
   {  return( COL_DOC_SUBJECT  ) ; }
   if( s_var.CmpNoCase( "Keywords" ) == 0 )
   {  return( COL_DOC_KEYWORDS ) ; }

   /*--( Prog that created the original document (even if it's not PDF) )--*/
   if( s_var.CmpNoCase( "Creator" ) == 0 )
   {  return( COL_DOC_APPLICATION ) ; }
   /*-----------------------------------------------------------------------+
   ! Prog that converted this document to PDF.                              !
   ! The value of "APPLICATION" will be concatenated to it.                 !
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Producer" ) == 0 )
   {  return( COL_DOC_TRADEMARKS ) ; }

   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "CreationDate" ) == 0 )
   {  return( COL_DOC_CRE_DATE ) ; }
   if( s_var.CmpNoCase( "ModDate" ) == 0 )
   {  return( COL_DOC_MOD_DATE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Count" ) == 0 )
   {  return( COL_DOC_PAGECOUNT ) ; }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The format to follow is supposed to be: D:YYYYMMDDHHMMSSxHH'MM'           !
!                                                                           !
! "xHH'MM'" is the difference with UT. It is not used here.                 !
+--------------------------------------------------------------------------*/
static int st_pdf_conv_string_datetime( const wxString &s, wxDateTime &dt )
{
   /*----------------------------------------------------------------------*/
   const char *p_co_c_format = "D:%4d%2d%2d%2d%2d%2d" ;
   int i_pos_yyyy ;

   /*--( The "D:" is not mandatory but strongly recommended )--------------*/
   if( s.StartsWith( "D:" ) )
   {  i_pos_yyyy = 2 ; }
   else
   {  i_pos_yyyy = 0 ;
      p_co_c_format += 2 ;
   }

   /*--( Only the year is mandatory )--------------------------------------*/
   if( s.size() < 4 ) { return( -1 ) ; }

   /*--( All the characters must be numerical ones )-----------------------*/
   sr::wxString_cit cit     = s.begin() + i_pos_yyyy ;
   sr::wxString_cit cit_end = cit + wxMin( s.size() - i_pos_yyyy, 14 ) ;
   for( ; cit < cit_end ; ++cit )
   {  if( !sr::bool_isdigit( *cit ) ) { return( -2 ) ; } }

   /*----------------------------------------------------------------------*/
   if( sr::init_date_ymdhms( s, p_co_c_format, dt ) != 0 )
   {  return( -3 ) ; }
   /*--( Conversion from GMT to local time zone )--------------------------*/
   dt.MakeFromTimezone( wxDateTime::GMT0 ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static bool st_pdf_is_indirect_ref( const wxString &s,
                                    int &i_obj_num, int &i_gen_num
                                  )
{
   /*----------------------------------------------------------------------*/
   char tb_c_info[ 11 ] ;

   /*----------------------------------------------------------------------*/
   if( wxSscanf( s, "%d %d %4s", &i_obj_num, &i_gen_num, tb_c_info ) != 3 )
   {  i_obj_num = -1 ; return( false ) ; }

   /*----------------------------------------------------------------------*/
   return( true ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static bool st_pdf_is_indirect_ref( const wxString &s, int &i_obj_num )
{
   /*----------------------------------------------------------------------*/
   int i_dummy_gen_num ;
   /*----------------------------------------------------------------------*/
   return( st_pdf_is_indirect_ref( s, i_obj_num, i_dummy_gen_num ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function will check that the given password is ok and will           !
! initialize the encryption key info: "m_i_key_len" and "m_tb_b_key"        !
+--------------------------------------------------------------------------*/
int str_pdf_encrypt::init_key( const char *p_c_password )
{
   /*----------------------------------------------------------------------*/
   wxUint8      tb_b_work_key[ 32 ]   ;
   int          i_work_key_len        ;
   wxUint8      tb_b_computed_U[ 32 ] ;
   int          i_computed_U_len      ;
   CChecksumMD5 md5                   ;
   CCryptRC4    rc4                   ;
   int          i_Length_in_byte      ;
   int          i_num                 ;
   int          i_val                 ;

   /*----------------------------------------------------------------------*/
   static const wxUint8 st_co_tb_b_pad_password[ 32 ]
   = { 0x28, 0xBF, 0x4E, 0x5E, 0x4E, 0x75, 0x8A, 0x41,
       0x64, 0x00, 0x4E, 0x56, 0xFF, 0xFA, 0x01, 0x08,
       0x2E, 0x2E, 0x00, 0xB6, 0xD0, 0x68, 0x3E, 0x80,
       0x2F, 0x0C, 0xA9, 0xFE, 0x64, 0x53, 0x69, 0x7A
     } ;

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! Check if the given password is ok (it can be empty).                   !
   !                                                                        !
   ! Therefore the algorithm described in PDF Doc 3.6 is used               !
   !                                                                        !
   +-----------------------------------------------------------------------*/

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! PDF Doc 3.2: Encryption KEY computation                                !
   !                                                                        !
   +-----------------------------------------------------------------------*/
   i_Length_in_byte = m_i_Length / 8 ;

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.1                                                          !
   +-----------------------------------------------------------------------*/

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.2                                                          !
   ! Empty password means that all comes from the pad password array        !
   +-----------------------------------------------------------------------*/
   i_work_key_len = wxMin( 32, strlen( p_c_password ) ) ;
   memcpy( tb_b_work_key, p_c_password, i_work_key_len ) ;
   /*----------------------------------------------------------------------*/
   if( i_work_key_len < 32 )
   {  memcpy( tb_b_work_key + i_work_key_len, st_co_tb_b_pad_password,
              32 - i_work_key_len
            ) ;
   }
   /*----------------------------------------------------------------------*/
   md5.init() ;
   md5.update( tb_b_work_key, 32 ) ;

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.3                                                          !
   +-----------------------------------------------------------------------*/
   md5.update( m_tb_b_O, sizeof( m_tb_b_O ) ) ;
   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.4                                                          !
   +-----------------------------------------------------------------------*/
   md5.update( ( wxUint32 )wxINT32_SWAP_ON_BE( m_i_P ) ) ;
   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.5                                                          !
   +-----------------------------------------------------------------------*/
   md5.update( m_tb_b_ID, m_i_ID_len ) ;

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.6                                                          !
   ! Not used: metadata not encrypted                                       !
   +-----------------------------------------------------------------------*/

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.7                                                          !
   +-----------------------------------------------------------------------*/
   md5.final() ;
   i_work_key_len = md5.get_result_len() ;
   memcpy( tb_b_work_key, md5.get_result(), i_work_key_len ) ;

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.8                                                          !
   +-----------------------------------------------------------------------*/
   if( m_i_R >= 3 )
   {
      /*-------------------------------------------------------------------*/
      for( i_num = 0 ; i_num < 50 ; ++i_num )
      {  md5.calc( tb_b_work_key, i_Length_in_byte, tb_b_work_key ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.2.9                                                          !
   +-----------------------------------------------------------------------*/
   i_work_key_len = ( m_i_R == 2 ? 5 : i_Length_in_byte ) ;

   /*-----------------------------------------------------------------------+
   ! Check if the the password is empty                                     !
   +------------------------------------------------------------------------+
   ! PDF Doc 3.6                                                            !
   +-----------------------------------------------------------------------*/
   if( m_i_R == 2 )
   {
      /*--------------------------------------------------------------------+
      ! PDF Doc 3.4.2                                                       !
      +--------------------------------------------------------------------*/
      rc4.init( tb_b_work_key, i_work_key_len ) ;
      rc4.crypt( st_co_tb_b_pad_password, sizeof( st_co_tb_b_pad_password ),
                 tb_b_computed_U
               ) ;
      i_computed_U_len = i_work_key_len ;
      /*-------------------------------------------------------------------*/
   }
   else
   {
      /*--------------------------------------------------------------------+
      ! PDF Doc 3.5.2                                                       !
      +--------------------------------------------------------------------*/
      md5.init() ;
      md5.update( st_co_tb_b_pad_password, sizeof( st_co_tb_b_pad_password));
      /*--------------------------------------------------------------------+
      ! PDF Doc 3.5.3                                                       !
      +--------------------------------------------------------------------*/
      md5.update( m_tb_b_ID, m_i_ID_len ) ;
      md5.final() ;
      /*--------------------------------------------------------------------+
      ! PDF Doc 3.5.4                                                       !
      +--------------------------------------------------------------------*/
      rc4.init( tb_b_work_key, i_work_key_len ) ;
      i_computed_U_len = md5.get_result_len() ;
      rc4.crypt( md5.get_result(), i_computed_U_len, tb_b_computed_U ) ;

      /*--------------------------------------------------------------------+
      ! PDF Doc 3.5.5                                                       !
      +--------------------------------------------------------------------*/
      wxUint8 tb_b_work_key_3_5_5[ sizeof( tb_b_work_key ) ] ;
      int i_time ;
      for( i_time = 1 ; i_time <= 19 ; ++i_time )
      {
         /*----------------------------------------------------------------*/
         for( i_num = 0 ; i_num < i_work_key_len ; ++i_num )
         {  tb_b_work_key_3_5_5[ i_num ] = tb_b_work_key[ i_num ] ^ i_time ;}
         /*----------------------------------------------------------------*/
         rc4.init( tb_b_work_key_3_5_5, i_work_key_len ) ;
         rc4.crypt( tb_b_computed_U, i_computed_U_len, tb_b_computed_U ) ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! PDF Doc 3.5.6                                                       !
      +--------------------------------------------------------------------*/
      memset( tb_b_computed_U + i_computed_U_len, '\0',
              sizeof( tb_b_computed_U ) - i_computed_U_len
            ) ;
       /*------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! Password OK ?                                                          !
   !                                                                        !
   +-----------------------------------------------------------------------*/
   if( memcmp( m_tb_b_U, tb_b_computed_U, sizeof( m_tb_b_U ) ) != 0 )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! Compute the RC4 encryption for the "INFO" dictionary                   !
   !                                                                        !
   +-----------------------------------------------------------------------*/

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.1.1                                                          !
   +-----------------------------------------------------------------------*/

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.1.2                                                          !
   +-----------------------------------------------------------------------*/

   /*--( Add the 3 last digits of the object number )----------------------*/
   i_val = m_i_obj_num_Info ;
   for( i_num = 0 ; i_num < 3 ; ++i_num )
   {  tb_b_work_key[ i_work_key_len++ ] = i_val % 256 ; i_val /= 256 ; }
   /*--( Add the 2 last digits of the generation number )------------------*/
   i_val = m_i_gen_num_Info ;
   for( i_num = 0 ; i_num < 2 ; ++i_num )
   {  tb_b_work_key[ i_work_key_len++ ] = i_val % 256 ; i_val /= 256 ; }

   /*-----------------------------------------------------------------------+
   ! PDF Doc 3.1.3                                                          !
   +-----------------------------------------------------------------------*/
   md5.calc( tb_b_work_key, i_work_key_len, tb_b_work_key ) ;
   i_work_key_len = wxMin( i_work_key_len, 16 ) ;

   /*--( Here it is ... ready to go ! )------------------------------------*/
   m_i_key_len = i_work_key_len ;
   memcpy( m_tb_b_key, tb_b_work_key, m_i_key_len ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_pdf::goto_indirect_ref( int i_num )
{
   /*----------------------------------------------------------------------*/
   CFileInit_pdf::t_map_int_fo_cit it_map ;

   /*----------------------------------------------------------------------*/
   it_map = m_map_xref.find( i_num ) ;
   if( it_map == m_map_xref.end() ) { return( -1 ) ; }
   if( m_finput.SeekI( it_map->second ) == wxInvalidOffset )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function will read the name and the value associated to it.          !
! The '/' variable name prefix has been read by the calling routine.        !
!                                                                           !
! boo_info_val: The var/val couple is part of the "Info" dictionary ?       !
+--------------------------------------------------------------------------*/
int CFileInit_pdf::read_var_val( wxString &s_var, wxString &s_val,
                                 bool boo_info_val
                               )
{
   /*----------------------------------------------------------------------*/
   wxUint8 *p_b_data  ;
   int     i_data_len ;

   /*----------------------------------------------------------------------*/
   m_vec_mb_raw_data.clear() ;
   /*----------------------------------------------------------------------*/
   s_var.clear() ;
   s_val.clear() ;

   /*--( Skip empty information )------------------------------------------*/
   if( skip_spaces() != 0 ) { return( -1 ) ; }
   /*--( A name must be found )--------------------------------------------*/
   if( m_finput.Peek() != '/' ) { return( -2 ) ; }
   /*--( Read the name )---------------------------------------------------*/
   if( read_data() != 0 || m_vec_mb_raw_data.empty() )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
     s_var
   = wxString::From8BitData( ( char * )m_vec_mb_raw_data[ 0 ].GetData(),
                             m_vec_mb_raw_data[ 0 ].GetDataLen()
                           ) ;

   /*--( The name has been read. The array can be cleared )----------------*/
   m_vec_mb_raw_data.clear() ;

   /*--( And then read its associated data )-------------------------------*/
   if( read_data() != 0 || m_vec_mb_raw_data.empty() )
   {  return( -4 ) ; }

   /*----------------------------------------------------------------------*/
   p_b_data   = ( wxUint8 * )m_vec_mb_raw_data[ 0 ].GetData() ;
   i_data_len = m_vec_mb_raw_data[ 0 ].GetDataLen() ;

   /*--( If there then the data can be "decrypted" )-----------------------*/
   if(    boo_info_val
       && m_encrypt.m_boo_encrypted
       && m_encrypt.m_boo_EncryptMetadata
     )
   {
      /*-------------------------------------------------------------------*/
      CCryptRC4 rc4( m_encrypt.m_tb_b_key, m_encrypt.m_i_key_len ) ;
      rc4.crypt( p_b_data, i_data_len ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Doc: PDFDocEncoding or UTF-16BE with a leading byte-order marker )*/
   if( sr::has_BOM16( ( char * )p_b_data, i_data_len ) )
   {  sr::prepare_string( ( char * )p_b_data, i_data_len / 2 * 2, s_val,
                          sr::CONV_FROM_UTF16LEBE
                        ) ;
   }
   else
   {  /*--( PdfDocEndocding conversion )-----------------------------------*/
      int i_nb ;
      s_val.reserve( i_data_len ) ;
      for( i_nb = 0 ; i_nb < i_data_len ; ++i_nb )
      {  s_val += st_co_tb_c_pdfDocEncoding[ p_b_data[ i_nb ] ] ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_fi.prepare_string( s_val ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_pdf::read_full_obj( int i_obj_num, wxString &s_val )
{
   /*----------------------------------------------------------------------*/
   wxString s               ;
   bool     boo_end = false ;
   /*----------------------------------------------------------------------*/
   s_val.erase() ;

   /*----------------------------------------------------------------------*/
   if( goto_indirect_ref( i_obj_num ) != 0 ) { return( -1 ) ; }

   /*--( Read the "obj" line )---------------------------------------------*/
   if( read_line( s ) != 0 || s.empty() ) { return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   do
   {
      /*-------------------------------------------------------------------*/
      if( read_line( s ) != 0 ) { return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      if( s.CmpNoCase( "endobj" ) == 0 )
      {  boo_end = true ; }
      else
      {  if( !s_val.empty() ) { s_val += ' ' ; }
         s_val += s ;
      }
      /*-------------------------------------------------------------------*/
   } while( !boo_end ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_pdf::read_dict( wxUint32 ui_mask, int i_dict_obj_num )
{
   /*----------------------------------------------------------------------*/
#define PDF_VAR_OK( m, v ) \
   ( ( ui_mask & m ) != 0 && s_var.CmpNoCase( v ) == 0 )
   /*----------------------------------------------------------------------*/
   int      i_col           ;
   wxString s_var           ;
   wxString s_val           ;
   bool     boo_end = false ;

   /*--( Move file pointer based on object number ? )----------------------*/
   if( i_dict_obj_num >= 0 && goto_indirect_ref( i_dict_obj_num ) != 0 )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! The next two interesting characters should be "<<"                     !
   ! In the data read there will be the object and generation numbers.      !
   +-----------------------------------------------------------------------*/
   while( !m_finput.Eof() && m_finput.GetC() != '<' )
   {  ; }
   if( m_finput.Eof() ) { return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_finput.GetC() != '<' ) { return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   while( !boo_end )
   {
      /*--------------------------------------------------------------------+
      ! The data may be decrypted if it comes out of an INFO dictionary     !
      +--------------------------------------------------------------------*/
      if( read_var_val( s_var, s_val, ( ui_mask & PDF_DDM_INFO ) != 0 ) != 0)
      {  return( -4 ) ; }

      /*-------------------------------------------------------------------*/
      if( PDF_VAR_OK( PDF_DDM_XREF, "Info" ) )
      {  /*----------------------------------------------------------------*/
         if( !st_pdf_is_indirect_ref( s_val, m_i_obj_num_info,
                                      m_encrypt.m_i_gen_num_Info
                                    )
           )
         {  return( -5 ) ; }
         /*----------------------------------------------------------------*/
         m_encrypt.m_i_obj_num_Info = m_i_obj_num_info ;
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_XREF, "Root" ) )
      {  /*----------------------------------------------------------------*/
         if( !st_pdf_is_indirect_ref( s_val, m_i_obj_num_root ) )
         {  return( -6 ) ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_XREF, "Prev" ) )
      {  /*----------------------------------------------------------------*/
         m_fo_offset_prev_xref = wxAtol( s_val ) ;
         /*-----------------------------------------------------------------+
         ! xref are supposed to be appended to the pdf file.                !
         ! This test avoids cyclic references.                              !
         +-----------------------------------------------------------------*/
         if( m_fo_offset_prev_xref <= m_finput.TellI() )
         {  m_fo_offset_prev_xref = -1 ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_XREF, "Encrypt" ) )
      {
         /*----------------------------------------------------------------*/
         if( !st_pdf_is_indirect_ref( s_val, m_i_obj_num_encrypt ) )
         {  return( -7 ) ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_XREF, "ID" ) )
      {
         /*----------------------------------------------------------------*/
         m_encrypt.m_i_ID_len = m_vec_mb_raw_data[ 0 ].GetDataLen() ;
         /*----------------------------------------------------------------*/
         memcpy( m_encrypt.m_tb_b_ID,
                 m_vec_mb_raw_data[ 0 ].GetData(),
                 m_encrypt.m_i_ID_len
               ) ;
         m_encrypt.m_boo_ID_present = true ;
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "Filter" ) )
      {  m_encrypt.m_s_Filter = s_val ; }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "Length" ) )
      {  m_encrypt.m_i_Length = wxAtol( s_val ) ; }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "V" ) )
      {  m_encrypt.m_i_V = wxAtol( s_val ) ; }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "P" ) )
      {  m_encrypt.m_i_P = wxAtol( s_val ) ;
         m_encrypt.m_boo_P_present = true ;
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "R" ) )
      {  m_encrypt.m_i_R = wxAtol( s_val ) ;
         m_encrypt.m_boo_R_present = true ;
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "EncryptMetadata" ) )
      {    m_encrypt.m_boo_EncryptMetadata
         = ( s_val.CmpNoCase( "true" ) == 0 ) ;
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "U" ) )
      {
         /*--( The length is defined in pdf reference document )-----------*/
         if(    m_vec_mb_raw_data[ 0 ].GetDataLen()
             != sizeof( m_encrypt.m_tb_b_U )
           )
         {  return( -8 ) ; }
         /*-----------------------------------------------------------------+
         ! "s_val" is a "prepared" version of the "raw val".                !
         ! The password contains "binary" data that can be removed          !
         ! by the "preparation". Therefore the "raw" version is used.       !
         +-----------------------------------------------------------------*/
         memcpy( m_encrypt.m_tb_b_U, m_vec_mb_raw_data[ 0 ].GetData(),
                 sizeof( m_encrypt.m_tb_b_U )
               ) ;
         m_encrypt.m_boo_U_present = true ;
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ENCRYPT, "O" ) )
      {
         /*--( The length is defined in pdf reference document )-----------*/
         if(    m_vec_mb_raw_data[ 0 ].GetDataLen()
             != sizeof( m_encrypt.m_tb_b_O )
           )
         {  return( -9 ) ; }
         /*----------------------------------------------------------------*/
         memcpy( m_encrypt.m_tb_b_O, m_vec_mb_raw_data[ 0 ].GetData(),
                 sizeof( m_encrypt.m_tb_b_O )
               ) ;
         m_encrypt.m_boo_O_present = true ;
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ROOT, "Pages" ) )
      {
         /*----------------------------------------------------------------*/
         if( !st_pdf_is_indirect_ref( s_val, m_i_obj_num_pages ) )
         {  return( -10 ) ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( PDF_VAR_OK( PDF_DDM_ROOT, "Metadata" ) )
      {
         /*----------------------------------------------------------------*/
         if( !st_pdf_is_indirect_ref( s_val, m_i_obj_num_meta ) )
         {  return( -11 ) ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( ( ui_mask & PDF_DDM_PAGES ) != 0 )
      {
         /*----------------------------------------------------------------*/
         i_col = st_pdf_conv_tag_col( s_var ) ;
         /*--( The column is reserved even if the value is empty )---------*/
         if( !m_fi.reserve_col( i_col ) || s_val.empty() ) { continue ; }

         /*--( Indirect ref "loading" is done at the end of the process )--*/
         if( i_col != COL_DOC_PAGECOUNT )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         if( !st_pdf_is_indirect_ref( s_val, m_i_obj_num_page_count ) )
         {  /*-------------------------------------------------------------*/
            m_f.val_ll( i_col ) = wxAtol( s_val ) ;
            m_f.val_s( i_col )  = s_val ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else
      if( ( ui_mask & PDF_DDM_INFO ) != 0 )
      {
         /*----------------------------------------------------------------*/
         i_col = st_pdf_conv_tag_col( s_var ) ;
         /*--( The column is reserved even if the value is empty )---------*/
         if( !m_fi.reserve_col( i_col ) || s_val.empty() ) { continue ; }
         /*----------------------------------------------------------------*/
         if( i_col == COL_DOC_CRE_DATE || i_col == COL_DOC_MOD_DATE )
         {  /*-------------------------------------------------------------*/
            wxDateTime dt ;
            /*-------------------------------------------------------------*/
            if( st_pdf_conv_string_datetime( s_val, dt ) == 0 )
            {  m_fi.init_date( i_col, dt ) ; }
            else /*--( The date can be "pure" text )-----------------------*/
            {  m_f.val_s( i_col ) = s_val ; }
            /*-------------------------------------------------------------*/
         }
         else
         if( i_col == COL_DOC_APPLICATION )
         {  /*-------------------------------------------------------------*/
            sr::prepend_with_sep( s_val, " - ",
                                  m_f.val_s( COL_DOC_APPLICATION )
                                ) ;
            /*-------------------------------------------------------------*/
         }
         else /*--( The software that made the pdf conversion )------------*/
         if( i_col == COL_DOC_TRADEMARKS )
         {  /*-------------------------------------------------------------*/
            sr::append_with_sep( s_val, " - ",
                                 m_f.val_s( COL_DOC_APPLICATION )
                               ) ;
            /*-------------------------------------------------------------*/
         }
         else
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
      }

      /*--( End of dictionary ? )------------------------------------------*/
      if( skip_spaces() != 0 )
      {  return( -12 ) ; }
      /*-------------------------------------------------------------------*/
      if( m_finput.Peek() == '>' )
      {  /*----------------------------------------------------------------*/
         m_finput.GetC() ;
         if( m_finput.Peek() == '>' )
         {  m_finput.GetC() ; boo_end = true ; continue ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
#undef PDF_VAR_OK
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The cross reference can be stored in two different ways:                  !
! - The cross reference table                                               !
! - The cross reference stream (started with pdf 1.4)                       !
+--------------------------------------------------------------------------*/
int CFileInit_pdf::read_xref( wxFileOffset fo_offset )
{
   /*----------------------------------------------------------------------*/
   wxString          s         ;
   int               i_num_ref ;
   int               i_nb_ref  ;
   str_pdf_xref_line xref_line ;
   bool              boo_end   ;

   /*--( Jump at the beginning of the table )------------------------------*/
   if( m_finput.SeekI( fo_offset ) == wxInvalidOffset )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   m_fo_offset_prev_xref = -1 ;

   /*----------------------------------------------------------------------*/
   if( read_line( s ) != 0 ) { return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   if( s != "xref" ) { return( -3 ) ; }

   /*-----------------------------------------------------------------------+
   ! The structure is:                                                      !
   ! start: numeric value indicating the index of the first xref            !
   !        (xref "tables" can be separated in many blocks)                 !
   ! nb   : number of entries in this part of xref table                    !
   !                                                                        !
   !      start nb                                                          !
   !      9999999999 99999 Xxx                                              !
   !      .... nb times ......                                              !
   +-----------------------------------------------------------------------*/
   do
   {  /*-------------------------------------------------------------------*/
      if( read_line( s ) != 0 || s.empty() )
      {  return( -4 ) ; }

      /*-------------------------------------------------------------------*/
      if( ( boo_end = ( s.CmpNoCase( "trailer" ) == 0 ) ) == true )
      {  continue ; }

      /*--( Extract number of the first object and the number of objects )-*/
      if( wxSscanf( s, "%d %d", &i_num_ref, &i_nb_ref ) != 2 )
      {  return( -5 ) ; }

      /*--------------------------------------------------------------------+
      ! Read the "offset lines".                                            !
      ! Some pdf do not respect the standard of the 20 characters line.     !
      ! Their data won't be read ...                                        !
      +--------------------------------------------------------------------*/
      for( ; i_nb_ref > 0 ; ++i_num_ref, --i_nb_ref )
      {
         /*----------------------------------------------------------------*/
         if( !m_finput.Read( &xref_line, sizeof( xref_line ) ).IsOk() )
         {  return( -6 ) ; }
         /*--( "n" means that this entry is currently used )---------------*/
         if( xref_line.c_entry_usage == 'n' )
         {  /*--------------------------------------------------------------+
            ! Terminates the string to be sure "atol" does right.           !
            ! "atol" is used because the data in ansi.                      !
            +--------------------------------------------------------------*/
            xref_line.c_separator_1 = '\0' ;
            m_map_xref[ i_num_ref ] = atol( xref_line.tb_c_offset ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
   } while( !boo_end ) ;

   /*--( Just after the data comes the dictionary )------------------------*/
   if( read_dict( PDF_DDM_XREF ) != 0 )
   {  return( -7 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static wxString st_previous_non_empty_line( char *tb_c_buffer, int &i_from )
{
   /*----------------------------------------------------------------------*/
   int i_end = i_from ;
   /*----------------------------------------------------------------------*/
   if( i_from >= 0 ) { --i_from ; }
   /*--( Search the beginning of a non empty line )------------------------*/
   while(    i_from >= 0
          && (    tb_c_buffer[ i_from ] == '\r'
               || tb_c_buffer[ i_from ] == '\n'
             )
        )
   {  --i_from ; }
   /*----------------------------------------------------------------------*/
   i_end = i_from ;
   /*----------------------------------------------------------------------*/
   while(    i_from >= 0
          && tb_c_buffer[ i_from ] != '\r' && tb_c_buffer[ i_from ] != '\n'
        )
   {  --i_from ; }
   /*--( Start of the line )-----------------------------------------------*/
   if( i_from > 0 ) { ++i_from ; }

   /*----------------------------------------------------------------------*/
   return( wxString::From8BitData( &tb_c_buffer[ i_from ],
                                   i_end - i_from + 1
                                 )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_pdf::pdf_read()
{
   /*----------------------------------------------------------------------*/
   int      i_num                    ;
   wxString s                        ;
   char     tb_c_buffer[ 256 ]       ;
   bool     boo_info_readable = true ;

   /*--( Minimal size )----------------------------------------------------*/
   if( m_f.get_size() <= 1024 ) { return( -1 ) ; }

   /*--( "Magic info" in the first line )----------------------------------*/
   if( read_line( s ) != 0 || !s.StartsWith( "%PDF-" ) )
   {  return( -2 ) ; }

   /*--( Keep the pdf version )--------------------------------------------*/
   m_f.val_s( COL_DOC_VERSION ).assign( s, 1, wxString::npos ) ;

   /*--( Then go to the end of the file )----------------------------------*/
   if( m_finput.SeekI( - ( int )sizeof( tb_c_buffer ), wxFromEnd
                     ) == wxInvalidOffset
     )
   {  return( -3 ) ; }

   /*--( Read the end )----------------------------------------------------*/
   if( !m_finput.Read( tb_c_buffer, sizeof( tb_c_buffer ) ).IsOk() )
   {  return( -4 ) ; }

   /*--( Some empty lines may follow the ending "Magic info" )-------------*/
   i_num = wxMin( m_f.get_size().ToULong(), ( int )sizeof( tb_c_buffer ) ) ;
   if( st_previous_non_empty_line( tb_c_buffer, i_num ) != "%%EOF" )
   {  return( -5 ) ; }

   /*--( The line before contains the offset of the xref table )-----------*/
   if( (   m_fo_offset_prev_xref
         = wxAtol( st_previous_non_empty_line( tb_c_buffer, i_num ) )
       ) == 0
     )
   {  return( -6 ) ; }

   /*-----------------------------------------------------------------------+
   ! Load the xref tables information                                       !
   +-----------------------------------------------------------------------*/
   while( m_fo_offset_prev_xref >= 0 )
   {  /*-------------------------------------------------------------------*/
      if( read_xref( m_fo_offset_prev_xref ) != 0 )
      {  return( -7 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Encrypted file ? )------------------------------------------------*/
   if( m_i_obj_num_encrypt != -1 )
   {
      /*-------------------------------------------------------------------*/
      read_dict( PDF_DDM_ENCRYPT, m_i_obj_num_encrypt ) ;
      /*-------------------------------------------------------------------*/
      m_encrypt.m_boo_encrypted = true ;

      /*--( MetaData written "in clear" ? )--------------------------------*/
      if(     m_encrypt.m_i_V == 4
          && !m_encrypt.m_boo_EncryptMetadata
        )
      {  boo_info_readable = true ; }
      else /*--( Info can be decrypted ? )---------------------------------*/
      if(     m_encrypt.m_s_Filter.CmpNoCase( "Standard" ) != 0
          ||  m_encrypt.m_i_V <= 0
          ||  m_encrypt.m_i_V >= 4
          ||  m_encrypt.m_i_R < 2
          ||  m_encrypt.m_i_R > 4
          ||  m_encrypt.m_i_Length < 40
          ||  m_encrypt.m_i_Length > 128
          ||  m_encrypt.m_i_Length % 8 != 0
          || !m_encrypt.m_boo_R_present
          || !m_encrypt.m_boo_O_present
          || !m_encrypt.m_boo_U_present
          || !m_encrypt.m_boo_P_present
          || !m_encrypt.m_boo_ID_present
        )
      {  /*----------------------------------------------------------------*/
         m_f.val_s( COL_DOC_INFO ) = _( "Unsupported file encryption" ) ;
         boo_info_readable = false ;
         /*----------------------------------------------------------------*/
      }
      else
      {
         /*--( Try to use an empty password )------------------------------*/
         boo_info_readable = ( m_encrypt.init_key( "" ) == 0 ) ;
         /*----------------------------------------------------------------*/
           m_f.val_s( COL_DOC_INFO )
         = ( boo_info_readable
             ? _( "File encrypted (empty password)" )
             : _( "File encrypted (non empty password)" )
           ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Read the "dictionaries" )-----------------------------------------*/
   read_dict( PDF_DDM_ROOT, m_i_obj_num_root ) ;
   read_dict( PDF_DDM_PAGES, m_i_obj_num_pages ) ;
   if( boo_info_readable )
   {  read_dict( PDF_DDM_INFO, m_i_obj_num_info ) ; }

   /*--( The number of pages is defined indirectly ? )---------------------*/
   if( m_i_obj_num_page_count != -1 )
   {
      /*-------------------------------------------------------------------*/
      wxString s_val ;
      /*--( The page number is the only info in the object )---------------*/
      if( read_full_obj( m_i_obj_num_page_count, s_val ) == 0 )
      {  /*----------------------------------------------------------------*/
         m_f.val_ll( COL_DOC_PAGECOUNT ) = wxAtol( s_val ) ;
         m_f.val_s( COL_DOC_PAGECOUNT )  = s_val ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_pdf()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "pdf" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_pdf( *this ).pdf_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                        End of file CFileInit_pdf.cpp                      !
+==========================================================================*/
