/*====================================+=====================================+
! File CFileInit_ogg.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                       Audio/Video file: ".ogg"                            !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Are only supported:                                                       !
!     - vorbis for the audio                                                !
!     - theora for the video                                                !
!                                                                           !
! The file can't contain other types of headers                             !
!                                                                           !
! For the video part, the duration is computed by scanning the file ...     !
! so I it is an approximative computation.                                  !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_ogg_stream_header ;
struct str_ogg_common_header ;
struct str_ogg_theora_identification_header ;
struct str_ogg_vorbis_identification_header ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_ogg : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_ogg( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_le_data( str_ogg_stream_header &osh ) ;
      int read_le_data( str_ogg_common_header &och ) ;
      int read_be_data( str_ogg_theora_identification_header &otih ) ;
      int read_le_data( str_ogg_vorbis_identification_header &ovih ) ;
      int read_varchar_c( wxString &s ) ;
      int read_var_val( wxString &s_var, wxString &s_val ) ;
      int read_stream_header( int &i_packet_type, wxUint32 &dw_serial ) ;
      int read_useful_stream_header( int &i_packet_type,
                                     wxUint32 &dw_serial
                                   ) ;
      int read_vorbis_comment( bool boo_video ) ;
      int ogg_read() ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_ogg_stream_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_capture_pattern            ; // "OggS"
   wxUint8  ___b_stream_structure_version ;
   wxUint8  b_header_type_flag            ;
   wxUint64 ddw_abs_granule_pos           ;
   wxUint32 dw_stream_serial_number       ;
   wxUint32 ___dw_page_sequence_no        ;
   wxUint32 ___dw_page_checksum           ;
   wxUint8  b_page_segments               ; // number of segments
   /*----------------------------------------------------------------------*/
// Segment_table: tb_b_page_segments[ 0 ] + 26
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ogg_common_header
{
   /*----------------------------------------------------------------------*/
   wxUint8 b_packet_type         ;
   char    tb_c_vorbis_text[ 6 ] ; // "theora" / "vorbis"
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ogg_theora_identification_header
{
   /*----------------------------------------------------------------------*/
   wxUint8  b_vmaj             ; // The major version number
   wxUint8  b_vmin             ; // The minor version number
   wxUint8  b_vrev             ; // The version revision number
   wxUint16 ___w_fmbw          ; // The width of frame in macro blocks
   wxUint16 ___w_fmbh          ; // The height of frame in macro blocks
   wxUint8  tb_b_fmbw[ 3 ]     ; // The width of picture region in pixels
   wxUint8  tb_b_fmbh[ 3 ]     ; // The height of picture region in pixels
   wxUint8  ___b_picx          ; // The X offset of picture region in pixels
   wxUint8  ___b_pixy          ; // The Y offset of picture region in pixels
   wxUint32 dw_frn             ; // The frame-rate numerator
   wxUint32 dw_frd             ; // The frame-rate denominator
   wxUint8  ___tb_b_parn[ 3 ]  ; // The pixel aspect-ratio numerator
   wxUint8  ___tb_b_pard[ 3 ]  ; // The pixel aspect-ratio denominator
   wxUint8  ___b_cs            ; // The colour space
   wxUint8  ___tb_b_nombr[ 3 ] ; // The pixel aspect-ratio denominator
   wxUint16 w_qu_ks_pf         ; // QUAL: 6 bits
                                 // KFGSHIFT: 5 bits
                                 // PF: 2 bits
                                 // 3 bits: reserved
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ogg_vorbis_identification_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_vorbis_version     ; // = 0
   wxUint8  b_audio_channels      ; // > 0
   wxUint32 dw_audio_sample_rate  ; // > 0
   wxUint32 ___dw_bitrate_maximum ;
   wxUint32 ___dw_bitrate_nominal ;
   wxUint32 ___dw_bitrate_minimum ;
   wxUint8  b_blocksize           ; // [blocksize_0] = 4 bits
                                    // [blocksize_1] = 4 bits
   wxUint8  b_framing_flag        ; // != 0
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_ogg::read_le_data( str_ogg_stream_header &osh )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( osh ), &osh ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   osh.dw_capture_pattern  = wxUINT32_SWAP_ALWAYS( osh.dw_capture_pattern  );
   osh.ddw_abs_granule_pos = wxUINT64_SWAP_ALWAYS( osh.ddw_abs_granule_pos );
     osh.dw_stream_serial_number
   = wxUINT32_SWAP_ALWAYS( osh.dw_stream_serial_number ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ogg::read_le_data( str_ogg_common_header &och )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( och ), &och ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ogg::read_be_data( str_ogg_theora_identification_header &otih )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( otih ), &otih ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   otih.dw_frn     = wxUINT32_SWAP_ALWAYS( otih.dw_frn     ) ;
   otih.dw_frd     = wxUINT32_SWAP_ALWAYS( otih.dw_frd     ) ;
   otih.w_qu_ks_pf = wxUINT16_SWAP_ALWAYS( otih.w_qu_ks_pf ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ogg::read_le_data( str_ogg_vorbis_identification_header &ovih )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ovih ), &ovih ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     ovih.dw_vorbis_version
   = wxUINT32_SWAP_ALWAYS( ovih.dw_vorbis_version ) ;
     ovih.dw_audio_sample_rate
   = wxUINT32_SWAP_ALWAYS( ovih.dw_audio_sample_rate ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static const wxUint32 st_co_dw_OggS = SR_FOURCC( 'O','g','g','S' ) ;

/*-------------------------------------------------------------------------*/
static const int st_co_i_ogg_max_seach_size = 40 * 1024 ;

/*--( The packet should appear in the enum declaration order )-------------*/
enum e_packet_type { VORBIS_PT_IDENTIFICATION = 0x01,
                     VORBIS_PT_COMMENT        = 0x03,
                     VORBIS_PT_SETUP          = 0x05,
                     THEORA_PT_IDENTIFICATION = 0x80,
                     THEORA_PT_COMMENT        = 0x81,
                     THEORA_PT_SETUP          = 0x82,
                     OGG_SKELETON_30          = 'f'
                   } ;

/*-------------------------------------------------------------------------*/
static int st_ogg_conv_tag_col( bool boo_video, const wxString &s_var )
{
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "TITLE" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_TITLE : COL_AUDTAG_TITLE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "ALBUM" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_ALBUM ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "ARTIST" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ARTIST ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "TRACKNUMBER" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_TRACK ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "TRACKTOTAL" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_TRACK_NB ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "DISCNUMBER" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_DISK ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "COPYRIGHT" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_COPYRIGHT : COL_AUDTAG_COPYRIGHT ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "GENRE" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_GENRE : COL_AUDTAG_GENRE ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "URL" ) == 0      // TGF
       || s_var.CmpNoCase( "LICENSE" ) == 0  // EasyTAG
     )
   {  return( boo_video ? COL_NB : COL_AUDTAG_URL ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "COMMENT" ) == 0
       || s_var.CmpNoCase( "DESCRIPTION" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_COMMENT : COL_AUDTAG_COMMENT ) ; }

   /*--( ... maybe not very good "transfer" )------------------------------*/
   if( s_var.CmpNoCase( "DATE" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_CRE_DATE : COL_AUDTAG_YEAR ) ; }
   if(    s_var.CmpNoCase( "COMPOSER" ) == 0
       || s_var.CmpNoCase( "ORGANIZATION" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_ENGINEER : COL_AUDTAG_COMPOS ) ; }

   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "ORIGINALARTIST" ) == 0
       || s_var.CmpNoCase( "PERFORMER" ) == 0   // EasyTAG
     )
   {  return( boo_video ? COL_VIDTAG_TECHNICIAN : COL_AUDTAG_ORG_ART ) ; }

   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "ENCODEDBY" ) == 0   // TGF
       || s_var.CmpNoCase( "ENCODED-BY" ) == 0  // EasyTAG
       || s_var.CmpNoCase( "ENCODER" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_SOFTWARE : COL_AUDTAG_ENCOD_BY ) ; }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! size(WORD) + Non unicode data                                             !
+--------------------------------------------------------------------------*/
int CFileInit_ogg::read_varchar_c( wxString &s )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_size ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( dw_size ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( m_fi.file_read_tb_c( dw_size, s ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ogg::read_var_val( wxString &s_var, wxString &s_val )
{
   /*--( VVVV=XXXX )-------------------------------------------------------*/
   wxString s_vector ;
   int      i_pos    ;

   /*----------------------------------------------------------------------*/
   s_var.clear() ;
   s_val.clear() ;

   /*--( Name and value of the variable )----------------------------------*/
   if( read_varchar_c( s_vector ) != 0 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   if( ( i_pos = ( int )s_vector.find( '=' ) ) != ( int )wxString::npos )
   {
      /*-------------------------------------------------------------------*/
      s_var.assign( s_vector, 0        , i_pos ) ;
      s_val.assign( s_vector, i_pos + 1, wxString::npos ) ;
      m_fi.prepare_string( s_val ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! If the "stream" is ok but not "useful" the returned value is 1            !
! The "useless" packet is skipped                                           !
+--------------------------------------------------------------------------*/
int CFileInit_ogg::read_stream_header( int &i_packet_type,
                                       wxUint32 &dw_serial
                                     )
{
   /*----------------------------------------------------------------------*/
   str_ogg_stream_header stream_header      ;
   str_ogg_common_header header             ;
   wxMemoryBuffer        mb_buffer_segments ;

   /*----------------------------------------------------------------------*/
   if( read_le_data( stream_header ) != 0 )
   {  return( -1 ) ; }
   if( stream_header.dw_capture_pattern != st_co_dw_OggS )
   {  return( -2 ) ; }

   /*--( This info uniquely identifies the stream )------------------------*/
   dw_serial = stream_header.dw_stream_serial_number ;

   /*-----------------------------------------------------------------------+
   ! Keep the segments to jump to the next stream                           !
   ! For theora & vorbis, as they contain only one segment, a simple read   !
   ! will make the jump too.                                                !
   +-----------------------------------------------------------------------*/
   if( m_fa.read_buffer( stream_header.b_page_segments, mb_buffer_segments
                       ) != 0
     )
   {  return( -3 ) ; }

   /*--( Then the header )-------------------------------------------------*/
   if( read_le_data( header ) != 0 ) { return( -4 ) ; }

   /*--( Control based on the text )---------------------------------------*/
   switch( i_packet_type = header.b_packet_type )
   {
      /*-------------------------------------------------------------------*/
      case VORBIS_PT_IDENTIFICATION :
      case VORBIS_PT_COMMENT :
         /*----------------------------------------------------------------*/
         if( memcmp( header.tb_c_vorbis_text, "vorbis", 6 ) != 0 )
         {  return( -5 ) ; }
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      case THEORA_PT_IDENTIFICATION :
      case THEORA_PT_COMMENT :
         /*----------------------------------------------------------------*/
         if( memcmp( header.tb_c_vorbis_text, "theora", 6 ) != 0 )
         {  return( -6 ) ; }
         /*----------------------------------------------------------------*/
         break ;

      /*--( Unmanaged stream: jump over it )-------------------------------*/
      default :
      {
         /*----------------------------------------------------------------*/
         long l_off_skip ;
         int  i_num      ;
         /*--( Skip back over the "header" )-------------------------------*/
         l_off_skip = - ( int )sizeof( header ) ;
         /*----------------------------------------------------------------*/
         for( i_num = 0 ; i_num < stream_header.b_page_segments ; ++i_num )
         {     l_off_skip
            += *( ( wxUint8 * )mb_buffer_segments.GetData() + i_num ) ;
         }
         /*--( Normally a positive value should have been found ... )------*/
         if( l_off_skip < 0 ) { return( -7 ) ; }
         /*----------------------------------------------------------------*/
         if( l_off_skip > 0 && m_fa.skip_nb_byte( l_off_skip ) != 0 )
         {  return( -8 ) ; }
         /*--( Unmanaged packet )------------------------------------------*/
         return( 1 ) ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Skip the non recognized packet types including the Ogg 3.0                !
+--------------------------------------------------------------------------*/
int CFileInit_ogg::read_useful_stream_header( int &i_packet_type,
                                              wxUint32 &dw_serial
                                            )
{
   /*----------------------------------------------------------------------*/
   int i_ret ;
   /*----------------------------------------------------------------------*/
   do
   {  i_ret = read_stream_header( i_packet_type, dw_serial ) ;
   } while( i_ret == 1 ) ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Supposed to start at the current file pointer position )-------------*/
int CFileInit_ogg::read_vorbis_comment( bool boo_video )
{
   /*----------------------------------------------------------------------*/
   wxString s_vendor     ;
   wxString s_var        ;
   wxString s_val        ;
   int      i_col        ;
   wxUint32 dw_nb_vector ;
   int      i_nb_vector  ;

   /*--( This function can be called "externally" too,so a reinit is done )*/
   m_fi.set_conv_from_utf8() ;

   /*--( Vendor info: Video will contain codec info )----------------------*/
   if( read_varchar_c( s_vendor ) != 0 )
   {  return( -1 ) ; }
   if( !s_vendor.empty() && !boo_video )
   {  m_f.val_s( COL_AUDIO_INFO ) = s_vendor ; }

   /*--( Number of variables )---------------------------------------------*/
   if( m_fa.read_le_data( dw_nb_vector ) != 0 )
   {  return( -2 ) ; }

   /*--( Treat all variables )---------------------------------------------*/
   for( i_nb_vector = dw_nb_vector ; i_nb_vector > 0 ; --i_nb_vector )
   {
      /*-------------------------------------------------------------------*/
      if( read_var_val( s_var, s_val ) != 0 ) { return( -3 ) ; }

      /*-------------------------------------------------------------------*/
      i_col = st_ogg_conv_tag_col( boo_video, s_var ) ;
      /*--( The column is reserved even if the value is empty )------------*/
      if( m_fi.reserve_col( i_col ) && !s_val.empty() )
      {
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = s_val ;
         /*----------------------------------------------------------------*/
         if( !boo_video && i_col == COL_AUDTAG_TRACK )
         {  m_fi.init_audio_num_and_nb_track( s_val ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !boo_video ) { m_f.init_info_s_track() ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ogg::ogg_read()
{
   /*--( Initializations to avoid compilation warnings )-------------------*/
   str_ogg_theora_identification_header ident_theora = { 0 } ; // Warning
   str_ogg_vorbis_identification_header ident_vorbis = { 0 } ; // Warning
   /*----------------------------------------------------------------------*/
   wxUint32              dw_serial_theora = 0     ;
   wxUint32              dw_serial_vorbis = 0     ;
   wxUint32              dw_serial                ;
   int                   i_packet_type            ;
   bool                  boo_has_video    = false ;
   bool                  boo_has_audio    = false ;
   double                do_fps           = 0     ;
   str_ogg_stream_header header_last              ;
   double                do_duration              ;
   wxFileOffset          fo_offset_last   = 0     ;
   int                   i_ret                    ;

   /*----------------------------------------------------------------------*/
   m_fi.set_conv_from_utf8() ;

   /*-----------------------------------------------------------------------+
   ! Identification                                                         !
   ! Assuming that the first packets are identification ones starting with  !
   ! the theora in case of a video file.                                    !
   +-----------------------------------------------------------------------*/
   if( read_useful_stream_header( i_packet_type, dw_serial ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   if( i_packet_type == THEORA_PT_IDENTIFICATION )
   {  /*-------------------------------------------------------------------*/
      boo_has_video = true ;
      dw_serial_theora = dw_serial ;
      if( read_be_data( ident_theora ) != 0 ) { return( -2 ) ; }
      /*--( If no audio, then a come back will be made for the comment )---*/
      fo_offset_last = m_fa.get_offset() ;
      /*--( Normally should follow the audio common header )---------------*/
      if( read_useful_stream_header( i_packet_type, dw_serial ) != 0 )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( i_packet_type == VORBIS_PT_IDENTIFICATION )
   {  /*-------------------------------------------------------------------*/
      boo_has_audio = true ;
      dw_serial_vorbis = dw_serial ;
      if( read_le_data( ident_vorbis ) != 0 ) { return( -4 ) ; }
      /*-------------------------------------------------------------------*/
   }
   else /*--( No audio, go back to start of stream )-----------------------*/
   if( m_fa.set_offset( fo_offset_last ) != 0 )
   {  return( -5 ) ; }

   /*--( At least one of those ... )---------------------------------------*/
   if( !boo_has_video && !boo_has_audio ) { return( -6 ) ; }

   /*--( Fill the information )--------------------------------------------*/
   if( boo_has_video )
   {
      /*--( The couple codec audio / video is known ... )------------------*/
      m_f.val_s( COL_VIDEO_INFO ).Printf( "Vorbis - Theora (%d.%d.%d)",
                                          ident_theora.b_vmaj,
                                          ident_theora.b_vmin,
                                          ident_theora.b_vrev
                                        ) ;
      /*-------------------------------------------------------------------*/
      m_fi.init_video_x_y( SR_THREECC( ident_theora.tb_b_fmbw[ 2 ],
                                       ident_theora.tb_b_fmbw[ 1 ],
                                       ident_theora.tb_b_fmbw[ 0 ]
                                     ),
                           SR_THREECC( ident_theora.tb_b_fmbh[ 2 ],
                                       ident_theora.tb_b_fmbh[ 1 ],
                                       ident_theora.tb_b_fmbh[ 0 ]
                                     )
                         ) ;
      /*-------------------------------------------------------------------*/
      if( ident_theora.dw_frd != 0 )
      {  do_fps =   ( double )ident_theora.dw_frn
                  / ( double )ident_theora.dw_frd ;
         m_fi.init_video_fps( do_fps ) ;
      }
      /*-------------------------------------------------------------------*/
      if( boo_has_audio )
      {  m_fi.init_video_channel( ident_vorbis.b_audio_channels ) ;
         m_fi.init_video_samprate( ident_vorbis.dw_audio_sample_rate ) ;
      }
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*-------------------------------------------------------------------*/
      m_fi.init_audio_channel( ident_vorbis.b_audio_channels ) ;
      m_fi.init_audio_samprate( ident_vorbis.dw_audio_sample_rate ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Then should come the "Comment"                                         !
   +-----------------------------------------------------------------------*/
   while(   (    ( i_ret = read_stream_header( i_packet_type, dw_serial )
                 ) == 0
              && i_packet_type != THEORA_PT_COMMENT
              && i_packet_type != VORBIS_PT_COMMENT
            )
         || ( i_ret == 1 && i_packet_type == OGG_SKELETON_30 )
        )
   {  ; }
   /*----------------------------------------------------------------------*/
   if( i_ret < 0 ) { return( -7 ) ; }

   /*----------------------------------------------------------------------*/
   if( i_ret != 1 && read_vorbis_comment( boo_has_video ) != 0 )
   {  return( -8 ) ; }

   /*-----------------------------------------------------------------------+
   ! Computation of the duration                                            !
   +-----------------------------------------------------------------------*/
   if( boo_has_video )
   {
      /*-------------------------------------------------------------------*/
      if( do_fps <= 0 ) { return( -9 ) ; }
      /*--( Search for the last stream with the right stream id )----------*/
      if( m_fa.search_last_seq( dw_serial_theora, st_co_i_ogg_max_seach_size,
                                fo_offset_last
                              ) != 0
        )
      {  return( -10 ) ; }

      /*--( Then comes the data )------------------------------------------*/
      if(    m_fa.skip_nb_byte( - ( int )offsetof( str_ogg_stream_header,
                                                   ___dw_page_sequence_no
                                                 )
                              ) != 0
          || read_le_data( header_last ) != 0
          || header_last.dw_capture_pattern != st_co_dw_OggS
        )
      {  return( -11 ) ; }

      /*-------------------------------------------------------------------*/
      int      i_kfgshift     ;
      wxUint64 ddw_framecount ;

      /*--( The "shift" of the granule value to extract the frame count )--*/
      i_kfgshift = ( ( ident_theora.w_qu_ks_pf >> 5 ) & 0x1F ) ;
      ddw_framecount  = header_last.ddw_abs_granule_pos >> i_kfgshift ;
      ddw_framecount +=   header_last.ddw_abs_granule_pos
                        - ( ddw_framecount << i_kfgshift ) ;
      do_duration = ddw_framecount / do_fps ;
      if( ( int )do_duration < 0 ) { return( -12 ) ; }
      /*-------------------------------------------------------------------*/
      m_fi.init_video_duration( ( int )do_duration ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   if( boo_has_audio  )
   {
      /*-------------------------------------------------------------------*/
      int i_bitrate ;
      /*-------------------------------------------------------------------*/
      if( ident_vorbis.dw_audio_sample_rate == 0 ) { return( -13 ) ; }

      /*--( Search for the last stream with the right stream id )----------*/
      if( m_fa.search_last_seq( dw_serial_vorbis, st_co_i_ogg_max_seach_size,
                                fo_offset_last
                              ) != 0
        )
      {  return( -14 ) ; }

      /*--( Then comes the data )------------------------------------------*/
      if(    m_fa.skip_nb_byte( - ( int )offsetof( str_ogg_stream_header,
                                                   ___dw_page_sequence_no
                                                 )
                              ) != 0
          || read_le_data( header_last ) != 0
          || header_last.dw_capture_pattern != st_co_dw_OggS
        )
      {  return( -15 ) ; }

      /*-------------------------------------------------------------------*/
      do_duration =   ( double )header_last.ddw_abs_granule_pos
                    / ident_vorbis.dw_audio_sample_rate ;
      if( ( int )do_duration < 0 ) { return( -16 ) ; }
      /*-------------------------------------------------------------------*/
      m_fi.init_audio_duration( ( int )do_duration ) ;
      /*--( The bitrate is computed based on the file size )---------------*/
        i_bitrate
      = ( int )( m_f.get_size().ToDouble() * 8 / 1000 / do_duration ) ;
      /*--( For now the bitrate is displayed as CBR ... )------------------*/
      m_fi.init_audio_bitrate( TBR_CBR, i_bitrate ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_fi.m_s_type_det = ( boo_has_video ? "ogv" : "ogg" ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_ogg()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "oga/ogv" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_ogg( *this ).ogg_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
!                                                                           !
! This function is directly used by other init modules ...                  !
!                                                                           !
+--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
int CFileInit::ogg_read_vorbis_comment( bool boo_video )
{  return( CFileInit_ogg( *this ).read_vorbis_comment( boo_video ) ) ; }

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_ogg.cpp                       !
+==========================================================================*/
