/*====================================+=====================================+
! Filer CFileInit_mpg.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                      Video file: ".mpg", ".mpeg"                          !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_mpg : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_mpg( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int search_next_seq( wxUint32 &dw_fm, int &i_search_size ) ;
      int read_audio_data( int i_mpg_version ) ;
      int read_video_data( int &i_mpg_version ) ;
      int mpg_read() ;
   /*----------------------------------------------------------------------*/
} ;


/*-------------------------------------------------------------------------*/
static const int st_co_i_mpg_max_search_size = 500 * 1024 ;

/*--( Frame Maker )--------------------------------------------------------*/
static const wxUint32 st_dw_FM_UNKNOWN   = SR_FOURCC( 0x00,0x00,0x00,0x00 ) ;
static const wxUint32 st_dw_FM_USER_DATA = SR_FOURCC( 0x00,0x00,0x01,0xB2 ) ;
static const wxUint32 st_dw_FM_VIDEO     = SR_FOURCC( 0x00,0x00,0x01,0xB3 ) ;
static const wxUint32 st_dw_FM_EXTENSION = SR_FOURCC( 0x00,0x00,0x01,0xB5 ) ;
static const wxUint32 st_dw_FM_GOP       = SR_FOURCC( 0x00,0x00,0x01,0xB8 ) ;
static const wxUint32 st_dw_FM_PROGRAM   = SR_FOURCC( 0x00,0x00,0x01,0xBA ) ;
static const wxUint32 st_dw_FM_PES_AUDIO = SR_FOURCC( 0x00,0x00,0x01,0xC0 ) ;
static const wxUint32 st_dw_FM_PES_VIDEO = SR_FOURCC( 0x00,0x00,0x01,0xE0 ) ;

/*-------------------------------------------------------------------------*/
static const double st_tb_do_picture_rates[]
= {
        0           ,
    24000.0 / 1001.0,
       24.0         ,
       25.0         ,
    30000.0 / 1001.0,
       30.0         ,
       50.0         ,
    60000.0 / 1001.0,
       60.0
  } ;

/*-------------------------------------------------------------------------*/
static wxUint32 st_seq_is_frame_marker( wxUint32 dw_val )
{
   /*----------------------------------------------------------------------*/
   switch( dw_val )
   {
      /*-------------------------------------------------------------------*/
      case st_dw_FM_USER_DATA : return( st_dw_FM_USER_DATA ) ;
      case st_dw_FM_EXTENSION : return( st_dw_FM_EXTENSION ) ;
      case st_dw_FM_PROGRAM   : return( st_dw_FM_PROGRAM   ) ;
      case st_dw_FM_VIDEO     : return( st_dw_FM_VIDEO     ) ;
      case st_dw_FM_GOP       : return( st_dw_FM_GOP       ) ;
      case st_dw_FM_PES_AUDIO : return( st_dw_FM_PES_AUDIO ) ;
      case st_dw_FM_PES_VIDEO : return( st_dw_FM_PES_VIDEO ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( st_dw_FM_UNKNOWN ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mpg::search_next_seq( wxUint32 &dw_fm, int &i_search_size )
{
   /*----------------------------------------------------------------------*/
   wxUint8  tb_b_buffer[ 1024 * 5 ] ;
   size_t   sz_len_buffer           ;
   bool     boo_first_buffer = true ;
   wxUint32 dw_val           = 0    ;
   int      i_num                   ;
   int      i_overread              ;
   /*----------------------------------------------------------------------*/
   if( i_search_size <= 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   do
   {  /*-------------------------------------------------------------------*/
        sz_len_buffer
      = wxMin( ( int )sizeof( tb_b_buffer ), i_search_size ) ;
      if( m_fa.read_buffer_max( sz_len_buffer, tb_b_buffer ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      if( !boo_first_buffer )
      {  i_num = 0 ; }
      else
      {  /*----------------------------------------------------------------*/
         boo_first_buffer = false ;
         i_num = sizeof( dw_val ) ;
         if( sz_len_buffer >= sizeof( dw_val ) )
         {  dw_val = sr::pb_to_ui32( tb_b_buffer ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      for( ;  i_num < ( int )sz_len_buffer
           && ( dw_fm = st_seq_is_frame_marker( dw_val )
              ) == st_dw_FM_UNKNOWN ;
           ++i_num
         )
      {  /*----------------------------------------------------------------*/
         dw_val >>= 8                         ;
         dw_val |= tb_b_buffer[ i_num ] << 24 ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      i_search_size -= sz_len_buffer ;
      /*-------------------------------------------------------------------*/
   } while(    dw_fm == st_dw_FM_UNKNOWN
            && sz_len_buffer == sizeof( tb_b_buffer )
            && i_search_size > 0
          ) ;

   /*--( Not found ? )-----------------------------------------------------*/
   if( dw_fm == st_dw_FM_UNKNOWN ) { return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   i_overread = ( int )sz_len_buffer - i_num ;
   /*--( Set the pos just after the sequence )-----------------------------*/
   if( m_fa.skip_nb_byte( - i_overread ) != 0 )
   {  return( -4 ) ; }
   /*----------------------------------------------------------------------*/
   i_search_size += i_overread ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static void st_gop_nb_sec( wxUint32 dw_tms, int &i_nb_sec, int &i_nb_pic )
{
   /*----------------------------------------------------------------------*/
   int i_hour   ;
   int i_minute ;
   int i_second ;
   /*----------------------------------------------------------------------*/
   // xHHHHHMM MMMMxSSS SSSPPPPP Pxxxxxxx
   // 0x00     0x08     0x00     0x40
   // 00000000 00001000 00000000 01000000
   // 0x00     0x08     0x21     0x80
   // 00000000 00001000 00100001 10000000
   /*----------------------------------------------------------------------*/
#define EXTR_BYTE( dw, num ) ( ( dw / ( 1 << ( 8 * num ) ) ) % 256 )
   /*----------------------------------------------------------------------*/
   i_hour    = ( EXTR_BYTE( dw_tms, 0 ) >> 2 ) & 0x1F ;
   i_minute  = (   ( EXTR_BYTE( dw_tms, 0 ) << 4 )
                 | ( EXTR_BYTE( dw_tms, 1 ) >> 4 )
               ) & 0x3F ;
   i_second  = (   ( EXTR_BYTE( dw_tms, 1 ) << 3 )
                 | ( EXTR_BYTE( dw_tms, 2 ) >> 5 )
               ) & 0x3F ;
   i_nb_pic  = (   ( EXTR_BYTE( dw_tms, 2 ) << 1 )
                 | ( EXTR_BYTE( dw_tms, 3 ) >> 7 )
               ) & 0x3F ;
   /*----------------------------------------------------------------------*/
   i_nb_sec = i_hour * 3600 + i_minute * 60 + i_second ;
   /*----------------------------------------------------------------------*/
#undef EXTR_BYTE
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mpg::read_audio_data( int i_mpg_version )
{
   /*----------------------------------------------------------------------*/
   wxUint8            tb_b_buffer[ 256 ] ;
   size_t             sz_len_buffer      ;
   str_mp3_frame_info s_fi               ;
   int                i_pos = 0          ;

   /*----------------------------------------------------------------------*/
   sz_len_buffer = sizeof( tb_b_buffer ) ;
   if( m_fa.read_buffer_max( sz_len_buffer, tb_b_buffer ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( sz_len_buffer < 20 ) { return( -2 ) ; }
   /*--( Set pos in front of the frame )-----------------------------------*/
   switch( i_mpg_version )
   {
      /*-------------------------------------------------------------------*/
      case 1 :
         /*--( Skip the size )---------------------------------------------*/
         i_pos = 2 ;
         /*--( Jump over the "stuff" byte )--------------------------------*/
         while(    ( tb_b_buffer[ i_pos ] & 0x80 ) != 0
                && i_pos < ( int )sz_len_buffer
              )
         {  ++i_pos ; }
         /*----------------------------------------------------------------*/
         if( sz_len_buffer - i_pos < 20 ) { return( -3 ) ; }
         /*--( Jump over the two following bytes )-------------------------*/
         if( ( tb_b_buffer[ i_pos ] & 0xC0 ) == 0x40 )
         {  i_pos += 2 ; }
         /*----------------------------------------------------------------*/
         switch( tb_b_buffer[ i_pos ] & 0xF0 )
         {
            /*-------------------------------------------------------------*/
            case 0x20 : i_pos +=  5 ; break ;
            case 0x30 : i_pos += 10 ; break ;
            default : ++i_pos ; break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      case 2 :
         /*--( It's a PES )------------------------------------------------*/
         i_pos = 5 + tb_b_buffer[ 4 ] ;
         if( sz_len_buffer - i_pos < 20 ) { return( -4 ) ; }
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      default :
         /*----------------------------------------------------------------*/
         i_pos = 10 ;
         break ;
         /*----------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( sz_len_buffer - i_pos < 10 ) { return( -5 ) ; }

   /*--( Normally here begins a Mpeg audio frame )-------------------------*/
   if( m_fi.mp3_extr_frame_info( sr::pb_to_ui32( &tb_b_buffer[ i_pos ] ),
                                 s_fi
                               ) != 0
     )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.skip_nb_byte( - ( int )sz_len_buffer + i_pos ) != 0 )
      {  return( -6 ) ; }
      /*-------------------------------------------------------------------*/
      if( m_fi.mp3_search_next_frame_header( s_fi, 1024 ) != 0 )
      {  return( -7 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_fi.init_video_samprate( s_fi.i_samprate ) ;
   m_fi.init_col_mp3_channel( COL_VIDEO_CHANNEL, s_fi.i_channel ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( It will also determine the mpeg type )-------------------------------*/
int CFileInit_mpg::read_video_data( int &i_mpg_version )
{
   /*----------------------------------------------------------------------*/
   wxUint32     dw_fm                         ;
   wxUint8      tb_b_buffer[ 7 ]              ;
   int          i_x                           ;
   int          i_y                           ;
   int          i_picture_rate_index          ;
   double       do_fps     =  0               ;
   //int        i_aspect_ratio                ; // For now unused
   int          i_duration_based_on_size = -1 ;
   int          i_duration_based_on_gop  = -1 ;
   unsigned int ui_bitrate =  0               ;
   int          i_search_size_GOP             ;

   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tb_b_buffer ), tb_b_buffer ) != 0 )
   {  return( -1 ) ; }
   /*-----------------------------------------------------------------------+
   ! Extraction of the video information                                    !
   ! XX XY YY AB                                                            !
   ! X:Width, Y:Height, A:Aspect ratio, B:picture rate index                !
   +-----------------------------------------------------------------------*/
   i_x = ( ( tb_b_buffer[ 0 ] << 8 ) + tb_b_buffer[ 1 ] ) >> 4 & 0x0FFF ;
   i_y = ( ( tb_b_buffer[ 1 ] << 8 ) + tb_b_buffer[ 2 ] )      & 0x0FFF ;
   //i_aspect_ratio     = ( tb_b_buffer[ 3 ] >> 4 ) & 0x0F              ;
   i_picture_rate_index = tb_b_buffer[ 3 ] & 0x0F                       ;
   /*----------------------------------------------------------------------*/
   m_fi.init_video_x_y( i_x, i_y ) ;
   /*----------------------------------------------------------------------*/
   if( i_picture_rate_index < ( int )WXSIZEOF( st_tb_do_picture_rates ) )
   {  /*-------------------------------------------------------------------*/
      do_fps = st_tb_do_picture_rates[ i_picture_rate_index ] ;
      /*-------------------------------------------------------------------*/
      if( do_fps > 0 ) { m_fi.init_video_fps( do_fps ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( The next 18 bits are the bitrate / 400 )--------------------------*/
   ui_bitrate = ( tb_b_buffer[ 4 ] << 8 ) + tb_b_buffer[ 5 ] ;
   ui_bitrate <<= 2 ;
   ui_bitrate |= tb_b_buffer[ 6 ] >> 6 ;

   /*--( Ok ? )------------------------------------------------------------*/
   if( ui_bitrate < 10 || ui_bitrate > 20000 )
   {  ui_bitrate = 0 ; }

   /*-----------------------------------------------------------------------+
   ! By default the duration is based on the size: not very accurate        !
   +-----------------------------------------------------------------------*/
   if( ui_bitrate != 0 )
   {    i_duration_based_on_size
      = sr::round_0(   ( m_f.get_size().ToDouble() * 8.0 )
                     / ( ui_bitrate * 400.0 )
                   ) ;
   }

   /*----------------------------------------------------------------------*/
   i_search_size_GOP = st_co_i_mpg_max_search_size ;

   /*--( Stop on a GOP )---------------------------------------------------*/
   while(    search_next_seq( dw_fm, i_search_size_GOP ) == 0
          && dw_fm != st_dw_FM_GOP
        )
   {
      /*-------------------------------------------------------------------*/
      switch( dw_fm )
      {
         /*----------------------------------------------------------------*/
         case st_dw_FM_USER_DATA :
         {
            /*-------------------------------------------------------------*/
            wxUint8  tb_b_info[ g_co_i_string_sz_max ] ;
            wxString s_info                            ;
            size_t   sz_size                           ;
            int      i_overread                        ;
            /*-------------------------------------------------------------*/
            if( m_fa.read_buffer( sizeof( tb_b_info ), tb_b_info ) != 0 )
            {  return( -3 ) ; }
            /*--( The data are "closed" by the next FM => 00 00 01 )-------*/
            for( sz_size = 0 ;
                    sz_size < sizeof( tb_b_info )
                 && tb_b_info[ sz_size ] != '\0' ;
                 ++sz_size
               )
            {  ; }
            /*-------------------------------------------------------------*/
            if( m_fi.prepare_string( ( char * )tb_b_info, sz_size, s_info
                                   ) > 0
              )
            {  m_f.val_s( COL_VIDEO_INFO ) = s_info ; }
            /*-------------------------------------------------------------*/
            i_overread = sizeof( tb_b_info ) - sz_size ;
            /*--( Back ... )-----------------------------------------------*/
            if( m_fa.skip_nb_byte( - i_overread ) != 0 )
            {  return( -4 ) ; }
            /*-------------------------------------------------------------*/
            i_search_size_GOP += i_overread ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case st_dw_FM_EXTENSION :
         {
            /*-------------------------------------------------------------*/
            wxUint8 b_ext_id ;
            /*-------------------------------------------------------------*/
            if( m_fa.read_data( b_ext_id ) != 0 )
            {  return( -5 ) ; }
            /*-------------------------------------------------------------*/
            --i_search_size_GOP ;
            /*-------------------------------------------------------------*/
            if( ( ( b_ext_id >> 4 ) & 0x0F ) == 1 )
            {  i_mpg_version = 2 ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Try to get the duration based on first/last GOP )-----------------*/
   if( dw_fm == st_dw_FM_GOP )
   {
      /*-------------------------------------------------------------------*/
      wxUint32     dw_gop_tms    ;
      int          i_nb_sec_from ;
      int          i_nb_pic      ;
      wxFileOffset fo_offset     ;
      /*-------------------------------------------------------------------*/
      if( m_fa.read_data( dw_gop_tms ) != 0 )
      {  return( -6 ) ; }
      st_gop_nb_sec( dw_gop_tms, i_nb_sec_from, i_nb_pic ) ;
      /*-------------------------------------------------------------------*/
      if( m_fa.search_last_seq( st_dw_FM_GOP, st_co_i_mpg_max_search_size,
                                fo_offset
                              ) == 0
        )
      {  /*----------------------------------------------------------------*/
         int i_nb_sec_to ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_le_data( dw_gop_tms ) != 0 )
         {  return( -7 ) ; }
         st_gop_nb_sec( dw_gop_tms, i_nb_sec_to, i_nb_pic ) ;
         /*----------------------------------------------------------------*/
         i_duration_based_on_gop = i_nb_sec_to - i_nb_sec_from ;
         /*----------------------------------------------------------------*/
         if( do_fps > 0 )
         {  i_duration_based_on_gop += sr::round_0( i_nb_pic / do_fps ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Is there a duration ? )-------------------------------------------*/
   if( i_duration_based_on_size >= 0 || i_duration_based_on_gop >= 0 )
   {
      /*--( Which one to take into account ? )-----------------------------*/
      m_fi.init_video_duration(    i_duration_based_on_size >= 0
                                && (    i_duration_based_on_gop < 0
                                     ||   i_duration_based_on_size
                                        > 5 * i_duration_based_on_gop
                                     ||   i_duration_based_on_size
                                        < i_duration_based_on_gop / 5
                                   )
                                ? i_duration_based_on_size
                                : i_duration_based_on_gop
                              ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mpg::mpg_read()
{
   /*----------------------------------------------------------------------*/
   int      i_mpg_version = 1 ;
   wxUint32 dw_fm             ;

   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( dw_fm ) != 0 )
   {  return( -1 ) ; }

   /*--( Not a mpeg ? )----------------------------------------------------*/
   if(    ( dw_fm & SR_FOURCC( 0xFF, 0xFF, 0xFF, 0x00 ) )
       != SR_FOURCC( 0x00, 0x00, 0x01, 0x00 )
     )
   {  return( -2 ) ; }

   /*--( Video only ? )----------------------------------------------------*/
   if( dw_fm == st_dw_FM_VIDEO )
   {  if( read_video_data( i_mpg_version ) != 0 ) {  return( -3 ) ; } }
   else /*--( Audio and video ... )----------------------------------------*/
   if( dw_fm == st_dw_FM_PROGRAM )
   {
      /*--( Look for audio and video information )-------------------------*/
      wxUint32     dw_fm                ;
      wxFileOffset fo_offset_video = -1 ;
      wxFileOffset fo_offset_audio = -1 ;
      int          i_search_size   = st_co_i_mpg_max_search_size ;
      /*-------------------------------------------------------------------*/
      while(    search_next_seq( dw_fm, i_search_size ) == 0
             && ( fo_offset_video < 0 || fo_offset_audio < 0 )
           )
      {  /*----------------------------------------------------------------*/
         if( dw_fm == st_dw_FM_VIDEO )
         {  if( fo_offset_video == -1 )
            {  fo_offset_video = m_fa.get_offset() ; }
         }
         else
         if( dw_fm == st_dw_FM_PES_AUDIO )
         {  if( fo_offset_audio == -1 )
            {  fo_offset_audio = m_fa.get_offset() ; }
         }
         /*----------------------------------------------------------------*/
      }
      /*--------------------------------------------------------------------+
      ! Start with the video because it "computes" the mpeg version that    !
      ! is needed for the audio part                                        !
      +--------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_video ) == 0 )
      {  read_video_data( i_mpg_version ) ; }
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_audio ) == 0 )
      {  read_audio_data( i_mpg_version ) ; }
      /*-------------------------------------------------------------------*/
   }
   else /*--( File KO )----------------------------------------------------*/
   {  return( -4 ) ; }

   /*----------------------------------------------------------------------*/
   if( i_mpg_version == 1 || i_mpg_version == 2 )
   {
      /*-------------------------------------------------------------------*/
      wxString s_info = m_f.val_s( COL_VIDEO_INFO ) ;
      wxString s_version ;
      /*-------------------------------------------------------------------*/
      s_version.Printf( "Mpeg%d", i_mpg_version ) ;
      if( !s_info.empty() ) { s_version += " - " ; }
      /*-------------------------------------------------------------------*/
      m_f.val_s( COL_VIDEO_INFO ) = s_version + s_info ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_mpg()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "mpg" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_mpg( *this ).mpg_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_mpg.cpp                       !
+==========================================================================*/
