/*====================================+=====================================+
! File CFileInit_mpc.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                      Audio file: ".mpc", ".mpp"                           !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_mpc_stream_header ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_mpc : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_mpc( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_le_data( str_mpc_stream_header &mpc_sh ) ;
      int mpc_read( wxFileOffset fo_offset_end ) ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_mpc_stream_header
{
   /*----------------------------------------------------------------------*/
   char     tb_c_id[ 3 ] ; // MP+
   wxUint8  b_version    ; // bits 0..3 significant
   wxUint32 dw_nb_frames ;
   wxUint32 dw_infos     ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_mpc::read_le_data( str_mpc_stream_header &mpc_sh )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( mpc_sh ), &mpc_sh ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   mpc_sh.dw_nb_frames = wxUINT32_SWAP_ALWAYS( mpc_sh.dw_nb_frames ) ;
   mpc_sh.dw_infos     = wxUINT32_SWAP_ALWAYS( mpc_sh.dw_nb_frames ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static const int st_tb_i_samprate[] = { 44100, 48000, 37800, 32000 } ;

/*-------------------------------------------------------------------------*/
static const char * const st_mpc_profile( const str_mpc_stream_header &sh )
{
   /*----------------------------------------------------------------------*/
   switch( ( sh.dw_infos >> 20 ) & 0x0F )
   {
      /*-------------------------------------------------------------------*/
      case  1 : return( "Unstable/Experimental" ) ;
      case  5 :
      case  6 : return( "below Telephone" )       ;
      case  7 : return( "Telephone" )             ;
      case  8 : return( "Thumb" )                 ;
      case  9 : return( "Radio" )                 ;
      case 10 : return( "Standard" )              ;
      case 11 : return( "Xtreme" )                ;
      case 12 : return( "Insane" )                ;
      case 13 : return( "BrainDead" )             ;
      case 14 :
      case 15 : return( "above BrainDead" )       ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( "N/A" ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mpc::mpc_read( wxFileOffset fo_offset_end )
{
   /*----------------------------------------------------------------------*/
   str_mpc_stream_header header        ;
   int                   i_samprate    ;
   int                   i_bitrate     ;
   int                   i_duration    ;
   int                   i_pcm_samples ;

   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( 0 ) != 0 || read_le_data( header ) != 0 )
   {  return( -1 ) ; }

   /*--( At least Stream Version 7 )---------------------------------------*/
   if( memcmp( header.tb_c_id, "MP+", 3 ) != 0 )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   if( header.b_version > 0x71 ) { return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   i_samprate = st_tb_i_samprate [ ( header.dw_infos >> 16 ) & 0x03 ] ;

   /*----------------------------------------------------------------------*/
   m_f.val_s( COL_AUDIO_INFO ).Printf( "SV%d.%d %s",
                                       header.b_version & 0x0F,
                                       ( header.b_version >> 4 ) & 0x0F,
                                       st_mpc_profile( header )
                                     ) ;

   /*--( The bitrate is "average" )----------------------------------------*/
   i_pcm_samples = 1152 * header.dw_nb_frames - 576 ;
   if( i_pcm_samples > 0 )
   {  i_bitrate = sr::round_0(   fo_offset_end * 8.0 / 1000
                               * i_samprate / i_pcm_samples
                             ) ;
   }
   else
   {  i_bitrate = 0 ; }

   /*--( 36 * 32 samples per frame: the samprate is surely != 0 )----------*/
   i_duration = (int)( header.dw_nb_frames * ( 36.0 * 32.0 ) / i_samprate ) ;

   /*----------------------------------------------------------------------*/
   if( i_bitrate  > 0  ) { m_fi.init_audio_bitrate( TBR_ABR, i_bitrate ) ; }
   if( i_duration >= 0 ) { m_fi.init_audio_duration( i_duration ) ; }
   if( i_samprate > 0  ) { m_fi.init_audio_samprate( i_samprate ) ; }

   /*--( Always stereo )---------------------------------------------------*/
   m_fi.init_audio_channel( 2 ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_mpc()
{
   /*----------------------------------------------------------------------*/
   wxFileOffset fo_offset_end ;

   /*----------------------------------------------------------------------*/
   m_s_type_det = "mpc" ;

   /*--( If no APE tag then try the id3v1 one )----------------------------*/
   if( ape_init_apetag( fo_offset_end ) != 0 )
   {  mp3_read_id3v1( fo_offset_end ) ; }

   /*----------------------------------------------------------------------*/
   int i_ret = CFileInit_mpc( *this ).mpc_read( fo_offset_end ) ;
   /*----------------------------------------------------------------------*/
   m_f.init_info_s_track() ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_mpc.cpp                       !
+==========================================================================*/
