/*====================================+=====================================+
! File CFileInit_mp3.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                     Audio file: ".mp3", ".mp2"                            !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CApp.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_id3v23_frame_header ;



/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_mp3 : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_mp3( CFileInit &parent )
                   : CFileInit_type_base( parent ),
                     m_i_id3v2_version( 0 ) // for cppcheck
      {  ; }
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      int m_i_id3v2_version ;
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      enum e_strid3v2_flags
           { STRID3V2_NONE = 0,
             STRID3V2_LANG = 1 << 0,
             STRID3V2_DESC = 1 << 1
           } ;
      int string_id3v2( size_t sz_size, wxString &s,
                        wxUint8 b_strid3v2_flags = STRID3V2_NONE
                      ) ;
      int read_id3v22_frame_header( str_id3v23_frame_header &id3v23,
                                    int &i_size
                                  ) ;
      int read_frame_header( str_id3v23_frame_header &id3v23,
                             int &i_size
                           ) ;
      int image_info( size_t sz_size ) ;
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int extr_frame_info( wxUint32 dw_val, str_mp3_frame_info &s_fi ) ;
      int search_next_frame_header( str_mp3_frame_info &s_fi,
                                    int i_max_read_len
                                  ) ;
      int init_id3v1( wxFileOffset &fo_offset_data_end ) ;
      int init_id3v2() ;
      int init_general_info( size_t sz_size ) ;
   /*----------------------------------------------------------------------*/
} ;



/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_id3v1
{
   /*----------------------------------------------------------------------*/
   char    tb_c_ident  [  3 ] ;
   char    tb_c_title  [ 30 ] ;
   char    tb_c_artist [ 30 ] ;
   char    tb_c_album  [ 30 ] ;
   char    tb_c_year   [  4 ] ;
   char    tb_c_comment[ 29 ] ;
   wxUint8 b_track            ;
   wxUint8 b_genre            ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_id3v2_header
{
   /*----------------------------------------------------------------------*/
   wxUint8  tb_b_ident  [ 3 ] ;
   wxUint8  tb_b_version[ 2 ] ;
   wxUint8  b_flags           ;
   wxUint32 dw_size           ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_id3v22_frame_header
{
   /*----------------------------------------------------------------------*/
   wxUint8 tb_b_id  [ 3 ] ;
   wxUint8 tb_b_size[ 3 ] ;
   /*----------------------------------------------------------------------*/
} ;

/*--( id32.4 header is the same as the 2.3 one )---------------------------*/
struct str_id3v23_frame_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id           ;
   wxUint32 dw_size         ;
   wxUint8  tb_b_flags[ 2 ] ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
static const char * const st_tb_c_id3_genre[]
=
{
/*-------------------------------------------------------------------------*/
"Blues"           , "Classic Rock"    , "Country"         , "Dance"         ,
"Disco"           , "Funk"            , "Grunge"          , "Hip-Hop"       ,
"Jazz"            , "Metal"           , "New Age"         , "Oldies"        ,
"Other"           , "Pop"             , "R&B"             , "Rap"           ,
"Reggae"          , "Rock"            , "Techno"          , "Industrial"    ,
"Alternative"     , "Ska"             , "Death Metal"     , "Pranks"        ,
"Soundtrack"      , "Euro-Techno"     , "Ambient"         , "Trip-Hop"      ,
"Vocal"           , "Jazz+Funk"       , "Fusion"          , "Trance"        ,
"Classical"       , "Instrumental"    , "Acid"            , "House"         ,
"Game"            , "Sound Clip"      , "Gospel"          , "Noise"         ,
"Alt. Rock"       , "Bass"            , "Soul"            , "Punk"          ,
"Space"           , "Meditative"   , "Instrumental Pop", "Instrumental Rock",
"Ethnic"          , "Gothic"       , "Darkwave"        , "Techno-Industrial",
"Electronic"      , "Pop-Folk"        , "Eurodance"       , "Dream"         ,
"Southern Rock"   , "Comedy"          , "Cult"            , "Gangsta"       ,
"Top 40"          , "Christian Rap"   , "Pop/Funk"        , "Jungle"        ,
"Native American" , "Cabaret"         , "New Wave"        , "Psychadelic"   ,
"Rave"            , "Showtunes"       , "Trailer"         , "Lo-Fi"         ,
"Tribal"          , "Acid Punk"       , "Acid Jazz"       , "Polka"         ,
"Retro"           , "Musical"         , "Rock & Roll"     , "Hard Rock"     ,
/*--( Winamp & Lame extensions )-------------------------------------------*/
"Folk"            , "Folk-Rock"       , "National Folk"   , "Swing"         ,
"Fast Fusion"     , "Bebob"           , "Latin"           , "Revival"       ,
"Celtic"          , "Bluegrass"       , "Avantgarde"      , "Gothic Rock"   ,
"Progressive Rock", "Psychedelic Rock", "Symphonic Rock"  , "Slow Rock"     ,
"Big Band"        , "Chorus"          , "Easy Listening"  , "Acoustic"      ,
"Humour"          , "Speech"          , "Chanson"         , "Opera"         ,
"Chamber Music"   , "Sonata"          , "Symphony"        , "Booty Bass"    ,
"Primus"          , "Porn Groove"     , "Satire"          , "Slow Jam"      ,
"Club"            , "Tango"           , "Samba"           , "Folklore"      ,
"Ballad"          , "Power Ballad"    , "Rhythmic Soul"   , "Freestyle"     ,
"Duet"            , "Punk Rock"       , "Drum Solo"       , "A Capella"     ,
"Euro-House"      , "Dance Hall"      , "Goa"             , "Drum & Bass"   ,
"Club-House"      , "Hardcore"        , "Terror"          , "Indie"         ,
"BritPop"         , "Negerpunk"       , "Polsk Punk"      , "Beat"          ,
"Christian Gangsta","Heavy Metal"     , "Black Metal"     , "Crossover"     ,
"Contemporary C"  , "Christian Rock"  , "Merengue"        , "Salsa"         ,
"Thrash Metal"    , "Anime"           , "JPop"            , "SynthPop"
/*-------------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! Bitrate based on the MPEG type and the layer: [ mpeg type ][ value ][ L ] !
! 0: Layer I                                                                !
! 1: Layer II                                                               !
! 2: Layer III                                                              !
+--------------------------------------------------------------------------*/
static const int
         st_tb_tb_i_bitrate[ str_mp3_frame_info::MPEG_TYPE_NB ][ 16 ][ 3 ]
       = {
           { // mpeg type 1
             {   0,   0,   0 },
             {  32,  32,  32 },
             {  64,  48,  40 },
             {  96,  56,  48 },
             { 128,  64,  56 },
             { 160,  80,  64 },
             { 192,  96,  80 },
             { 224, 112,  96 },
             { 256, 128, 112 },
             { 288, 160, 128 },
             { 320, 192, 160 },
             { 352, 224, 192 },
             { 384, 256, 224 },
             { 416, 320, 256 },
             { 448, 384, 320 },
             {  -1,  -1,  -1 }
           },
           { // mpeg type 2
             {   0,   0,   0 },
             {  32,   8,   8 },
             {  48,  16,  16 },
             {  56,  24,  24 },
             {  64,  32,  32 },
             {  80,  40,  40 },
             {  96,  48,  48 },
             { 112,  56,  56 },
             { 128,  64,  64 },
             { 144,  80,  80 },
             { 160,  96,  96 },
             { 176, 112, 112 },
             { 192, 128, 128 },
             { 224, 144, 144 },
             { 256, 160, 160 },
             {  -1,  -1,  -1 }
           }
         } ;

/*--------------------------------------------------------------------------+
! Sampling rate based on the MPEG version: [ value ][ MPEG ]                !
! 0: MPEG1                                                                  !
! 1: MPEG2                                                                  !
! 2: MPEG 2.5                                                               !
+--------------------------------------------------------------------------*/
static const int st_tb_tb_i_samprate[ 4 ][ 3 ]
       = { { 44100, 22050, 11025 },
           { 48000, 24000, 12000 },
           { 32000, 16000,  8000 },
           {    -1,    -1,    -1 }
         } ;

/*--------------------------------------------------------------------------+
! Info by mpeg type and layer : [ mpeg type ][ layer ]                      !
! 0: Layer I                                                                !
! 1: Layer II                                                               !
! 2: Layer III                                                              !
+--------------------------------------------------------------------------*/
static const int
         st_tb_i_nb_sample_per_frame[ str_mp3_frame_info::MPEG_TYPE_NB ][ 3 ]
       = { { 384, 1152, 1152 }, // mpeg type 1
           { 384, 1152,  576 }  // mpeg type 2
         } ;
static const int st_tb_i_mul_fact[ str_mp3_frame_info::MPEG_TYPE_NB ][ 3 ]
       = { { 48000, 144000, 144000 } , // mpeg type 1
           { 24000,  72000,  72000 }   // mpeg type 2
         } ;

/*--( String encoding type )-----------------------------------------------*/
enum e_enc_type { ET_ISO, ET_U16_BOM, ET_U16_NOBOM, ET_UTF8, ET_NB } ;

/*--------------------------------------------------------------------------+
!                                                                           !
! All sizes are stored "big endian" but some of them can be "synchronized". !
! Only the size of the general header seems concerned.                      !
! It seems that "synchronized" means:                                       !
!      - Big endian                                                         !
!      - Eight th bit of each byte removed (rest shifted)                   !
!                                                                           !
+--------------------------------------------------------------------------*/
static wxUint32 st_conv_id3v2_synch_size( wxUint32 dw_size )
{
   /*----------------------------------------------------------------------*/
   return( wxUINT32_SWAP_ON_BE(   ( ( dw_size       ) & 0x7F ) << ( 3 * 7 )
                                | ( ( dw_size >>  8 ) & 0x7F ) << ( 2 * 7 )
                                | ( ( dw_size >> 16 ) & 0x7F ) << ( 1 * 7 )
                                | ( ( dw_size >> 24 ) & 0x7F )
                              )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::string_id3v2( size_t sz_size, wxString &s,
                                 wxUint8 b_strid3v2_flags
                               )
{
   /*----------------------------------------------------------------------*/
   char            tb_c_val[ g_co_i_string_sz_max ] ;
   char            *p_c_val                         ;
   e_enc_type      enc_type                         ;
   sr::e_conv_type conv_type                        ;
   size_t          sz_size_to_skip = 0              ;

   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   if( sz_size == 0 ) { return( -1 ) ; }

   /*--( Limit the amount of data )----------------------------------------*/
   if( sz_size > sizeof( tb_c_val ) )
   {  sz_size_to_skip = sz_size - sizeof( tb_c_val ) ;
      sz_size         = sizeof( tb_c_val ) ;
   }
   /*----------------------------------------------------------------------*/
   if(    m_fa.read_buffer( sz_size, tb_c_val ) != 0
       || (    sz_size_to_skip > 0
            && m_fa.skip_nb_byte( sz_size_to_skip ) != 0
          )
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   p_c_val = tb_c_val ;

   /*-----------------------------------------------------------------------+
   ! What is the text format ?                                              !
   ! 0x00 : Iso                    ending 00                                !
   ! 0x01 : Unicode 16 with BOM    ending 00 00                             !
   ! 0x02 : Unicode 16 without BOM ending 00 00                             !
   ! 0x03 : Unicode  8             ending 00                                !
   +-----------------------------------------------------------------------*/
   enc_type = ( e_enc_type )*p_c_val ;
   if( enc_type >= 0 && enc_type < ET_NB )
   {  ++p_c_val ; --sz_size ; }
   else
   {  enc_type = ET_NB ; }

   /*--( Skip the language if present )------------------------------------*/
   if( ( b_strid3v2_flags & STRID3V2_LANG ) != 0 )
   {  if( sz_size < 3 ) { return( -3 ) ; }
      p_c_val += 3 ; sz_size -= 3 ;
   }

   /*--( Skip the description if present )---------------------------------*/
   if( ( b_strid3v2_flags & STRID3V2_DESC ) != 0 )
   {
      /*--( Skip the '\0' at the end of the string too )-------------------*/
      if( enc_type == ET_U16_BOM || enc_type == ET_U16_NOBOM )
      {  while( sz_size > 0 && *( wxUint16 * )p_c_val != '\0' )
         {  sz_size -= 2 ; p_c_val += 2 ; }
         if( sz_size >= 2 ) { sz_size -= 2 ; p_c_val += 2 ; }
      }
      else
      {  while( sz_size > 0 && *p_c_val != '\0' )
         {  --sz_size ; ++p_c_val ; }
         if( sz_size > 0 ) { --sz_size ; ++p_c_val ; }
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( And now comes the string )----------------------------------------*/
   switch( enc_type )
   {  /*-------------------------------------------------------------------*/
      case ET_U16_BOM   : conv_type = sr::CONV_FROM_UTF16LEBE ; break ;
      case ET_U16_NOBOM : conv_type = sr::CONV_FROM_UTF16LE   ; break ;
      case ET_UTF8      : conv_type = sr::CONV_FROM_UTF8      ; break ;
      default           : conv_type = sr::CONV_FROM_8BIT      ; break ;
      /*-------------------------------------------------------------------*/
   }
   sr::prepare_string( p_c_val, sz_size, s, conv_type ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_id3v2_conv_tag_col( wxUint32 dw_tag )
{
   /*----------------------------------------------------------------------*/
   switch( dw_tag )
   {
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'T','I','T','2' ) : return( COL_AUDTAG_TITLE     ) ;
      case SR_FOURCC( 'T','P','E','1' ) : return( COL_AUDTAG_ARTIST    ) ;
      case SR_FOURCC( 'T','A','L','B' ) : return( COL_AUDTAG_ALBUM     ) ;
      case SR_FOURCC( 'T','Y','E','R' ) : return( COL_AUDTAG_YEAR      ) ;
      case SR_FOURCC( 'C','O','M','M' ) : return( COL_AUDTAG_COMMENT   ) ;
      case SR_FOURCC( 'T','R','C','K' ) : return( COL_AUDTAG_TRACK     ) ;
      case SR_FOURCC( 'T','P','O','S' ) : return( COL_AUDTAG_DISK      ) ;
      case SR_FOURCC( 'T','C','O','N' ) : return( COL_AUDTAG_GENRE     ) ;
      case SR_FOURCC( 'T','C','O','M' ) : return( COL_AUDTAG_COMPOS    ) ;
      case SR_FOURCC( 'T','O','P','E' ) : return( COL_AUDTAG_ORG_ART   ) ;
      case SR_FOURCC( 'T','C','O','P' ) : return( COL_AUDTAG_COPYRIGHT ) ;
      case SR_FOURCC( 'W','X','X','X' ) : return( COL_AUDTAG_URL       ) ;
      case SR_FOURCC( 'T','E','N','C' ) : return( COL_AUDTAG_ENCOD_BY  ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The idea is to read an id3v22 frame header and convert it to a id3v23 one !
+--------------------------------------------------------------------------*/
int CFileInit_mp3::read_id3v22_frame_header( str_id3v23_frame_header &id3v23,
                                             int &i_size_tag
                                           )
{
   /*----------------------------------------------------------------------*/
   str_id3v22_frame_header id3v22 ;

   /*----------------------------------------------------------------------*/
   i_size_tag -= sizeof( id3v22 ) ;

   /*--( Read and transform )----------------------------------------------*/
   if( m_fa.read_buffer( sizeof( id3v22 ), &id3v22 ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   SR_SET_0( id3v23 ) ;

   /*--( Convert the v22 tag to v23 )--------------------------------------*/
   switch( SR_THREECC( id3v22.tb_b_id[ 0 ],
                       id3v22.tb_b_id[ 1 ],
                       id3v22.tb_b_id[ 2 ]
                     )
         )
   {
      /*-------------------------------------------------------------------*/
      case SR_THREECC('T','T','2') :
         id3v23.dw_id = SR_FOURCC( 'T','I','T','2' ) ; break ;
      case SR_THREECC('T','P','1') :
         id3v23.dw_id = SR_FOURCC( 'T','P','E','1' ) ; break ;
      case SR_THREECC('T','A','L') :
         id3v23.dw_id = SR_FOURCC( 'T','A','L','B' ) ; break ;
      case SR_THREECC('T','Y','E') :
         id3v23.dw_id = SR_FOURCC( 'T','Y','E','R' ) ; break ;
      case SR_THREECC('C','O','M') :
         id3v23.dw_id = SR_FOURCC( 'C','O','M','M' ) ; break ;
      case SR_THREECC('T','R','K') :
         id3v23.dw_id = SR_FOURCC( 'T','R','C','K' ) ; break ;
      case SR_THREECC('T','P','A') :
         id3v23.dw_id = SR_FOURCC( 'T','P','O','S' ) ; break ;
      case SR_THREECC('T','C','O') :
         id3v23.dw_id = SR_FOURCC( 'T','C','O','N' ) ; break ;
      case SR_THREECC('T','C','M') :
         id3v23.dw_id = SR_FOURCC( 'T','C','O','M' ) ; break ;
      case SR_THREECC('T','O','A') :
         id3v23.dw_id = SR_FOURCC( 'T','O','P','E' ) ; break ;
      case SR_THREECC('T','C','R') :
         id3v23.dw_id = SR_FOURCC( 'T','C','O','P' ) ; break ;
      case SR_THREECC('W','X','X') :
         id3v23.dw_id = SR_FOURCC( 'W','X','X','X' ) ; break ;
      case SR_THREECC('T','E','N') :
         id3v23.dw_id = SR_FOURCC( 'T','E','N','C' ) ; break ;
      /*-------------------------------------------------------------------*/
      case SR_THREECC('P','I','C') :
         id3v23.dw_id = SR_FOURCC( 'A','P','I','C' ) ; break ;
      /*--------------------------------------------------------------------+
      ! If the ID is 0, the calling function will stop ... ;-)              !
      +--------------------------------------------------------------------*/
      default: id3v23.dw_id = SR_FOURCC( 'R','P','L','R' ) ; break ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Convert the size of the associated value too )--------------------*/
   id3v23.dw_size = SR_FOURCC( 0,
                               id3v22.tb_b_size[ 0 ],
                               id3v22.tb_b_size[ 1 ],
                               id3v22.tb_b_size[ 2 ]
                             ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::read_frame_header( str_id3v23_frame_header &id3v23,
                                      int &i_size_tag
                                    )
{
   /*----------------------------------------------------------------------*/
   if( m_i_id3v2_version == 2 )
   {  /*-------------------------------------------------------------------*/
      if( read_id3v22_frame_header( id3v23, i_size_tag ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      return( 0 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   i_size_tag -= sizeof( id3v23 ) ;
   if( m_fa.read_buffer( sizeof( id3v23 ), &id3v23 ) != 0 )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::image_info( size_t sz_size )
{
   /*----------------------------------------------------------------------*/
   char         tb_c_val[ 512 ]     ;
   e_enc_type   enc_type            ;
   wxFileOffset fo_offset_deb       ;
   size_t       sz_size_to_skip = 0 ;
   wxString     s_mime              ;
   size_t       sz_num              ;

   /*--( Minimum ... )-----------------------------------------------------*/
   if( sz_size <= 4 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   fo_offset_deb = m_fa.get_offset() ;

   /*--( Limit the amount of data )----------------------------------------*/
   if( sz_size > sizeof( tb_c_val ) )
   {  sz_size_to_skip = sz_size - sizeof( tb_c_val ) ;
      sz_size = sizeof( tb_c_val ) ;
   }
   /*----------------------------------------------------------------------*/
   if(    m_fa.read_buffer( sz_size, tb_c_val ) != 0
       || (    sz_size_to_skip > 0
            && m_fa.skip_nb_byte( sz_size_to_skip ) != 0
          )
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   sz_num = 0 ;

   /*--( The first byte is the text encoding )-----------------------------*/
   enc_type = ( e_enc_type )tb_c_val[ sz_num++ ] ;

   /*-----------------------------------------------------------------------+
   ! The mime type is in ascii                                              !
   ! In ID3V22 it is only 3 characters without terminator                   !
   +-----------------------------------------------------------------------*/
   if( m_i_id3v2_version == 2 )
   {  m_fi.prepare_string( &tb_c_val[ 1 ], 3, s_mime ) ;
      sz_num += 3 ;
   }
   else
   {  m_fi.prepare_string( &tb_c_val[ 1 ], sz_size, s_mime ) ;
      sz_num += s_mime.size() + 1 ;
   }
   /*----------------------------------------------------------------------*/
   if( !s_mime.empty() )
   {  m_f.val_s( COL_AUDTAG_IMG_FORMAT ) = s_mime ; }

   /*----------------------------------------------------------------------*/
   if( sz_num >= sz_size ) { return( -3 ) ; }

   /*--( Skip over the "Picture Type" too )--------------------------------*/
   ++sz_num ;

   /*--( The picture description in "Text encoding" format )---------------*/
   if( enc_type == ET_U16_BOM || enc_type == ET_U16_NOBOM )
   {
      /*-------------------------------------------------------------------*/
      while( sz_num < sz_size && sr::pb_to_ui16( &tb_c_val[ sz_num ] ) != 0 )
      {  sz_num += 2 ; }
      /*-------------------------------------------------------------------*/
      sz_num += 2 ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*-------------------------------------------------------------------*/
      while( sz_num < sz_size && tb_c_val[ sz_num ] != '\0' )
      {  ++sz_num ; }
      /*-------------------------------------------------------------------*/
      ++sz_num ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( sz_num >= sz_size ) { return( -4 ) ; }

   /*--( Try to extract the image info )-----------------------------------*/
   if( m_fi.read_embedded_image_info( fo_offset_deb + sz_num,
                                      sz_size + sz_size_to_skip - sz_num,
                                      s_mime
                                    ) != 0
     )
   {  return( -5 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::extr_frame_info( wxUint32 dw_val,
                                    str_mp3_frame_info &s_fi
                                  )
{
   /*----------------------------------------------------------------------*/
   dw_val = wxUINT32_SWAP_ON_LE( dw_val ) ;
   /*--( There must be the synchro )---------------------------------------*/
   if( ( ( dw_val >> 21 ) & 0x7FF ) != 0x7FF )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   int i_layer_desc     ;
   int i_bitrate_index  ;
   int i_samprate_index ;
   int i_mul_fact       ;
   /*-----------------------------------------------------------------------+
   ! Frame Header Structure:                                                !
   !                                                                        !
   ! AAAAAAAA AAABBCCD EEEEFFGH IIJJKLMM                                    !
   !                                                                        !
   ! A : Synchro                                                            !
   ! B : MPEG version                                                       !
   ! C : Layer                                                              !
   ! D : Protection                                                         !
   ! E : Bitrate                                                            !
   ! F : Sampling                                                           !
   ! G : Padding                                                            !
   ! H : Bit "private"                                                      !
   ! I : Channel mode                                                       !
   ! J : Mode extension                                                     !
   ! K : Copyright                                                          !
   ! L : Original                                                           !
   ! M : Emphasis                                                           !
   !                                                                        !
   ! mpeg_version :                                                         !
   ! 00 - MPEG Version 2.5                                                  !
   ! 01 - reserved                                                          !
   ! 10 - MPEG Version 2                                                    !
   ! 11 - MPEG Version 1                                                    !
   ! layer_desc                                                             !
   ! 00 - reserved                                                          !
   ! 01 - Layer III                                                         !
   ! 10 - Layer II                                                          !
   ! 11 - Layer I                                                           !
   +-----------------------------------------------------------------------*/
   s_fi.i_mpeg_version = ( dw_val >> 19 ) & 0x03 ;
   i_layer_desc        = ( dw_val >> 17 ) & 0x03 ;
   i_bitrate_index     = ( dw_val >> 12 ) & 0x0F ;
   i_samprate_index    = ( dw_val >> 10 ) & 0x03 ;
   s_fi.i_padding      = ( dw_val >>  9 ) & 0x01 ;
   s_fi.i_channel      = ( dw_val >>  6 ) & 0x03 ;
   /*----------------------------------------------------------------------*/
   switch( i_layer_desc )
   {  /*-------------------------------------------------------------------*/
      case 1 : s_fi.i_layer_ind =  2 ; break ; // Layer 3
      case 2 : s_fi.i_layer_ind =  1 ; break ; // Layer 2
      case 3 : s_fi.i_layer_ind =  0 ; break ; // Layer 1
      default : return( -2 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   switch( s_fi.i_mpeg_version )
   {
      /*--( MPEG 1 )-------------------------------------------------------*/
      case 3 :
         /*----------------------------------------------------------------*/
         s_fi.mpeg_type = str_mp3_frame_info::MPEG_TYPE_1 ;
         s_fi.i_samprate = st_tb_tb_i_samprate[ i_samprate_index ][ 0 ]  ;
         /*----------------------------------------------------------------*/
         break ;
      /*--( MPEG 2 )-------------------------------------------------------*/
      case 2 :
         /*----------------------------------------------------------------*/
         s_fi.mpeg_type = str_mp3_frame_info::MPEG_TYPE_2 ;
         s_fi.i_samprate = st_tb_tb_i_samprate[ i_samprate_index ][ 1 ]  ;
         /*----------------------------------------------------------------*/
         break ;
      /*--( MPEG 2.5 )-----------------------------------------------------*/
      case 0 :
         /*----------------------------------------------------------------*/
         s_fi.mpeg_type = str_mp3_frame_info::MPEG_TYPE_2 ;
         s_fi.i_samprate = st_tb_tb_i_samprate[ i_samprate_index ][ 2 ]  ;
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      default :
         s_fi.mpeg_type = str_mp3_frame_info::MPEG_TYPE_NB ;
         s_fi.i_bitrate = s_fi.i_samprate = i_mul_fact = -1 ;
         return( -3 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Minimal check )---------------------------------------------------*/
   if(    s_fi.mpeg_type < 0
       || s_fi.mpeg_type >= str_mp3_frame_info::MPEG_TYPE_NB
       || s_fi.i_samprate <= 0
     )
   {  return( -4 ) ; }

   /*----------------------------------------------------------------------*/
     s_fi.i_bitrate
   = st_tb_tb_i_bitrate[s_fi.mpeg_type][i_bitrate_index][ s_fi.i_layer_ind ];
   i_mul_fact = st_tb_i_mul_fact[ s_fi.mpeg_type ][ s_fi.i_layer_ind ] ;

   /*--( Minimal check )---------------------------------------------------*/
   if( s_fi.i_bitrate <= 0 )
   {  return( -5 ) ; }

   /*--( Frame length computation for an eventual jump )-------------------*/
     s_fi.i_frame_len
   = ( i_mul_fact * s_fi.i_bitrate ) / s_fi.i_samprate + s_fi.i_padding ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::search_next_frame_header( str_mp3_frame_info &s_fi,
                                             int i_max_read_len
                                           )
{
   /*----------------------------------------------------------------------*/
   wxUint8  tb_b_buffer[ 1024 ]      ;
   size_t   sz_read                  ;
   bool     boo_first_buffer = true  ;
   wxUint32 dw_val           = 0     ;
   bool     boo_found        = false ;
   int      i_num                    ;

   /*--( Reposition one byte forward to "forget" the current frame )-------*/
   if( m_fa.skip_nb_byte( +1 ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   do
   {  /*-------------------------------------------------------------------*/
      sz_read = wxMin( i_max_read_len, ( int )sizeof( tb_b_buffer ) ) ;
      if( m_fa.read_buffer_max( sz_read, tb_b_buffer ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      i_max_read_len -= sz_read ;
      /*-------------------------------------------------------------------*/
      if( !boo_first_buffer )
      {  i_num = 0 ; }
      else
      {  /*----------------------------------------------------------------*/
         boo_first_buffer = false  ;
         i_num = sizeof( dw_val ) ;
         if( sz_read >= ( int )sizeof( dw_val ) )
         {  dw_val = sr::pb_to_ui32( tb_b_buffer ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      for( ;    i_num < ( int )sz_read
             && extr_frame_info( dw_val, s_fi ) != 0 ;
           ++i_num
         )
      {  /*----------------------------------------------------------------*/
         dw_val >>= 8 ;
         dw_val |= tb_b_buffer[ i_num ] << 24 ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      boo_found = ( i_num < ( int )sz_read ) ;
      /*-------------------------------------------------------------------*/
   } while(    !boo_found && sz_read == sizeof( tb_b_buffer )
            && i_max_read_len > 0
          ) ;

   /*----------------------------------------------------------------------*/
   if( !boo_found ) { return( -3 ) ; }

   /*-----------------------------------------------------------------------+
   ! Found it ! Set position just before the start of the sequence          !
   +-----------------------------------------------------------------------*/
   if( m_fa.skip_nb_byte( - ( int )sz_read + i_num - 4 ) != 0 )
   {  return( -4 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::init_id3v1( wxFileOffset &fo_offset_data_end )
{
   /*----------------------------------------------------------------------*/
   str_id3v1 id3v1 ;
   wxString  s_val ;

   /*----------------------------------------------------------------------*/
   fo_offset_data_end = m_f.get_size().ToULong() ;

   /*----------------------------------------------------------------------*/
   if(    m_fa.set_offset_from_end( sizeof( id3v1 ) ) != 0
       || m_fa.read_buffer( sizeof( id3v1 ), &id3v1 ) != 0
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   if(    id3v1.tb_c_ident[ 0 ] != 'T'
       || id3v1.tb_c_ident[ 1 ] != 'A'
       || id3v1.tb_c_ident[ 2 ] != 'G'
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   m_fi.set_conv_from_8bit() ;
   /*----------------------------------------------------------------------*/
   fo_offset_data_end -= sizeof( id3v1 ) ;
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( id3v1.tb_c_title, sizeof( id3v1.tb_c_title ),
                            s_val
                          ) > 0
     )
   {  m_f.val_s( COL_ID3V1_TITLE ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( id3v1.tb_c_artist, sizeof( id3v1.tb_c_artist ),
                            s_val
                          ) > 0
     )
   {  m_f.val_s( COL_ID3V1_ARTIST ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( id3v1.tb_c_album, sizeof( id3v1.tb_c_album ),
                            s_val
                          ) > 0
     )
   {  m_f.val_s( COL_ID3V1_ALBUM ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( id3v1.tb_c_year, sizeof( id3v1.tb_c_year ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_ID3V1_YEAR ) = s_val ; }

   /*--( The comment can contain the track number )------------------------*/
   if( m_fi.prepare_string( id3v1.tb_c_comment,
                              sizeof( id3v1.tb_c_comment )
                            + sizeof( id3v1.b_track),
                            s_val
                          ) > 0
     )
   {  m_f.val_s( COL_ID3V1_COMMENT ) = s_val ; }

   /*----------------------------------------------------------------------*/
   if( id3v1.b_track != '\0' && id3v1.tb_c_comment[ 28 ] == '\0' )
   {  m_f.val_ll( COL_ID3V1_TRACK_NUM ) = id3v1.b_track ; }

   /*----------------------------------------------------------------------*/
   if( m_fi.mp3_genre_id_ok( id3v1.b_genre ) )
   {  m_f.val_s( COL_ID3V1_GENRE ) = m_fi.mp3_get_genre( id3v1.b_genre ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( On return the offset is set after the header )-----------------------*/
int CFileInit_mp3::init_id3v2()
{
   /*----------------------------------------------------------------------*/
   str_id3v2_header        id3v2_header             ;
   str_id3v23_frame_header id3v23_fh                ;
   int                     i_len_header         = 0 ;
   int                     i_len_ext_header     = 0 ;
   int                     i_len_tag                ;
   int                     i_len_val                ;
   int                     i_col                    ;
   wxFileOffset            fo_offset_header_end = 0 ;
   int                     i_ret                = 0 ;
   wxString                s_val                    ;

   /*----------------------------------------------------------------------*/
   if(    m_fa.set_offset( 0 ) != 0
       || m_fa.read_buffer( sizeof( id3v2_header ), &id3v2_header ) != 0
     )
   {  return( -1 ) ; }

   /*--( Minimal check )---------------------------------------------------*/
   if(    id3v2_header.tb_b_ident  [ 0 ] != 'I'
       || id3v2_header.tb_b_ident  [ 1 ] != 'D'
       || id3v2_header.tb_b_ident  [ 2 ] != '3'
       || id3v2_header.tb_b_version[ 0 ] == 0xFF
       || id3v2_header.tb_b_version[ 1 ] == 0xFF
       || (   i_len_header
            = st_conv_id3v2_synch_size( id3v2_header.dw_size )
          ) <= 0
     )
   {  i_ret = -2 ; goto func_end ; }

   /*--( Now the end is known )--------------------------------------------*/
   fo_offset_header_end = sizeof( id3v2_header ) + i_len_header ;
   /*----------------------------------------------------------------------*/
   m_i_id3v2_version = id3v2_header.tb_b_version[ 0 ] ;

   /*--( Extended Header present ? )---------------------------------------*/
   if( ( id3v2_header.b_flags & 0x40 ) != 0 )
   {
      /*-------------------------------------------------------------------*/
      wxUint32 dw_id3v2_ext_header ;

      /*--( Read the size )------------------------------------------------*/
      if( m_fa.read_be_data( dw_id3v2_ext_header ) != 0 )
      {  i_ret = -3 ; goto func_end ; }

      /*--( Jump over the extended header )--------------------------------*/
      i_len_ext_header = dw_id3v2_ext_header ;
      if(    i_len_ext_header <= 0
          || i_len_ext_header >= i_len_header
          || m_fa.skip_nb_byte( i_len_ext_header ) != 0
        )
      {  i_ret = -4 ; goto func_end ; }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Treat all tag frames.                                                  !
   ! The size of the frame header depends on the id3v2 version. The correct !
   ! size is subtracted from the tag size during the reading.               !
   +-----------------------------------------------------------------------*/
   for( i_len_tag = i_len_header - i_len_ext_header ;
           i_len_tag > 0
        && read_frame_header( id3v23_fh, i_len_tag ) == 0
        && *( wxUint8 * )&id3v23_fh.dw_id != 0 ;
        i_len_tag -= i_len_val
      )
   {
      /*--( Starting version 2.4, the frame sizes are stored synchronized )*/
      i_len_val = ( m_i_id3v2_version >= 4
                    ? st_conv_id3v2_synch_size( id3v23_fh.dw_size )
                    : wxUINT32_SWAP_ON_LE( id3v23_fh.dw_size )
                  ) ;
      /*--------------------------------------------------------------------+
      ! Let go through if the size equal to 0 ... it happens !              !
      ! No "continue" because the functions treat this case.                !
      +--------------------------------------------------------------------*/
      if( i_len_val < 0 ) { i_ret = -5 ; goto func_end ; }

      /*--------------------------------------------------------------------+
      ! The frames with the following characteristics are skipped:          !
      !    - compressed                                                     !
      !    - crypted                                                        !
      !    - unsynchro                                                      !
      !    - data length indicator                                          !
      +--------------------------------------------------------------------*/
      if( ( id3v23_fh.tb_b_flags[ 1 ] & 0x0F ) != 0 )
      {  m_fa.skip_nb_byte( i_len_val ) ; continue ; }

      /*--( TAG APIC ? .... miam miam :-) )--------------------------------*/
      if(    id3v23_fh.dw_id == SR_FOURCC( 'A','P','I','C' )
          && m_f.get_image_offset() == -1
        )
      {  image_info( i_len_val ) ; continue ; }

      /*--( "compute" the siren column equivalent )------------------------*/
      i_col = st_id3v2_conv_tag_col( id3v23_fh.dw_id ) ;

      /*--------------------------------------------------------------------+
      ! Problem ... what to do with repeated tags ?                         !
      ! "the first to talk is right" has been chosen.                       !
      ! This seems right except for the "comment".                          !
      ! This is the case for the "MusicMatch_..." descriptions.             !
      ! So for the comment and only for it (for now) the last non empty is  !
      ! taken into account.                                                 !
      +--------------------------------------------------------------------*/
      if(    i_col == COL_NB
          || (    id3v23_fh.dw_id != SR_FOURCC( 'C','O','M','M' )
               && !m_fi.reserve_col( i_col )
             )
        )
      {  m_fa.skip_nb_byte( i_len_val ) ; continue ; }

      /*-------------------------------------------------------------------*/
      switch( id3v23_fh.dw_id )
      {
         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'T','I','T','2' ) :
         case SR_FOURCC( 'T','P','E','1' ) :
         case SR_FOURCC( 'T','A','L','B' ) :
         case SR_FOURCC( 'T','Y','E','R' ) :
         case SR_FOURCC( 'T','C','O','M' ) :
         case SR_FOURCC( 'T','O','P','E' ) :
         case SR_FOURCC( 'T','C','O','P' ) :
         case SR_FOURCC( 'T','E','N','C' ) :
         case SR_FOURCC( 'T','P','O','S' ) :
            /*-------------------------------------------------------------*/
            string_id3v2( i_len_val, s_val ) ;
            if( !s_val.empty() ) { m_f.val_s( i_col ) = s_val ; }
            /*-------------------------------------------------------------*/
            break ;

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'C','O','M','M' ) :
            /*--------------------------------------------------------------+
            ! The comment can contain a lot of information:                 !
            ! The language (3 characters), then two strings in which the    !
            ! first one is the description of the second.                   !
            ! "XXXCDDB Disc ID", "MusicMatch_Preference" ... etc ...        !
            +--------------------------------------------------------------*/
            string_id3v2( i_len_val, s_val, STRID3V2_LANG | STRID3V2_DESC ) ;
            if( !s_val.empty() ) { m_f.val_s( i_col ) = s_val ; }
            /*-------------------------------------------------------------*/
            break ;

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'T','R','C','K' ) :
         {  /*------------------------------------------------------------*/
            string_id3v2( i_len_val, s_val ) ;
            if( !s_val.empty() )
            {  /*----------------------------------------------------------*/
               m_f.val_s( i_col ) = s_val ;
               m_fi.init_audio_num_and_nb_track( s_val ) ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'T','C','O','N' ) :
         {
            /*-------------------------------------------------------------*/
            int i_end = 0 ; // Initialization to avoid compilation warning
            /*-------------------------------------------------------------*/
            string_id3v2( i_len_val, s_val ) ;
            s_val.Trim() ;
            /*--( The genre code can be in front surrounded with '()' )----*/
            if(    s_val.size() > 0
                && s_val[ 0 ] == '('
                && ( i_end = s_val.find( ')' ) ) != ( int )wxString::npos
              )
            {  /*--( If some text follows, it is used )--------------------*/
               if( i_end < ( int )s_val.size() - 1 )
               {  s_val = s_val.Mid( i_end + 1 ) ; }
               else /*--( Else the text associated to the code is used )---*/
               {
                  /*-------------------------------------------------------*/
                  int i_genre = 0 ;
                  sr::wxString_it it = s_val.begin() + 1 ;
                  /*-------------------------------------------------------*/
                  sr::conv_str_to_int( it, s_val.end(), i_genre ) ;
                  if( m_fi.mp3_genre_id_ok( i_genre ) )
                  {  s_val = m_fi.mp3_get_genre( i_genre ) ; }
                  else
                  {  s_val.clear() ; }
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            if( !s_val.empty() ) { m_f.val_s( i_col ) = s_val ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'W','X','X','X' ) :
            /*-------------------------------------------------------------*/
            string_id3v2( i_len_val, s_val, STRID3V2_DESC ) ;
            if( !s_val.empty() ) { m_f.val_s( i_col ) = s_val ; }
            /*-------------------------------------------------------------*/
            break ;

         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
func_end :
   /*--( Set pointer after header or at the beginning in case of error )---*/
   if( m_fa.set_offset( fo_offset_header_end ) != 0 ) { return( -6 ) ; }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mp3::init_general_info( size_t sz_size )
{
   /*----------------------------------------------------------------------*/
   wxUint32           dw_frame_header        ;
   str_mp3_frame_info s_fi                   ;
   int                i_duration             ;
   e_bitrate_type     bitrate_type = TBR_CBR ;
   wxUint8            tb_b_buffer[ 512 ]     ;
   wxUint8            *p_b_buffer            ;
   int                i_pos_start            ;

   /*--( Read the first frame header and rewind ... )----------------------*/
   if(    m_fa.read_data( dw_frame_header ) != 0
       || m_fa.skip_nb_byte( - ( int )sizeof( dw_frame_header ) ) != 0
     )
   {  return( -1 ) ; }

   /*--( A header should be present )--------------------------------------*/
   if( extr_frame_info( dw_frame_header, s_fi ) != 0 )
   {
      /*--------------------------------------------------------------------+
      ! Many mp3 files can be "malformed". Old Siren versions did this      !
      ! lookup for the right frame.                                         !
      ! Starting with 3.14 and option enables to choose.                    !
      +--------------------------------------------------------------------*/
      if(    !wxGetApp().M_boo_loose_mp3_check.get()
          || search_next_frame_header( s_fi, 50 * 1024 ) != 0
        )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! The first frame will give the bitrate (variable / fixed)               !
   +-----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( tb_b_buffer ), tb_b_buffer ) != 0 )
   {  return( -3 ) ; }

   /*-----------------------------------------------------------------------+
   ! Xing means Variable Bit Rate                                           !
   +-----------------------------------------------------------------------*/
   if( s_fi.i_mpeg_version == 3 )
   {  i_pos_start = ( s_fi.i_channel != 3 ? 36 : 21 ) ; }
   else
   {  i_pos_start = ( s_fi.i_channel != 3 ? 21 : 13 ) ; }
   /*----------------------------------------------------------------------*/
   p_b_buffer = tb_b_buffer + i_pos_start ;
   /*----------------------------------------------------------------------*/
   if( SR_FOURCC( p_b_buffer[ 0 ], p_b_buffer[ 1 ],
                  p_b_buffer[ 2 ], p_b_buffer[ 3 ]
                ) == SR_FOURCC( 'X','i','n','g' )
     )
   {
      /*-------------------------------------------------------------------*/
      p_b_buffer += 4 ;
      /*-------------------------------------------------------------------*/
      int      i_nb_frame  = -1      ;
      wxUint32 dw_head_flag          ;
      int      i_nb_sample_per_frame ;
      size_t   sz_size_vbr = sz_size ;
      /*-------------------------------------------------------------------*/
        i_nb_sample_per_frame
      = st_tb_i_nb_sample_per_frame[ s_fi.mpeg_type ][ s_fi.i_layer_ind ] ;
      /*-------------------------------------------------------------------*/
      dw_head_flag = wxUINT32_SWAP_ON_LE( *( wxUint32 * )p_b_buffer ) ;
      p_b_buffer += 4 ;
      /*-------------------------------------------------------------------*/
      #define FRAMES_FLAG     0x0001
      #define BYTES_FLAG      0x0002
      #define TOC_FLAG        0x0004
      #define VBR_SCALE_FLAG  0x0008
      /*-------------------------------------------------------------------*/
      if( ( dw_head_flag & FRAMES_FLAG ) != 0 )
      {  i_nb_frame = wxUINT32_SWAP_ON_LE( *( wxUint32 * )p_b_buffer ) ;
         p_b_buffer += 4 ;
      }
      /*--( Number of bytes )----------------------------------------------*/
      if( ( dw_head_flag & BYTES_FLAG ) != 0 )
      {
         /*----------------------------------------------------------------*/
         size_t sz_size_temp
              = wxUINT32_SWAP_ON_LE( *( wxUint32 * )p_b_buffer ) ;
         /*----------------------------------------------------------------*/
         p_b_buffer += 4 ;
         if( sz_size_temp > 0 && sz_size_temp < sz_size )
         {  sz_size_vbr = sz_size_temp ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( ( dw_head_flag & TOC_FLAG       ) != 0 ) { p_b_buffer += 100 ; }
      if( ( dw_head_flag & VBR_SCALE_FLAG ) != 0 ) { p_b_buffer +=   4 ; }

      /*--( Recomputation of the duration and the approximated bitrate )---*/
      if( i_nb_frame > 0 )
      {
         /*--( Don't round, truncate. "i_samprate" can't be 0 )------------*/
         i_duration = i_nb_frame * i_nb_sample_per_frame / s_fi.i_samprate ;
         if( i_duration <= 0 ) { return( -4 ) ; }
         /*----------------------------------------------------------------*/
         s_fi.i_bitrate = sz_size_vbr * 8 / 1000 / i_duration ;
         /*----------------------------------------------------------------*/
      }
      else /*--( Else it is the standard case )----------------------------*/
      {  i_duration = sz_size_vbr * 8 / s_fi.i_bitrate / 1000 ; }
      /*-------------------------------------------------------------------*/
      bitrate_type = TBR_VBR_XING ;
      m_f.val_s( COL_AUDIO_INFO ) = "VBR Xing" ;
      /*-------------------------------------------------------------------*/
      #undef FRAMES_FLAG
      #undef BYTES_FLAG
      #undef TOC_FLAG
      #undef VBR_SCALE_FLAG
      /*-------------------------------------------------------------------*/
   }
   else
      /*--------------------------------------------------------------------+
      ! Detection of VBR "Fraunhofer" : VBRI                                !
      ! Not detected by Winamp2 & 3                                         !
      ! Only WindowsMediaPlayer displays the right duration                 !
      +--------------------------------------------------------------------*/
   if( SR_FOURCC( tb_b_buffer[ 36 ], tb_b_buffer[ 37 ],
                  tb_b_buffer[ 38 ], tb_b_buffer[ 39 ]
                ) == SR_FOURCC( 'V','B','R','I')

     )
   {  /*--------------------------------------------------------------------+
      ! What follows has the format:                                        !
      ! WORD  w_VbriVersion                                                 !
      ! WORD  w_VbriDelay                                                   !
      ! WORD  w_VbriQuality                                                 !
      ! DWORD dw_VbriStreamBytes                                            !
      ! DWORD dw_VbriStreamFrames                                           !
      ! WORD  w_VbriTableSize                                               !
      ! WORD  w_VbriTableScale                                              !
      ! WORD  w_VbriEntryBytes                                              !
      ! WORD  w_VbriEntryFrames                                             !
      +--------------------------------------------------------------------*/
      size_t   sz_size_vbr ;
      wxUint32 *p_dw_buffer = ( wxUint32 * )&tb_b_buffer[ 46 ] ;
      size_t   sz_size_temp = wxUINT32_SWAP_ON_LE( p_dw_buffer[ 0 ] ) ;
      wxUint32 dw_VbriStreamFrames = wxUINT32_SWAP_ON_LE( p_dw_buffer[ 1 ] );
      int      i_nb_sample_per_frame
             = ( s_fi.i_samprate >= 32000 ? 1152 : 576 ) ;
      /*-------------------------------------------------------------------*/
      if( sz_size_temp != 0 && sz_size_temp < sz_size )
      {  sz_size_vbr = sz_size_temp ; }
      else
      {  sz_size_vbr = sz_size ; }
      /*--( Don't round, truncate. "i_samprate" can't be null )------------*/
        i_duration
      = dw_VbriStreamFrames * i_nb_sample_per_frame / s_fi.i_samprate ;
      if( i_duration <= 0 ) { return( -5 ) ; }
      /*-------------------------------------------------------------------*/
      s_fi.i_bitrate = sz_size_vbr * 8 / 1000 / i_duration ;
      /*-------------------------------------------------------------------*/
      bitrate_type  = TBR_VBR_FH ;
      m_f.val_s( COL_AUDIO_INFO ) = "VBR Fraunhofer" ;
      /*-------------------------------------------------------------------*/
   }
   else
   {
      /*--------------------------------------------------------------------+
      ! So ... It's Constant Bit Rate, but the frame can be incoherent with !
      ! the rest of the file. That's why, based on the size a search for    !
      ! the right one is done. (cf WharFor)                                 !
      +--------------------------------------------------------------------*/
      bool               boo_frame_ok = false ;
      str_mp3_frame_info s_fi_next_frame      ;

      /*--( The buffer was useless so "rewind" )---------------------------*/
      if( m_fa.skip_nb_byte( - ( int )sizeof( tb_b_buffer ) ) != 0 )
      {  return( -6 ) ; }
      /*-------------------------------------------------------------------*/
      do
      {  /*-----------------------------------------------------------------+
         ! The frame is ok if after jumping it a new one is present         !
         +-----------------------------------------------------------------*/
         if(    m_fa.skip_nb_byte( s_fi.i_frame_len ) != 0
             || m_fa.read_data( dw_frame_header ) != 0
           )
         {  return( -7 ) ; }
         /*----------------------------------------------------------------*/
         if( extr_frame_info( dw_frame_header, s_fi_next_frame ) == 0 )
         {  boo_frame_ok = true ; }
         else
         if( search_next_frame_header( s_fi, 3 * 1024 ) != 0  )
         {  return( -8 ) ; }
         /*----------------------------------------------------------------*/
      } while( !boo_frame_ok ) ;

      /*--( In the standard case the frames have the same length )---------*/
      i_duration = sz_size * 8 / s_fi.i_bitrate / 1000 ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_fi.init_audio_bitrate( bitrate_type, s_fi.i_bitrate ) ;
   /*----------------------------------------------------------------------*/
   if( i_duration > 0 ) { m_fi.init_audio_duration( i_duration ) ; }
   /*----------------------------------------------------------------------*/
   if( s_fi.i_samprate > 0 ) { m_fi.init_audio_samprate( s_fi.i_samprate ) ;}
   /*----------------------------------------------------------------------*/
   m_fi.init_col_mp3_channel( COL_AUDIO_CHANNEL, s_fi.i_channel ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_mp3()
{
   /*----------------------------------------------------------------------*/
   CFileInit_mp3 mp3( *this )  ;
   wxFileOffset  fo_offset_end ;

   /*----------------------------------------------------------------------*/
   m_s_type_det = "mp3" ;
   /*----------------------------------------------------------------------*/
   mp3.init_id3v1( fo_offset_end ) ;
   mp3.init_id3v2() ;

   /*-----------------------------------------------------------------------+
   ! Init size for computation                                              !
   ! ATTENTION !!!                                                          !
   ! Some test results:                                                     !
   ! 1) Tag&Rename computes the time the same way as Winamp3                !
   !    (tagv1 present ou not)                                              !
   ! 2) Winamp2 takes the V1 tag into account (subtract its size) while     !
   !    Winamp3 (and so T&R) don't do so                                    !
   ! 3) Winamp2, 3 and T&R take as data start what follows the eventual V2  !
   !    TAG, so if the header is rotten it is counted                       !
   +-----------------------------------------------------------------------*/
   int i_ret = mp3.init_general_info( fo_offset_end - m_fa.get_offset() ) ;
   /*----------------------------------------------------------------------*/
   m_f.init_info_s_track() ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool CFileInit::mp3_genre_id_ok( int i_genre_id ) const
{
   /*----------------------------------------------------------------------*/
   return(    i_genre_id >= 0
           && i_genre_id < ( int )WXSIZEOF( st_tb_c_id3_genre )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Genre is supposed to be ok ... )-------------------------------------*/
wxString CFileInit::mp3_get_genre( int i_genre_id ) const
{
   /*----------------------------------------------------------------------*/
   return( wxString::From8BitData( st_tb_c_id3_genre[ i_genre_id ] ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
!                                                                           !
! Those functions are directly used by other init modules ...               !
!                                                                           !
+--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
int CFileInit::mp3_extr_frame_info( wxUint32 dw_val,
                                    str_mp3_frame_info &s_fi
                                  )
{  return( CFileInit_mp3( *this ).extr_frame_info( dw_val, s_fi ) ) ; }

/*-------------------------------------------------------------------------*/
int CFileInit::mp3_search_next_frame_header( str_mp3_frame_info &s_fi,
                                             int i_max_read_len
                                           )
{  return( CFileInit_mp3( *this ).search_next_frame_header( s_fi,
                                                            i_max_read_len
                                                          )
         ) ;
}

/*-------------------------------------------------------------------------*/
wxUint32 CFileInit::mp3_conv_id3v2_synch_size( wxUint32 dw_size )
{  return( st_conv_id3v2_synch_size( dw_size ) ) ; }

/*-------------------------------------------------------------------------*/
int CFileInit::mp3_read_id3v1( wxFileOffset &fo_offset_data_end )
{  return( CFileInit_mp3( *this ).init_id3v1( fo_offset_data_end ) ) ; }

/*-------------------------------------------------------------------------*/
int CFileInit::mp3_read_id3v2()
{  return( CFileInit_mp3( *this ).init_id3v2() ) ; }

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                     End of file CFileInit_mp3.cpp                         !
+==========================================================================*/
