/*====================================+=====================================+
! File CFileInit_mkv.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                       Matroska File Format                                !
!                                                                           !
! Audio file: ".mka"                                                        !
! Video file: ".mkv", ".webm" (looks like those files are ok too ...)       !
!                                                                           !
! http://matroska.org/technical/index.html                                  !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Only the information of the first segment is extracted.                   !
! Except for the codec, all information is extracted from the first track   !
! (audio or video).                                                         !
!                                                                           !
! The global first level tags are extracted.                                !
!                                                                           !
! If an attachment is present with a mime type "image/..." it is "loaded".  !
!                                                                           !
! META SEEK:                                                                !
! - This section can contain or not (vsshort-aac.mkv) the minimum           !
!   acceptable like "segment info" and "tracks".                            !
!   Method used: stop after a "meta seek" if the "segment info" and         !
!   "tracks" have been read even if other like "attachment" or "tags" have  !
!   not been read yet.                                                      !
!   Maybe this will have to be rethought ...                                !
! - Don't take into account the nested ones (one referencing the other).    !
!   Based on the examples, the first meta seek contains all the information !
!   needed. Generally, if there is another one it references only the       !
!   clusters. And there can be a huge number of them ...                    !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#define DEF_ST_CO( MKV_V, MKV_X1, MKV_X2, MKV_X3, MKV_X4 ) \
   static const wxUint32 MKV_V \
        = wxUINT32_SWAP_ON_LE( SR_FOURCC( MKV_X1, MKV_X2, MKV_X3, MKV_X4 ) )
/*-------------------------------------------------------------------------*/
DEF_ST_CO( st_co_dw_mkv_header       , 0x1A,0x45,0xDF,0xA3 ) ;
DEF_ST_CO( st_co_dw_mkv_segment      , 0x18,0x53,0x80,0x67 ) ;
DEF_ST_CO( st_co_dw_mkv_seek_head    , 0x11,0x4D,0x9B,0x74 ) ;
DEF_ST_CO( st_co_dw_mkv_seek         , 0x4D,0xBB,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_seek_id      , 0x53,0xAB,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_seek_pos     , 0x53,0xAC,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_segment_info , 0x15,0x49,0xA9,0x66 ) ;
DEF_ST_CO( st_co_dw_mkv_segment_title, 0x7B,0xA9,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_muxing_app   , 0x4D,0x80,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_writing_app  , 0x57,0x41,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_timecodescale, 0x2A,0xD7,0xB1,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_duration     , 0x44,0x89,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_tracks       , 0x16,0x54,0xAE,0x6B ) ;
DEF_ST_CO( st_co_dw_mkv_track_entry  , 0xAE,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_language     , 0x22,0xB5,0x9C,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_codec_name   , 0x25,0x86,0x88,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_codec_id     , 0x86,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_codec_priv   , 0x63,0xA2,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_video        , 0xE0,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_audio        , 0xE1,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_pixel_width  , 0xB0,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_pixel_height , 0xBA,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_def_duration , 0x23,0xE3,0x83,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_sampling_freq, 0xB5,0x00,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_channel      , 0x9F,0x00,0x00,0x00 ) ;
/*-------------------------------------------------------------------------*/
DEF_ST_CO( st_co_dw_mkv_tags         , 0x12,0x54,0xC3,0x67 ) ;
DEF_ST_CO( st_co_dw_mkv_tag          , 0x73,0x73,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_simple_tag   , 0x67,0xC8,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_tag_name     , 0x45,0xA3,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_tag_string   , 0x44,0x87,0x00,0x00 ) ;
/*-------------------------------------------------------------------------*/
DEF_ST_CO( st_co_dw_mkv_attach       , 0x19,0x41,0xA4,0x69 ) ;
DEF_ST_CO( st_co_dw_mkv_attach_file  , 0x61,0xA7,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_attach_mime  , 0x46,0x60,0x00,0x00 ) ;
DEF_ST_CO( st_co_dw_mkv_attach_data  , 0x46,0x5C,0x00,0x00 ) ;
/*-------------------------------------------------------------------------*/
#undef DEF_ST_CO
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Data used during the extraction                                           !
+--------------------------------------------------------------------------*/
class CFileInit_mkv : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_mkv( CFileInit &parent )
                   : CFileInit_type_base( parent ),
                     m_fo_offset_seek_start( 0 ),
                     m_boo_segment_info_read( false ),
                     m_boo_tracks_read( false ),
                     m_boo_has_video( false ), m_boo_end( false ),
                     m_i_nb_track_sub( 0 ),
                     m_i_track_type( -1 ), // for cppcheck
                     m_i_width( -1 ), m_i_height( -1 ), m_do_fps( -1 ),
                     m_i_duration( -1 ), m_i_nb_channel( -1 ),
                     m_i_samprate( -1 )
      {  ; }

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      enum e_track_type
      {  TT_UNKNOWN, TT_VIDEO, TT_AUDIO, TT_SUB, TT_NB } ;
      /*-------------------------------------------------------------------*/
      wxFileOffset m_fo_offset_seek_start ;
      bool     m_boo_segment_info_read    ;
      bool     m_boo_tracks_read          ;
      /*-------------------------------------------------------------------*/
      bool     m_boo_has_video      ;
      bool     m_boo_end            ; // Time to stop reading the file ?
      int      m_i_nb_track_sub     ; // Number of subtitle tracks
      /*-------------------------------------------------------------------*/
      int      m_i_track_type       ;
      wxString m_s_codec            ;
      wxString m_s_language         ;
      int      m_i_width            ;
      int      m_i_height           ;
      double   m_do_fps             ;
      int      m_i_duration         ;
      int      m_i_nb_channel       ;
      int      m_i_samprate         ;
      wxString m_s_title            ;
      wxString m_s_mime_type_attach ;
      wxString m_s_software         ;
      /*-------------------------------------------------------------------*/
      wxString m_s_audio_info       ;
      wxString m_s_video_info       ;
      wxString m_s_sub_info         ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int read_elmt_id( wxUint32 &dw, int &i_nb_read ) ;
      int read_elmt_size( wxUint64 &ddw, int &i_nb_read ) ;
      int read_float( wxUint64 ddw_size, double &do_val ) ;
      int read_elmt( wxUint32 &dw_id, wxUint64 &ddw_size, int &i_nb_read ) ;
      int read_next_tag( wxUint64 ddw_size_parent ) ;
      int read_next_seek( wxUint64 ddw_size_parent, wxUint64 &ddw_seek_pos );
      int read_meta_seek( wxUint64 ddw_size_parent ) ;
      int read_next_attach( wxUint64 ddw_size_parent ) ;
      int read_struct( wxUint32 dw_id_parent, wxUint64 ddw_size_parent ) ;
      int mkv_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_elmt_id( wxUint32 &dw, int &i_nb_read )
{
   /*----------------------------------------------------------------------*/
   wxUint8 b_val        ;
   int     i_nb_to_read ;

   /*----------------------------------------------------------------------*/
   i_nb_read = 0 ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_data( b_val ) != 0 )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! Number of bytes to read: the size is coded with an UTF-8 like system   !
   +-----------------------------------------------------------------------*/
   for( i_nb_to_read = 0 ;
           i_nb_to_read < 4
        && ( b_val & ( 1 << ( 4 + 3 - i_nb_to_read ) ) ) == 0 ;
        ++i_nb_to_read
      )
   {  ; }
   /*--( "All zeroes" is not supposed to be possible )---------------------*/
   if( i_nb_to_read >= 4 ) { return( -2 ) ; }

   /*--( The id in the right order for the comparison )--------------------*/
   dw = 0 ; *( ( wxUint8 * )&dw ) = b_val ;

   /*----------------------------------------------------------------------*/
   if( i_nb_to_read > 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.read_buffer( i_nb_to_read, ( wxUint8 * )&dw + 1 ) != 0 )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   i_nb_read = i_nb_to_read + 1 ;
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   dw = wxUINT32_SWAP_ALWAYS( dw ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_elmt_size( wxUint64 &ddw, int &i_nb_read )
{
   /*----------------------------------------------------------------------*/
   wxUint8 b_val        ;
   int     i_nb_to_read ;
   wxUint8 *p_ddw       ;

   /*----------------------------------------------------------------------*/
   i_nb_read = 0 ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_data( b_val ) != 0 )
   {  return( -1 ) ; }

   /*--( Number of bytes to read )-----------------------------------------*/
   for( i_nb_to_read = 0 ;
           i_nb_to_read < 8
        && ( b_val & ( 1 << ( 4 + 3 - i_nb_to_read ) ) ) == 0 ;
        ++i_nb_to_read
      )
   {  ; }

   /*--( "All zeroes" is not supposed to be possible )---------------------*/
   if( i_nb_to_read >= 8 ) { return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   ddw = 0 ;
   p_ddw = ( wxUint8 * )&ddw + sizeof( ddw ) - i_nb_to_read - 1 ;
   /*--( Suppress the "limit" bit: it is not part of the value )-----------*/
   *p_ddw = ( b_val ^ ( 1 << ( 7 - i_nb_to_read ) ) ) ;

   /*----------------------------------------------------------------------*/
   if( i_nb_to_read > 0 )
   {
      /*--( Store the data after the first byte set )----------------------*/
      if( m_fa.read_buffer( i_nb_to_read, p_ddw + 1 ) != 0 )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   i_nb_read = i_nb_to_read + 1 ;
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   ddw = wxUINT64_SWAP_ALWAYS( ddw ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_float( wxUint64 ddw_size, double &do_val )
{
   /*----------------------------------------------------------------------*/
   switch( ddw_size )
   {
      /*-------------------------------------------------------------------*/
      case 4 :
      {  /*----------------------------------------------------------------*/
         sr::t_dw_fl uf ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_be_data( ddw_size, uf.dw_val ) != 0 )
         {  return( -1 ) ; }
         do_val = uf.f_val ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      case 8 :
      {  /*----------------------------------------------------------------*/
         sr::t_ddw_do uf ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_be_data( ddw_size, uf.ddw_val ) != 0 )
         {  return( -2 ) ;}
         do_val = uf.do_val ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      default : return( -3 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_elmt( wxUint32 &dw_id, wxUint64 &ddw_size,
                              int &i_nb_read
                            )
{
   /*----------------------------------------------------------------------*/
   int i_nb_read_id   = 0 ;
   int i_nb_read_size = 0 ;
   /*----------------------------------------------------------------------*/
   if( read_elmt_id( dw_id, i_nb_read_id ) != 0 ) { return( -1 ) ; }
   if( read_elmt_size( ddw_size, i_nb_read_size ) != 0 ) { return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   i_nb_read = i_nb_read_id + i_nb_read_size ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static wxString st_mkv_audio_codec_desc( const wxString &s_codec_id )
{
   /*----------------------------------------------------------------------*/
   wxString s_rest ;
   /*----------------------------------------------------------------------*/
   if( s_codec_id == "A_MPEG/L3" )
   {  return( "Mpeg L3" ) ; }
   if( s_codec_id == "A_MPEG/L2" )
   {  return( "Mpeg L2" ) ; }
   if( s_codec_id == "A_MPEG/L1" )
   {  return( "Mpeg L1" ) ; }
   if( s_codec_id == "A_PCM/INT/BIG" )
   {  return( "PCM BE" ) ; }
   if( s_codec_id == "A_PCM/INT/LIT" )
   {  return( "PCM LE" ) ; }
   if( s_codec_id == "A_PCM/FLOAT/IEEE" )
   {  return( "PCM float" ) ; }
   if( s_codec_id == "A_MPC" )
   {  return( "MPC SV8" ) ; }
   if( s_codec_id.StartsWith( "A_AC3" ) )
   {  return( "AC3" ) ; }
   if( s_codec_id == "A_DTS" )
   {  return( "DTS" ) ; }
   if( s_codec_id == "A_VORBIS" )
   {  return( "Vorbis" ) ; }
   if( s_codec_id == "A_FLAC" )
   {  return( "Flac" ) ; }
   /*----------------------------------------------------------------------*/
   if( s_codec_id.StartsWith( "A_REAL/", &s_rest ) )
   {  /*-------------------------------------------------------------------*/
      if( s_rest == "14_4" ) { return( "Real Audio 1" ) ; }
      if( s_rest == "28_8" ) { return( "Real Audio 2" ) ; }
      if( s_rest == "COOK" ) { return( "Real Audio Cook" ) ; }
      if( s_rest == "SIPR" ) { return( "Sipro voice" ) ; }
      if( s_rest == "RALF" ) { return( "Real Audio Lossless" ) ; }
      if( s_rest == "ATRC" ) { return( "Sony Atrac3" ) ; }
      /*-------------------------------------------------------------------*/
      return( "Real Audio" ) ;
      /*-------------------------------------------------------------------*/
   }
   if( s_codec_id == "A_MS/ACM" )
   {  return( "Microsoft ACM" ) ; }
   /*--( Many different codecs ... make it short )-------------------------*/
   if( s_codec_id == "A_AAC" || s_codec_id.StartsWith( "A_AAC/" ) )
   {  return( "AAC" ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_codec_id == "A_QUICKTIME"
       || s_codec_id.StartsWith( "A_QUICKTIME/" )
     )
   {  return( "Quicktime Audio" ) ; }
   if( s_codec_id == "A_TTA1" )
   {  return( "The True Audio" ) ; }
   if( s_codec_id == "A_WAVPACK4" )
   {  return( "WavPack" ) ; }
   if( s_codec_id == "A_TRUEHD" )
   {  return( "Dolby TrueHD" ) ; }
   /*----------------------------------------------------------------------*/
   return( s_codec_id ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static wxString st_mkv_video_codec_desc( const wxString &s_codec_id )
{
   /*----------------------------------------------------------------------*/
   wxString s_rest ;
   /*----------------------------------------------------------------------*/
   if( s_codec_id == "V_UNCOMPRESSED" )
   {  return( "Video raw" ) ; }
   if( s_codec_id.StartsWith( "V_MPEG4/ISO/", &s_rest ) )
   {  /*-------------------------------------------------------------------*/
      if( s_rest == "SP"  ) { return( "DivX 4" ) ; }
      if( s_rest == "ASP" ) { return( "DivX 5" ) ; }
      if( s_rest == "AP"  ) { return( "Mpeg4 ISO AP" ) ; }
      if( s_rest == "AVC" ) { return( "H.264" ) ; }
      /*-------------------------------------------------------------------*/
      return( "Mpeg4 ISO" ) ;
      /*-------------------------------------------------------------------*/
   }
   if( s_codec_id == "V_MPEG4/MS/V3" )
   {  return( "Microsoft Mpeg4 V3" ) ; }
   if( s_codec_id == "V_MPEG1" )
   {  return( "Mpeg1" ) ; }
   if( s_codec_id == "V_MPEG2" )
   {  return( "Mpeg2" ) ; }
   if( s_codec_id.StartsWith( "V_REAL/", &s_rest ) )
   {  /*-------------------------------------------------------------------*/
      if( s_rest == "RV10" ) { return( "Real Video 5"  ) ; }
      if( s_rest == "RV20" ) { return( "Real Video G2" ) ; }
      if( s_rest == "RV30" ) { return( "Real Video 8"  ) ; }
      if( s_rest == "RV40" ) { return( "Real Video 9"  ) ; }
      /*-------------------------------------------------------------------*/
      return( "Real Video" ) ;
      /*-------------------------------------------------------------------*/
   }
   if( s_codec_id == "V_QUICKTIME" )
   {  return( "Quicktime Video" ) ; }
   if( s_codec_id == "V_THEORA" )
   {  return( "Theora" ) ; }
   if( s_codec_id == "V_VP8" )
   {  return( "VP8" ) ; }
   /*----------------------------------------------------------------------*/
   return( s_codec_id ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_mkv_conv_tag_col( bool boo_video, const wxString &s_var )
{
   /*-----------------------------------------------------------------------+
   ! Unused COL: COL_AUDTAG_ALBUM, COL_AUDTAG_COMPOS, COL_AUDTAG_ORG_ART    !
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "TITLE" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_TITLE : COL_AUDTAG_TITLE ) ; }
   if( s_var.CmpNoCase( "SUBJECT" ) == 0 )
   {  return( boo_video ?  COL_VIDTAG_SUBJECT : COL_NB ) ; }
   if( s_var.CmpNoCase( "ARTIST" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ARTIST ) ; }
   if( s_var.CmpNoCase( "COMMENT" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_COMMENT : COL_AUDTAG_COMMENT ) ; }
   if( s_var.CmpNoCase( "KEYWORDS" ) == 0 )
   {  return( boo_video ?  COL_VIDTAG_KEYWORDS : COL_NB ) ; }
   if( s_var.CmpNoCase( "DIRECTOR" ) == 0 )
   {  return( boo_video ?  COL_VIDTAG_ENGINEER : COL_NB ) ; }
   if( s_var.CmpNoCase( "PRODUCER" ) == 0 )
   {  return( boo_video ?  COL_VIDTAG_TECHNICIAN : COL_NB ) ; }
   if( s_var.CmpNoCase( "GENRE" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_GENRE : COL_AUDTAG_GENRE ) ; }
   if( s_var.CmpNoCase( "DATE_RELEASED" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_CRE_DATE : COL_NB ) ; }
   if( s_var.CmpNoCase( "ORIGINAL_MEDIA_TYPE" ) == 0 )
   {  return( boo_video ?  COL_VIDTAG_SOURCE : COL_NB ) ; }
   if( s_var.CmpNoCase( "COPYRIGHT" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_COPYRIGHT : COL_AUDTAG_COPYRIGHT ) ; }
   if(    s_var.CmpNoCase( "ENCODER" ) == 0
       || s_var.CmpNoCase( "ENCODED_BY" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_SOFTWARE : COL_AUDTAG_ENCOD_BY  ) ; }
   if( s_var.CmpNoCase( "DATE_RELEASED" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_YEAR ) ; }
   if( s_var.CmpNoCase( "PART_NUMBER" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_TRACK_NUM ) ; }
   if( s_var.CmpNoCase( "TOTAL_PARTS" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_TRACK_NB ) ; }
   if( s_var.CmpNoCase( "CATALOG_NUMBER" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_DISK ) ; }
   if( s_var.CmpNoCase( "URL" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_URL ) ; }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_next_tag( wxUint64 ddw_size_parent )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id     ;
   wxUint64 ddw_size  ;
   int      i_nb_read ;
   wxString s_var     ;
   wxString s_val     ;

   /*----------------------------------------------------------------------*/
   m_fi.set_conv_from_utf8() ;
   /*----------------------------------------------------------------------*/
   while( ( s_var.empty() || s_val.empty() ) && ddw_size_parent > 0 )
   {
      /*--( Read the element id and size )---------------------------------*/
      if( read_elmt( dw_id, ddw_size, i_nb_read ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= i_nb_read ;

      /*-------------------------------------------------------------------*/
      switch( dw_id )
      {
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_tag_name :
            /*-------------------------------------------------------------*/
            if( m_fi.file_read_tb_c( ddw_size, s_var ) != 0 )
            {  return( -2 ) ; }
            break ;
            /*-------------------------------------------------------------*/
         case st_co_dw_mkv_tag_string :
            /*-------------------------------------------------------------*/
            if( m_fi.file_read_tb_c( ddw_size, s_val ) != 0 )
            {  return( -3 ) ; }
            break ;
         /*----------------------------------------------------------------*/
         default :
            /*-------------------------------------------------------------*/
            if( m_fa.skip_nb_byte( ddw_size ) != 0 ) { return( -4 ) ; }
            break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      ddw_size_parent -= ddw_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( ddw_size_parent > 0 && m_fa.skip_nb_byte( ddw_size_parent ) != 0 )
   {  return( -5 ) ; }

   /*----------------------------------------------------------------------*/
   if( !s_var.empty() && !s_val.empty() )
   {
      /*-------------------------------------------------------------------*/
      int i_col = st_mkv_conv_tag_col( m_boo_has_video, s_var ) ;
      /*-------------------------------------------------------------------*/
      if( m_fi.reserve_col( i_col ) )
      {  /*----------------------------------------------------------------*/
         long l_val ;
         /*--( Track "numbers" string )------------------------------------*/
         if(    (    i_col == COL_AUDTAG_TRACK_NUM
                  || i_col == COL_AUDTAG_TRACK_NB
                )
             && s_val.ToLong( &l_val )
           )
         {  m_f.val_ll( i_col ) = l_val ; }
         else
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_next_seek( wxUint64 ddw_size_parent,
                                   wxUint64 &ddw_seek_pos
                                 )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_seek_id      ;
   wxUint32 dw_id           ;
   wxUint64 ddw_size        ;
   int      i_nb_read       ;
   bool     boo_end = false ;

   /*----------------------------------------------------------------------*/
   dw_seek_id   = 0 ;
   ddw_seek_pos = 0 ;

   /*----------------------------------------------------------------------*/
   while(    !boo_end
          && ( dw_seek_id == 0 || ddw_seek_pos == 0 )
          && ddw_size_parent > 0
        )
   {
      /*--( Read the element id and size )---------------------------------*/
      if( read_elmt( dw_id, ddw_size, i_nb_read ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= i_nb_read ;

      /*-------------------------------------------------------------------*/
      switch( dw_id )
      {
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_seek_id :
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, dw_seek_id ) != 0 )
            {  return( -2 ) ; }
            /*--( Only interested in a few info )--------------------------*/
            if(    dw_seek_id != st_co_dw_mkv_segment_info
                && dw_seek_id != st_co_dw_mkv_tracks
                && dw_seek_id != st_co_dw_mkv_tags
                && dw_seek_id != st_co_dw_mkv_attach
              )
            {  boo_end = true ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         case st_co_dw_mkv_seek_pos :
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, ddw_seek_pos ) != 0 )
            {  return( -3 ) ; }
            break ;
         /*----------------------------------------------------------------*/
         default :
            /*-------------------------------------------------------------*/
            if( m_fa.skip_nb_byte( ddw_size ) != 0 ) { return( -4 ) ; }
            break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      ddw_size_parent -= ddw_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( ddw_size_parent > 0 && m_fa.skip_nb_byte( ddw_size_parent ) != 0 )
   {  return( -5 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( The "seek header" and its size are supposed to have been read )------*/
int CFileInit_mkv::read_meta_seek( wxUint64 ddw_size_parent )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id         ;
   int      i_nb_read     ;
   wxUint64 ddw_seek_pos  ;
   wxUint64 ddw_seek_size ;
   /*----------------------------------------------------------------------*/
   std::vector< wxUint64 > vec_seek_pos ;

   /*----------------------------------------------------------------------*/
   do
   {
      /*--( Now should come the seek info )--------------------------------*/
      if(    read_elmt( dw_id, ddw_seek_size, i_nb_read ) != 0
          || dw_id != st_co_dw_mkv_seek
        )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= i_nb_read ;

      /*-------------------------------------------------------------------*/
      if( read_next_seek( ddw_seek_size, ddw_seek_pos ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= ddw_seek_size ;

      /*--( To keep ? )----------------------------------------------------*/
      if( ddw_seek_pos != 0 ) { vec_seek_pos.push_back( ddw_seek_pos ) ; }
      /*-------------------------------------------------------------------*/

   } while( ddw_size_parent > 0 ) ;

   /*----------------------------------------------------------------------*/
   if( ddw_size_parent > 0 && m_fa.skip_nb_byte( ddw_size_parent ) != 0 )
   {  return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   if( !vec_seek_pos.empty() )
   {
      /*-------------------------------------------------------------------*/
      wxUint64 ddw_size ;
      size_t   sz_num   ;
      /*-------------------------------------------------------------------*/
      for( sz_num = 0 ; sz_num < vec_seek_pos.size() ; ++sz_num )
      {
         /*----------------------------------------------------------------*/
         if( m_fa.set_offset( m_fo_offset_seek_start + vec_seek_pos[ sz_num ]
                            ) != 0
           )
         {  return( -4 ) ; }
         /*--( The id read is a known sub structure )----------------------*/
         if( read_elmt( dw_id, ddw_size, i_nb_read ) != 0 )
         {  return( -5 ) ; }
         /*--( Analyse this referenced element )---------------------------*/
         if( read_struct( dw_id, ddw_size ) != 0 )
         {  return( -6 ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::read_next_attach( wxUint64 ddw_size_parent )
{
   /*----------------------------------------------------------------------*/
   wxUint32     dw_id             ;
   wxUint64     ddw_size          ;
   int          i_nb_read         ;
   wxString     s_mime            ;
   wxFileOffset fo_data       = 0 ;
   wxUint64     ddw_data_size = 0 ;

   /*----------------------------------------------------------------------*/
   while( ( s_mime.empty() || fo_data == 0 ) && ddw_size_parent > 0 )
   {
      /*--( Read the element id and size )---------------------------------*/
      if( read_elmt( dw_id, ddw_size, i_nb_read ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= i_nb_read ;

      /*-------------------------------------------------------------------*/
      switch( dw_id )
      {
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_attach_mime :
            /*-------------------------------------------------------------*/
            m_fi.set_conv_from_8bit() ;
            if( m_fi.file_read_tb_c( ddw_size, s_mime ) != 0 )
            {  return( -2 ) ; }
            break ;
            /*-------------------------------------------------------------*/
         case st_co_dw_mkv_attach_data :
            /*-------------------------------------------------------------*/
            fo_data       = m_fa.get_offset() ;
            ddw_data_size = ddw_size ;
            /*--( Got to pass the data to find the next element )----------*/
            if( m_fa.skip_nb_byte( ddw_data_size ) != 0 ) { return( -3 ) ; }
            break ;
         /*----------------------------------------------------------------*/
         default :
            /*-------------------------------------------------------------*/
            if( m_fa.skip_nb_byte( ddw_size ) != 0 ) { return( -4 ) ; }
            break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      ddw_size_parent -= ddw_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( ddw_size_parent > 0 && m_fa.skip_nb_byte( ddw_size_parent ) != 0 )
   {  return( -5 ) ; }

   /*--( An image is expected )--------------------------------------------*/
   if(    fo_data != 0
       && !s_mime.empty() && s_mime.Lower().StartsWith( "image/" )
     )
   {  /*-------------------------------------------------------------------*/
      m_s_mime_type_attach = s_mime ;
      m_fi.read_embedded_image_info( fo_data, ddw_data_size, s_mime ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The ID & size of the block are supposed to just have been read            !
! Caution: this function is recursive                                       !
+--------------------------------------------------------------------------*/
int CFileInit_mkv::read_struct( wxUint32 dw_id_parent,
                                wxUint64 ddw_size_parent
                              )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id                                 ;
   wxUint64 ddw_size                              ;
   int      i_nb_read                             ;
   wxUint64 ddw_timecodescale        = 1000000    ;
   bool     boo_codec_priv_ms_fourcc = false      ;
   bool     boo_skip                              ;

   /*----------------------------------------------------------------------*/
   while( !m_boo_end && ddw_size_parent > 0 )
   {
      /*--( Read the element id and size )---------------------------------*/
      if( read_elmt( dw_id, ddw_size, i_nb_read ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= i_nb_read ;
      boo_skip = true ;

      /*-------------------------------------------------------------------*/
      switch( dw_id )
      {
         /*-----------------------------------------------------------------+
         ! ATTACHMENT                                                       !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_attach_file :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_attach ) { break ; }
            /*--( Only the first attached image )--------------------------*/
            if( !m_s_mime_type_attach.empty() ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_next_attach( ddw_size ) != 0 )
            {  return( -2 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! SIMPLE TAG                                                       !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_simple_tag :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_tag ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_next_tag( ddw_size ) != 0 )
            {  return( -3 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! TAGS                                                             !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_tag :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_tags ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_struct( dw_id, ddw_size ) != 0 )
            {  return( -4 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! VIDEO                                                            !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_pixel_width :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_video ) { break ; }
            /*-------------------------------------------------------------*/
            wxUint64 ddw_width ;
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, ddw_width ) != 0 )
            {  return( -5 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            if( m_i_width < 0 ) { m_i_width = ddw_width ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_pixel_height :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_video ) { break ; }
            /*-------------------------------------------------------------*/
            wxUint64 ddw_height ;
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, ddw_height ) != 0 )
            {  return( -6 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            if( m_i_height < 0 ) { m_i_height = ddw_height ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*-----------------------------------------------------------------+
         ! AUDIO                                                            !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_channel :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_audio ) { break ; }
            /*-------------------------------------------------------------*/
            wxUint64 ddw_channel ;
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, ddw_channel ) != 0 )
            {  return( -7 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            if( m_i_nb_channel < 0 ) { m_i_nb_channel = ddw_channel ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_sampling_freq :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_audio ) { break ; }
            /*-------------------------------------------------------------*/
            double do_sampling_freq ;
            /*-------------------------------------------------------------*/
            if( read_float( ddw_size, do_sampling_freq ) != 0 )
            {  return( -8 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            if( m_i_samprate < 0 )
            {  m_i_samprate = ( int )do_sampling_freq ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*-----------------------------------------------------------------+
         ! TRACK ENTRY                                                      !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_def_duration :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_track_entry ) { break ; }
            /*--------------------------------------------------------------+
            ! What is called "default duration" is in fact the number of    !
            ! nanoseconds per frame                                         !
            +--------------------------------------------------------------*/
            wxUint64 ddw_def_duration ;
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, ddw_def_duration ) != 0 )
            {  return( -10 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            if(    ddw_def_duration > 0
                && m_do_fps < 0
              )
            {  m_do_fps = 1000000000.0 / ddw_def_duration ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_language :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_track_entry ) { break ; }
            /*--( Add it only for audio and sub tracks in video files )----*/
            if(    !m_boo_has_video
                || ( m_i_track_type != TT_AUDIO && m_i_track_type != TT_SUB )
              )
            {  break ; }
            /*-------------------------------------------------------------*/
            m_fi.set_conv_from_8bit() ;
            if( m_fi.file_read_tb_c( ddw_size, m_s_language ) != 0 )
            {  return( -11 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_codec_id :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_track_entry ) { break ; }
            /*-------------------------------------------------------------*/
            wxString s_codec_id ;
            /*-------------------------------------------------------------*/
            m_fi.set_conv_from_8bit() ;
            if( m_fi.file_read_tb_c( ddw_size, s_codec_id ) != 0 )
            {  return( -12 ) ; }
            boo_skip = false ;

            /*--------------------------------------------------------------+
            ! Codec information may appear after the track type. So the     !
            ! initialization is done based on the codec name.               !
            +--------------------------------------------------------------*/
            if( s_codec_id.StartsWith( "V_" ) )
            {  m_i_track_type = TT_VIDEO ; m_boo_has_video = true ; }
            else
            if( s_codec_id.StartsWith( "A_" ) )
            {  m_i_track_type = TT_AUDIO ; }
            else
            if( s_codec_id.StartsWith( "S_" ) )
            {  m_i_track_type = TT_SUB ; ++m_i_nb_track_sub ; }

            /*--------------------------------------------------------------+
            ! If the id is "V_MS/VFW/FOURCC" then the id                    !
            ! "st_co_dw_mkv_codec_priv" attribute will contain the full     !
            ! BITMAPINFOHEADER                                              !
            +--------------------------------------------------------------*/
            if( s_codec_id == "V_MS/VFW/FOURCC" )
            {  boo_codec_priv_ms_fourcc = true ; }
            else
            if( m_i_track_type == TT_AUDIO )
            {  m_s_codec = st_mkv_audio_codec_desc( s_codec_id ) ; }
            else
            if( m_i_track_type == TT_VIDEO )
            {  m_s_codec = st_mkv_video_codec_desc( s_codec_id ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_codec_priv :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_track_entry ) { break ; }
            /*--( FOURCC only meaningful for video )-----------------------*/
            if( m_i_track_type != TT_VIDEO || !boo_codec_priv_ms_fourcc )
            {  break ; }
            /*--------------------------------------------------------------+
            ! The BITMAPINFOHEADER is present in this element only for the  !
            ! "V_MS/VFW/FOURCC" codec                                       !
            +--------------------------------------------------------------*/
            str_BITMAPINFOHEADER bmp_ih    ;
            wxString             s_codec   ;
            int                  i_skip_nb ;
            /*--( Extract the information )--------------------------------*/
            if( m_fa.read_le_data( bmp_ih ) != 0 ) { return( -13 ) ; }
            /*--( And skip the rest )--------------------------------------*/
            i_skip_nb = ddw_size - sizeof( bmp_ih ) ;
            if( i_skip_nb > 0 && m_fa.skip_nb_byte( i_skip_nb ) != 0 )
            {  return( -14 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            m_fi.info_fcc_video( (char *)&bmp_ih.biCompression, s_codec ) ;
            m_fi.add_codec_info( s_codec, m_s_video_info ) ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_video :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_track_entry ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_struct( dw_id, ddw_size ) != 0 )
            {  return( -15 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_audio :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_track_entry ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_struct( dw_id, ddw_size ) != 0 )
            {  return( -16 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! TRACK                                                            !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_track_entry :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_tracks ) { break ; }
            /*-------------------------------------------------------------*/
            m_s_codec.clear() ;
            m_s_language.clear() ;
            m_i_track_type = TT_UNKNOWN ;

            /*--( Read complete track information )------------------------*/
            if( read_struct( dw_id, ddw_size ) != 0 )
            {  return( -17 ) ; }
            boo_skip = false ;

            /*--( Adding codec information )-------------------------------*/
            switch( m_i_track_type )
            {
               /*----------------------------------------------------------*/
               case TT_AUDIO :
                  m_fi.add_codec_info( m_s_codec, m_s_audio_info ) ;
                  break ;
               /*----------------------------------------------------------*/
               case TT_VIDEO :
                  m_fi.add_codec_info( m_s_codec, m_s_video_info ) ;
                  break ;
               /*----------------------------------------------------------*/
            }

            /*--( Default language is english )----------------------------*/
            if( m_s_language.empty() ) { m_s_language = "eng" ; }

            /*--( Adding language information to codec ones )--------------*/
            if( m_s_language != "und" )
            {
               /*----------------------------------------------------------*/
               switch( m_i_track_type )
               {
                  /*-------------------------------------------------------*/
                  case TT_AUDIO :
                     m_s_audio_info += " (" + m_s_language + ')' ;
                     break ;
                  /*-------------------------------------------------------*/
                  case TT_SUB :
                     if( !m_s_sub_info.empty() )
                     {  m_s_sub_info += ", " ; }
                     m_s_sub_info += m_s_language ;
                     break ;
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! SEGMENT INFO                                                     !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_segment_title :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_segment_info ) { break ; }
            /*--------------------------------------------------------------+
            ! The title is saved in a variable and maybe set later because  !
            ! it has been decided to give the priority to a title defined   !
            ! in a tags section.                                            !
            +--------------------------------------------------------------*/
            m_fi.set_conv_from_utf8() ;
            if( m_fi.file_read_tb_c( ddw_size, m_s_title ) != 0 )
            {  return( -18 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_muxing_app :
         case st_co_dw_mkv_writing_app :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_segment_info ) { break ; }
            /*-------------------------------------------------------------*/
            wxString s_app ;
            /*--------------------------------------------------------------+
            ! Why saving soft in a variable ? Same reason as for the title  !
            +--------------------------------------------------------------*/
            m_fi.set_conv_from_utf8() ;
            if( m_fi.file_read_tb_c( ddw_size, s_app ) != 0 )
            {  return( -19 ) ; }
            boo_skip = false ;
            /*--( Muxing app first )---------------------------------------*/
            if( dw_id == st_co_dw_mkv_writing_app )
            {  sr::append_with_sep( s_app, " - ", m_s_software ) ; }
            else
            {  sr::prepend_with_sep( s_app, " - ", m_s_software ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_duration :
         {
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_segment_info ) { break ; }
            /*-------------------------------------------------------------*/
            double do_duration ;
            /*-------------------------------------------------------------*/
            if( read_float( ddw_size, do_duration ) != 0 )
            {  return( -20 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            m_i_duration = ( int )(   do_duration
                                    * ddw_timecodescale / 1000000000.0
                                  ) ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_timecodescale :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_segment_info ) { break ; }
            /*-------------------------------------------------------------*/
            if( m_fa.read_be_data( ddw_size, ddw_timecodescale ) != 0 )
            {  return( -21 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! SEGMENT                                                          !
         +-----------------------------------------------------------------*/
         case st_co_dw_mkv_seek_head :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_segment ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_meta_seek( ddw_size ) != 0 )
            {  return( -22 ) ; }
            boo_skip = false ;
            /*--( The minimum has been done ? )----------------------------*/
            if( m_boo_segment_info_read && m_boo_tracks_read )
            {  m_boo_end = true ; }
            /*-------------------------------------------------------------*/
            break ;
         /*----------------------------------------------------------------*/
         case st_co_dw_mkv_segment_info :
         case st_co_dw_mkv_tracks :
         case st_co_dw_mkv_tags :
         case st_co_dw_mkv_attach :
            /*-------------------------------------------------------------*/
            if( dw_id_parent != st_co_dw_mkv_segment ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_struct( dw_id, ddw_size ) != 0 )
            {  return( -23 ) ; }
            boo_skip = false ;
            /*-------------------------------------------------------------*/
            break ;

         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      if( boo_skip && m_fa.skip_nb_byte( ddw_size ) != 0 ) { return( -24 ) ;}
      /*-------------------------------------------------------------------*/
      ddw_size_parent -= ddw_size ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Section done ? )--------------------------------------------------*/
   switch( dw_id_parent )
   {
      /*-------------------------------------------------------------------*/
      case st_co_dw_mkv_segment_info :
         m_boo_segment_info_read = true ;
         break ;
      /*-------------------------------------------------------------------*/
      case st_co_dw_mkv_tracks :
         m_boo_tracks_read = true ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_mkv::mkv_read()
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id     ;
   wxUint64 ddw_size  ;
   int      i_nb_read ;

   /*--( Is it a valid EBML file ? )---------------------------------------*/
   if( m_fa.read_be_data( dw_id ) != 0 )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! The file can start with a text ending with a Ctrl-Z character. This    !
   ! way under DOS a type command can be executed ...                       !
   ! So, if needed the first Ctrl-Z is searched for.                        !
   +-----------------------------------------------------------------------*/
   if( dw_id != st_co_dw_mkv_header )
   {  /*-------------------------------------------------------------------*/
      wxFileOffset fo_start ;
      /*-------------------------------------------------------------------*/
      if(    m_fa.set_offset( 0 ) != 0
          || m_fa.search_next_seq( sr::ctrl_char('Z'), 1024, fo_start ) != 0
        )
      {  return( -2 ) ; }
      /*--( The search has skipped the Ctrl-Z )----------------------------*/
      if( m_fa.set_offset( fo_start - 1 ) != 0 )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      if(    m_fa.read_be_data( dw_id ) != 0
          || dw_id != st_co_dw_mkv_header
        )
      {  return( -4 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( If so, skip the "basics" )----------------------------------------*/
   if(    read_elmt_size( ddw_size, i_nb_read ) != 0
       || m_fa.skip_nb_byte( ddw_size ) != 0
     )
   {  return( -5 ) ; }

   /*-----------------------------------------------------------------------+
   ! Then should come the first segment                                     !
   +-----------------------------------------------------------------------*/
   if( read_elmt( dw_id, ddw_size, i_nb_read ) != 0 )
   {  return( -6 ) ; }
   if( dw_id != st_co_dw_mkv_segment ) { return( -7 ) ; }
   /*----------------------------------------------------------------------*/
   m_fo_offset_seek_start = m_fa.get_offset() ;

   /*-----------------------------------------------------------------------+
   ! It starts here with this recursive function                            !
   ! No return test because the file may be strangely padded ....           !
   +-----------------------------------------------------------------------*/
   read_struct( dw_id, ddw_size ) ;

   /*--( Default values as specified in Matroska docs )--------------------*/
   if( m_i_nb_channel < 0 ) { m_i_nb_channel = 1 ; }
   if( m_i_samprate   < 0 ) { m_i_samprate   = 8000 ; }

   /*----------------------------------------------------------------------*/
   if( m_boo_has_video && m_i_width > 0 && m_i_height > 0 )
   {  m_fi.init_video_x_y( m_i_width, m_i_height ) ; }

   /*----------------------------------------------------------------------*/
   if( m_boo_has_video && m_do_fps > 0 )
   {  m_fi.init_video_fps( m_do_fps ) ; }

   /*----------------------------------------------------------------------*/
   if( m_i_duration >= 0 )
   {  /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {  m_fi.init_video_duration( m_i_duration ) ; }
      else
      {  m_fi.init_audio_duration( m_i_duration ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( m_i_nb_channel > 0 )
   {  /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {  m_fi.init_video_channel( m_i_nb_channel ) ; }
      else
      {  m_fi.init_audio_channel( m_i_nb_channel ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( m_i_samprate > 0 )
   {  /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {  m_fi.init_video_samprate( m_i_samprate ) ; }
      else
      {  m_fi.init_audio_samprate( m_i_samprate ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !m_s_audio_info.empty() || !m_s_video_info.empty() )
   {
      /*-------------------------------------------------------------------*/
      wxString s_info
             = m_fi.concat_av_codecs( m_s_audio_info, m_s_video_info ) ;
      /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {
         /*--( If any, add the number of subtitle tracks to the info )-----*/
         if( m_i_nb_track_sub > 0 )
         {
            /*-------------------------------------------------------------*/
            s_info.append( wxString::Format( m_i_nb_track_sub == 1
                                             ? " - %d sub" : " - %d subs",
                                             m_i_nb_track_sub
                                           )
                         ) ;
            /*-------------------------------------------------------------*/
            if( !m_s_sub_info.empty() )
            {  s_info.append( " (" + m_s_sub_info + ')' ) ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( COL_VIDEO_INFO ) = s_info ;
         /*----------------------------------------------------------------*/
      }
      else
      {  m_f.val_s( COL_AUDIO_INFO ) = s_info ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( If not already set with tags, the "global" title is used )--------*/
   if( !m_s_title.empty() )
   {  /*-------------------------------------------------------------------*/
      int i_col = ( m_boo_has_video ? COL_VIDTAG_TITLE : COL_AUDTAG_TITLE ) ;
      /*-------------------------------------------------------------------*/
      if( m_fi.reserve_col( i_col ) ) { m_f.val_s( i_col ) = m_s_title ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( If not already set with tags, the "global" software is used )-----*/
   if( !m_s_software.empty() )
   {  /*-------------------------------------------------------------------*/
      int i_col
        = ( m_boo_has_video ? COL_VIDTAG_SOFTWARE : COL_AUDTAG_ENCOD_BY ) ;
      /*-------------------------------------------------------------------*/
      if( m_fi.reserve_col( i_col ) )
      {  m_f.val_s( i_col ) = m_s_software ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !m_s_mime_type_attach.empty() )
   {  /*-------------------------------------------------------------------*/
      m_f.val_s( m_boo_has_video
                 ? COL_VIDTAG_IMG_FORMAT : COL_AUDTAG_IMG_FORMAT
               ) = m_s_mime_type_attach ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Convert numbers to strings )--------------------------------------*/
   if( !m_boo_has_video ) { m_f.init_info_s_track() ; }

   /*----------------------------------------------------------------------*/
   m_fi.m_s_type_det = ( m_boo_has_video ? "mkv" : "mka" ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_mkv()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "mka/mkv" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_mkv( *this ).mkv_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_mkv.cpp                       !
+==========================================================================*/
