/*====================================+=====================================+
! File CFileInit_iso.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                        ISO files: ".iso"                                  !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! This code is based on information found here:                             !
! http://alumnus.caltech.edu/~pje/iso9660.html                              !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_iso : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_iso( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }

      /*-------------------------------------------------------------------*/
      int iso_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*--( ISO primary volume descriptor )--------------------------------------*/
struct str_iso_pvd
{
   /*----------------------------------------------------------------------*/
   wxUint8 tb_b_header                [   8 ] ; // 0x01 C D 0 0 1 0x01 0x00
   char    tb_c_system_id             [  32 ] ;
   char    tb_c_volume_id             [  32 ] ;
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_dummy_1            [ 118 ] ; // Info not interesting 4 us
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_volume_set_id      [ 128 ] ;
   char    tb_c_publisher_id          [ 128 ] ;
   char    tb_c_data_preparer_id      [ 128 ] ;
   char    tb_c_application_id        [ 128 ] ;
   char    tb_c_copyright_file_id     [  37 ] ;
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_abstract_file_id   [  37 ] ;
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_biblio_file_id     [  37 ] ;
   char    tb_c_datetime_creation     [  17 ] ;
   char    tb_c_datetime_modification [  17 ] ;
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_datetime_expiration[  17 ] ;
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_datetime_effectiveness[  17 ] ;
   // cppcheck-suppress unusedStructMember
   char    ___tb_c_dummy_2            [   2 ] ; // 0x01 0x00
   char    tb_c_application_use       [ 512 ] ;
/*
   Then comes:
      653  zeros
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*--------------------------------------------------------------------------+
! The format used in ISOs is: YYYYMMDDhhmmddccx                             !
! MM: month 01-12                                                           !
! hh: hour  00-23                                                           !
! cc: hundredths of a second                                                !
! x : supposed to be signed byte giving the number of "15mn" to add to GMT. !
!     It is not used here. The conversion is made from GMT to local time    !
!     zone.                                                                 !
!                                                                           !
! "sr::init_date_ymdhms" is not used function because no default values     !
! are supposed to be used for ISO dates.                                    !
+--------------------------------------------------------------------------*/
static int st_iso_init_datetime( const char tb_c_datetime[ 17 ],
                                 wxDateTime &dt
                               )
{
   /*----------------------------------------------------------------------*/
   int i_year, i_month, i_day ;
   int i_hour, i_minute, i_second, i_hundredth ;

   /*----------------------------------------------------------------------*/
   if( wxSscanf( tb_c_datetime, "%4d%2d%2d%2d%2d%2d%2d",
                 &i_year, &i_month , &i_day,
                 &i_hour, &i_minute, &i_second, &i_hundredth
               ) != 7
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( sr::set_datetime( i_year, i_month, i_day, i_hour,
                         i_minute, i_second, i_hundredth * 10, dt
                       ) != 0
     )
   {  return( -2 ) ; }

   /*--( Conversion from GMT to local time zone )--------------------------*/
   dt.MakeFromTimezone( wxDateTime::GMT0 ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_iso::iso_read()
{
   /*----------------------------------------------------------------------*/
   str_iso_pvd iso_pvd ;
   wxString    s_val   ;
   wxDateTime  dt_val  ;

   /*--( The first 16 sectors are supposed to be empty )-------------------*/
   if( m_fa.set_offset( 2048 * 16 ) != 0 )
   {  return( -1 ) ; }

   /*--( Then comes the primary volume descriptor )------------------------*/
   if( m_fa.read_buffer( sizeof( iso_pvd ), &iso_pvd ) != 0 )
   {  return( -2 ) ; }

   /*--( Constant check )--------------------------------------------------*/
   if(  memcmp( iso_pvd.tb_b_header, "\01CD001\01\00", 8 ) != 0 )
   {  return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_system_id,
                            sizeof( iso_pvd.tb_c_system_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_SUBJECT ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_volume_id,
                            sizeof( iso_pvd.tb_c_volume_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_TITLE ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_publisher_id,
                            sizeof( iso_pvd.tb_c_publisher_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_AUTHOR ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_data_preparer_id,
                            sizeof( iso_pvd.tb_c_data_preparer_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_COMPANY ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_application_id,
                            sizeof( iso_pvd.tb_c_application_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_APPLICATION ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_copyright_file_id,
                            sizeof( iso_pvd.tb_c_copyright_file_id ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_COPYRIGHT ) = s_val ; }
   /*----------------------------------------------------------------------*/
   if( st_iso_init_datetime( iso_pvd.tb_c_datetime_creation, dt_val ) == 0 )
   {  m_fi.init_date( COL_DOC_CRE_DATE, dt_val ) ; }
   /*----------------------------------------------------------------------*/
   if( st_iso_init_datetime( iso_pvd.tb_c_datetime_modification, dt_val
                           ) == 0
     )
   {  m_fi.init_date( COL_DOC_MOD_DATE, dt_val ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.prepare_string( iso_pvd.tb_c_application_use,
                            sizeof( iso_pvd.tb_c_application_use ), s_val
                          ) > 0
     )
   {  m_f.val_s( COL_DOC_INFO ) = s_val ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_iso()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "iso" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_iso( *this ).iso_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_iso.cpp                        !
+==========================================================================*/
