/*====================================+=====================================+
! File CFileInit_flv.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                   Video file: ".flv" (Adobe Flash)                        !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Only codecs and the information present in the metadata are extracted.    !
! So, some information like width, height, fps ... may not be loaded.       !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_flv_header ;
struct str_flv_tag_header ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_flv : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_flv( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_be_data( str_flv_header &fh ) ;
      int read_be_data( str_flv_tag_header &fth ) ;
      int read_metadata( bool boo_video, int i_tag_length ) ;
      int flv_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_flv_header
{
   /*----------------------------------------------------------------------*/
   char     tb_c_signature[ 3 ] ;
   wxUint8  b_version           ;
   wxUint8  b_flags             ; // :0 = Video, :2 = Audio => 5=Audio+Video
   wxUint32 dw_offset           ; // Should be: 9
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_flv_tag_header
{
   /*----------------------------------------------------------------------*/
   wxUint8  b_type               ;
   wxUint8  tb_b_bodylength[ 3 ] ;
   wxUint32 dw_timestamp         ;
   wxUint8  tb_b_streamid  [ 3 ] ;
   /*----------------------------------------------------------------------*/
/*
   Then comes the data body
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_flv::read_be_data( str_flv_header &fh )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( fh ), &fh ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   fh.dw_offset = wxUINT32_SWAP_ALWAYS( fh.dw_offset ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_flv::read_be_data( str_flv_tag_header &fth )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( fth ), &fth ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   fth.dw_timestamp = wxUINT32_SWAP_ALWAYS( fth.dw_timestamp ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The documentation says it should be "00 00 09" but in reality             !
! many "00 00 00" sequences can be seen.                                    !
+--------------------------------------------------------------------------*/
static bool st_flv_is_block_end( const wxUint8 *p_b )
{
   /*----------------------------------------------------------------------*/
   return(    p_b[ 0 ] == '\x00'
           && p_b[ 1 ] == '\x00'
           && ( p_b[ 2 ] == '\x00' || p_b[ 2 ] == '\x09' )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_flv_conv_tag_col( bool boo_video, const wxString &s_var )
{
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "duration" ) == 0 )
   {  return( boo_video ? COL_VIDEO_DURATION : COL_AUDIO_DURATION ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "width" ) == 0 )
   {  return( boo_video ? COL_VIDEO_X : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "height" ) == 0 )
   {  return( boo_video ? COL_VIDEO_Y : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "framerate" ) == 0 )
   {  return( boo_video ? COL_VIDEO_FPS : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "audiosamplerate" ) == 0 )
   {  return( boo_video ? COL_VIDEO_SAMPRATE : COL_AUDIO_SAMPRATE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "creator" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ARTIST ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "creationdate" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_CRE_DATE : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "createdby" ) == 0
       || s_var.CmpNoCase( "metadatacreator" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_SOFTWARE : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--( If the pointer is NULL the "string" is simply skipped )--------------*/
static int st_flv_metadata_string( wxUint8  *&p_b_buffer,
                                   const wxUint8 * const p_b_buffer_end,
                                   wxString *p_s
                                 )
{
   /*----------------------------------------------------------------------*/
   int i_length ;
   /*----------------------------------------------------------------------*/
   if( p_b_buffer_end - p_b_buffer < 2 )
   {  return( -1 ) ; }

   /*--( A BE 16 integer contains the string length )----------------------*/
   i_length = wxUINT16_SWAP_ON_LE( *( wxUint16 * )p_b_buffer ) ;
   if( i_length <= 0 || i_length >= p_b_buffer_end - p_b_buffer )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   p_b_buffer += 2 ;
   if( p_s != NULL )
   {  *p_s = wxString::FromUTF8( ( char * )p_b_buffer, i_length ) ; }
   p_b_buffer += i_length ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! If the pointer is NULL the "val" is simply skipped.                       !
! This function is recursive.                                               !
+--------------------------------------------------------------------------*/
static int st_flv_metadata_val( wxUint8 *&p_b_buffer,
                                const wxUint8 * const p_b_buffer_end,
                                wxAny *p_val
                              )
{
   /*----------------------------------------------------------------------*/
   if( p_val != NULL ) { p_val->MakeNull() ; }

   /*--( The first info is the type )--------------------------------------*/
   switch( *p_b_buffer++ )
   {
      /*--( Double )-------------------------------------------------------*/
      case 0x00 :
         /*----------------------------------------------------------------*/
         if( p_b_buffer_end - p_b_buffer < 8 )
         {  return( -2 ) ; }
         /*----------------------------------------------------------------*/
         if( p_val != NULL )
         {  *p_val = sr::conv_be_do( *( wxUint64 * )p_b_buffer ) ; }
         /*----------------------------------------------------------------*/
         p_b_buffer += 8 ;
         /*----------------------------------------------------------------*/
         break ;

      /*--( Boolean )------------------------------------------------------*/
      case 0x01 :
         /*----------------------------------------------------------------*/
         if( p_b_buffer_end - p_b_buffer <= 1 )
         {  return( -3 ) ; }
         /*----------------------------------------------------------------*/
         ++p_b_buffer ;
         /*----------------------------------------------------------------*/
         if( p_val != NULL )
         {  *p_val = *p_b_buffer ; }
         /*----------------------------------------------------------------*/
         break ;

      /*--( String )-------------------------------------------------------*/
      case 0x02 :
      {
         /*----------------------------------------------------------------*/
         wxString s_val ;
         /*----------------------------------------------------------------*/
         if( st_flv_metadata_string( p_b_buffer, p_b_buffer_end, &s_val
                                   ) != 0
           )
         {  return( -4 ) ; }
         /*----------------------------------------------------------------*/
         if( p_val != NULL )
         {  *p_val = s_val ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--( Typed Object )-------------------------------------------------*/
      case 0x10 :
         /*--( One name )--------------------------------------------------*/
         if( st_flv_metadata_string( p_b_buffer, p_b_buffer_end, NULL ) != 0)
         {  return( -5 ) ; }
         /*--( No BREAK because the rest of a typed object is an Object )--*/

      /*--( Object: not used. It is "skipped" )----------------------------*/
      case 0x03 :
         /*----------------------------------------------------------------*/
         if( p_b_buffer_end - p_b_buffer < 3 )
         {  return( -6 ) ; }
         /*--( And another typed element ? )-------------------------------*/
         while(    p_b_buffer_end - p_b_buffer >= 3
                && !st_flv_is_block_end( p_b_buffer )
              )
         {  /*--( The name )-----------------------------------------------*/
            if( st_flv_metadata_string( p_b_buffer, p_b_buffer_end, NULL
                                      ) != 0
              )
            {  return( -7 ) ; }
            /*--( And the associated val )---------------------------------*/
            if( st_flv_metadata_val( p_b_buffer, p_b_buffer_end, NULL ) != 0)
            {  return( -8 ) ; }
            /*-------------------------------------------------------------*/
         }

         /*--( Then jump over the constant )-------------------------------*/
         p_b_buffer += 3 ;
         /*----------------------------------------------------------------*/
         break ;

      /*--( Undefined )----------------------------------------------------*/
      case 0x05 :
      case 0x06 :
         break ;

      /*--( Array: not used. It is "skipped" )-----------------------------*/
      case 0x0A :
      {
         /*----------------------------------------------------------------*/
         if( p_b_buffer_end - p_b_buffer <= 10 )
         {  return( -9 ) ; }

         /*--( Then the number of elements in the array )------------------*/
         long l_length = wxUINT32_SWAP_ON_LE( *( wxUint32 * )p_b_buffer ) ;
         if( l_length < 0 ) { return( -10 ) ; }
         p_b_buffer += 4 ;

         /*--( Then the data, they have to be scanned one by one ... )-----*/
         for( ; l_length > 0 ; --l_length )
         {  /*-------------------------------------------------------------*/
            if( st_flv_metadata_val( p_b_buffer, p_b_buffer_end, NULL ) != 0)
            {  return( -11 ) ; }
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--( Date, "skipped" )----------------------------------------------*/
      case 0x0B :
         /*----------------------------------------------------------------*/
         if( p_b_buffer_end - p_b_buffer < 10 )
         {  return( -12 ) ; }
         /*--( double time_t and an int for the timezone )-----------------*/
         p_b_buffer += 10 ;
         /*----------------------------------------------------------------*/
         break ;

      /*--( Unmanaged type )-----------------------------------------------*/
      default:
         return( -13 ) ;

      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_flv::read_metadata( bool boo_video, int i_tag_length )
{
   /*----------------------------------------------------------------------*/
   if( i_tag_length < 0 || i_tag_length >= 500 * 1024 ) { return( -1 ) ; }

   /*--( The work is done in memory )--------------------------------------*/
   wxMemoryBuffer mb_buffer       ;
   wxUint8        *p_b_buffer     ;
   wxUint8        *p_b_buffer_end ;
   wxUint8        *p_b_block_end  ;
   wxString       s_var           ;
   wxString       s_val           ;
   wxAny          any_val         ;
   int            i_col           ;
   int            i_width  = -1   ;
   int            i_height = -1   ;

   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( i_tag_length, mb_buffer ) != 0 )
   {  return( -2 ) ; }
   /*-----------------------------------------------------------------------+
   ! After reading, check minimum size expected: no error if less           !
   +-----------------------------------------------------------------------*/
   if( i_tag_length < 30 ) { return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   p_b_buffer = ( wxUint8 * )mb_buffer.GetData() ;
   p_b_buffer_end = p_b_buffer + i_tag_length ;
   p_b_block_end  = p_b_buffer_end - 3 ;

   /*-----------------------------------------------------------------------+
   ! The "metadata" is a "MixedArray" (0x08)                                !
   ! with a "special" ending                                                !
   +-----------------------------------------------------------------------*/
   if( !st_flv_is_block_end( p_b_block_end ) )
   {  return( -3 ) ; }
   /*--( Not finding the expected metadata is not an error )---------------*/
   if( memcmp( p_b_buffer, "\x02\x00\x0AonMetaData\x08", 14 ) != 0 )
   {  return( 0 ) ; }
   p_b_buffer += 14 ;

   /*-----------------------------------------------------------------------+
   ! The number of tags is supposed to follow but many files have "0" there !
   +-----------------------------------------------------------------------*/
   p_b_buffer += 4 ;

   /*--( Then comes the "couples" name/value )-----------------------------*/
   while( p_b_buffer < p_b_block_end )
   {
      /*--( Read the couple: var/val )-------------------------------------*/
      if( st_flv_metadata_string( p_b_buffer, p_b_buffer_end, &s_var ) != 0 )
      {  return( -5 ) ; }
      if( st_flv_metadata_val( p_b_buffer, p_b_buffer_end, &any_val ) != 0 )
      {  return( -6 ) ; }

      /*--( Something found ? May happen that a name without value exists )*/
      if( any_val.IsNull() ) { continue ; }

      /*--( Match a column ? )---------------------------------------------*/
      i_col = st_flv_conv_tag_col( boo_video, s_var ) ;

      /*--( Adapt the data to the type )-----------------------------------*/
      switch( i_col )
      {
         /*----------------------------------------------------------------*/
         case COL_AUDIO_DURATION :
            m_fi.init_audio_duration( ( int )any_val.As< double >() ) ;
            break ;
         /*----------------------------------------------------------------*/
         case COL_VIDEO_DURATION :
            m_fi.init_video_duration( ( int )any_val.As< double >() ) ;
            break ;
         /*----------------------------------------------------------------*/
         case COL_AUDIO_SAMPRATE :
            m_fi.init_audio_samprate( ( int )any_val.As< double >() ) ;
            break ;
         /*----------------------------------------------------------------*/
         case COL_VIDEO_SAMPRATE :
            m_fi.init_video_samprate( ( int )any_val.As< double >() ) ;
            break ;
         /*----------------------------------------------------------------*/
         case COL_VIDEO_X : i_width  = ( int )any_val.As< double >() ; break;
         case COL_VIDEO_Y : i_height = ( int )any_val.As< double >() ; break;
         /*----------------------------------------------------------------*/
         case COL_VIDEO_FPS :
            m_fi.init_video_fps( any_val.As< double >() ) ;
            break ;
         /*----------------------------------------------------------------*/
         case COL_NB : break ;
         /*----------------------------------------------------------------*/
         default :
         {
            /*-------------------------------------------------------------*/
            wxString s_val = any_val.As< wxString >() ;
            /*-------------------------------------------------------------*/
            if( m_fi.reserve_col( i_col ) )
            {  /*----------------------------------------------------------*/
               if( m_fi.prepare_string( s_val ) > 0 )
               {  m_f.val_s( i_col ) = s_val ; }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Init the video size if the two dimensions are known )-------------*/
   if( i_width > 0 && i_height > 0 )
   {  m_fi.init_video_x_y( i_width, i_height ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_flv::flv_read()
{
   /*----------------------------------------------------------------------*/
   str_flv_header     file_header            ;
   str_flv_tag_header tag_header             ;
   wxUint32           dw_previous_tag_size   ;
   bool               boo_has_video          ;
   bool               boo_video_done = false ;
   bool               boo_has_audio          ;
   bool               boo_audio_done = false ;
   wxFileOffset       fo_size_to_skip        ;
   wxUint8            b_val                  ;
   wxString           s_audio_info           ;
   wxString           s_video_info           ;
   int                i_num_tag              ;

   /*----------------------------------------------------------------------*/
   if( read_be_data( file_header ) != 0 )
   {  return( -1 ) ; }

   /*--( Minimal control )-------------------------------------------------*/
   if( memcmp( file_header.tb_c_signature, "FLV", 3 ) != 0 )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   if( file_header.dw_offset != 9 )
   {  return( -3 ) ; }

   /*--( Consider as "done" if not present )-------------------------------*/
   boo_has_audio  = ( ( file_header.b_flags & 0x04 ) != 0 ) ;
   boo_audio_done = !boo_has_audio ;
   boo_has_video  = ( ( file_header.b_flags & 0x01 ) != 0 ) ;
   boo_video_done = !boo_has_video ;

   /*-----------------------------------------------------------------------+
   ! Fix a limit to the number of tags to scan. Relying on the presence of  !
   ! one or another is not a good idea. Many flv look strangely formed.     !
   +-----------------------------------------------------------------------*/
   for( i_num_tag = 0 ; i_num_tag < 10 ; ++i_num_tag )
   {
      /*--------------------------------------------------------------------+
      ! Here comes the previous tag size. Of course, on the first one it    !
      ! must be 0                                                           !
      +--------------------------------------------------------------------*/
      if(    m_fa.read_be_data( dw_previous_tag_size ) != 0
          || ( i_num_tag == 0 && dw_previous_tag_size != 0 )
        )
      {  return( -4 ) ; }

      /*-------------------------------------------------------------------*/
      if( read_be_data( tag_header ) != 0 )
      {  return( -5 ) ; }

      /*--( Info are stored "big endian" )---------------------------------*/
      fo_size_to_skip = SR_THREECC( tag_header.tb_b_bodylength[ 2 ],
                                    tag_header.tb_b_bodylength[ 1 ],
                                    tag_header.tb_b_bodylength[ 0 ]
                                  ) ;

      /*--( What is this tag ? )-------------------------------------------*/
      if( fo_size_to_skip > 0 )
      {
         /*----------------------------------------------------------------*/
         switch( tag_header.b_type )
         {
            /*--------------------------------------------------------------+
            ! AUDIO                                                         !
            +--------------------------------------------------------------*/
            case 0x08 :
            {
               /*----------------------------------------------------------*/
               int        i_channel  ;
               int        i_samprate ;
               const char *p_c_info  ;
               /*----------------------------------------------------------*/
               if( boo_audio_done ) { break ; }
               boo_audio_done = true ;
               /*----------------------------------------------------------*/
               if( m_fa.read_data( b_val ) != 0 )
               {  return( -6 ) ; }
               /*--( One byte less to skip )-------------------------------*/
               --fo_size_to_skip ;

               /*----------------------------------------------------------*/
               i_channel = ( b_val & 0x01 ) + 1 ;

               /*--( Sampling rate )---------------------------------------*/
               switch( ( b_val & 0x0C ) >> 2 )
               {  /*-------------------------------------------------------*/
                  case 0 : i_samprate =  5512 ; break ;
                  case 1 : i_samprate = 11025 ; break ;
                  case 2 : i_samprate = 22050 ; break ;
                  case 3 : i_samprate = 44100 ; break ;
                  /*-------------------------------------------------------*/
                  default : return( -7 ) ;
                  /*-------------------------------------------------------*/
               }
               /*--( Codec )-----------------------------------------------*/
               switch( ( b_val & 0xF0 ) >> 4 )
               {  /*-------------------------------------------------------*/
                  case  0 : p_c_info = "Linear PCM"            ; break ;
                  case  1 : p_c_info = "ADPCM"                 ; break ;
                  case  2 : p_c_info = "Mpeg L3"               ; break ;
                  case  3 : p_c_info = "Linear PCM LE"         ; break ;
                  case  4 : p_c_info = "Nellymoser 16kHz mono" ; break ;
                  case  5 : p_c_info = "Nellymoser 8kHz mono"  ; break ;
                  case  6 : p_c_info = "Nellymoser"            ; break ;
                  case  7 : p_c_info = "G.711 A-law PCM"       ; break ;
                  case  8 : p_c_info = "G.711 mu-law PCM"      ; break ;
                  case 10 : p_c_info = "AAC"                   ; break ;
                  case 11 : p_c_info = "Speex"                 ; break ;
                  case 14 : p_c_info = "Mpeg L3 8-Khz"         ; break ;
                  case 15 : p_c_info = "Device specific sound" ; break ;
                  default: p_c_info = NULL ; break ;
                  /*-------------------------------------------------------*/
               }

               /*----------------------------------------------------------*/
               if( boo_has_video )
               {  /*-------------------------------------------------------*/
                  m_fi.init_video_channel( i_channel ) ;
                  m_fi.init_video_samprate( i_samprate ) ;
                  /*-------------------------------------------------------*/
               }
               else
               {  /*-------------------------------------------------------*/
                  m_fi.init_audio_channel( i_channel ) ;
                  m_fi.init_audio_samprate( i_samprate ) ;
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
               if( p_c_info != NULL )
               {  m_fi.add_codec_info( p_c_info, s_audio_info ) ; }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }

            /*--------------------------------------------------------------+
            ! VIDEO                                                         !
            +--------------------------------------------------------------*/
            case 0x09 :
            {
               /*----------------------------------------------------------*/
               const char *p_c_info ;
               /*----------------------------------------------------------*/
               if( boo_video_done ) { break ; }
               boo_video_done = true ;
               /*----------------------------------------------------------*/
               if( m_fa.read_data( b_val ) != 0 )
               {  return( -7 ) ; }
               /*--( One byte less to skip )-------------------------------*/
               --fo_size_to_skip ;

               /*--( Video codec )-----------------------------------------*/
               switch( b_val & 0x0F )
               {  /*-------------------------------------------------------*/
                  case 2 : p_c_info = "Sorenson H263" ; break ;
                  case 3 : p_c_info = "Screen video"  ; break ;
                  case 4 : p_c_info = "On2 VP6"       ; break ;
                  case 5 : p_c_info = "On2 VP6 Alpha" ; break ;
                  case 6 : p_c_info = "ScreenVideo 2" ; break ;
                  case 7 : p_c_info = "H.264"         ; break ; // AVC
                  default: p_c_info = NULL ; break ;
                  /*-------------------------------------------------------*/
               }

               /*----------------------------------------------------------*/
               if( p_c_info != NULL )
               {  m_fi.add_codec_info( p_c_info, s_video_info ) ; }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }

            /*--------------------------------------------------------------+
            ! METADATA                                                      !
            +--------------------------------------------------------------*/
            case 0x12 :
               /*-----------------------------------------------------------+
               ! No check to see if this tag has been already "done" to     !
               ! collect as much info as possible.                          !
               +-----------------------------------------------------------*/
               if( read_metadata( boo_has_video, fo_size_to_skip ) != 0 )
               {  return( -8 ) ; }
               /*--( All the tag has been "read" )-------------------------*/
               fo_size_to_skip = 0 ;
               /*----------------------------------------------------------*/
               break ;

            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }

      /*--( Jump over the rest of the body )-------------------------------*/
      if( fo_size_to_skip > 0 && m_fa.skip_nb_byte( fo_size_to_skip ) != 0 )
      {  return( -9 ) ; }

      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( !s_audio_info.empty() || !s_video_info.empty() )
   {
      /*-------------------------------------------------------------------*/
        m_f.val_s( boo_has_video ? COL_VIDEO_INFO : COL_AUDIO_INFO )
      = m_fi.concat_av_codecs( s_audio_info, s_video_info ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_flv()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "flv" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_flv( *this ).flv_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_flv.cpp                       !
+==========================================================================*/
