/*====================================+=====================================+
! File CFileInit_desktop.cpp          ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!        "desktop" files: ".desktop" freedesktop.org "desktop entry"        !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! File syntax is like the ".ini" ones (config file)                         !
! Storage column names may not correspond to data information names         !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/fileconf.h>
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
#include "CApp.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_desktop : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_desktop( CFileInit &parent, const wxString &s_path ) ;
   /*----------------------------------------------------------------------*/
   public :
      /*--( The vector content is independent of the file treated )--------*/
      static sr::t_vec_str st_m_vec_suffix ;
      /*-------------------------------------------------------------------*/
      wxFileConfig m_fc ;
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      static void st_init_vec_suffix() ;
      int read_best_val( const wxString &s_name, wxString &s_val ) ;
      int read_val( const wxString &s_name, int i_col ) ;
   /*----------------------------------------------------------------------*/
} ;



/*-------------------------------------------------------------------------*/
CFileInit_desktop::CFileInit_desktop( CFileInit &parent,
                                      const wxString &s_path
                                    )
                 : CFileInit_type_base( parent ),
                   m_fc( wxEmptyString, wxEmptyString, s_path,
                         wxEmptyString,
                           wxCONFIG_USE_LOCAL_FILE
                         | wxCONFIG_USE_RELATIVE_PATH
                         | wxCONFIG_USE_NO_ESCAPE_CHARACTERS
                       )
{  ; }

/*-------------------------------------------------------------------------*/
sr::t_vec_str CFileInit_desktop::st_m_vec_suffix ;

/*-------------------------------------------------------------------------*/
void CFileInit_desktop::st_init_vec_suffix()
{
   /*--( Initialization only done once )-----------------------------------*/
   if( !st_m_vec_suffix.empty() )
   {  return ; }

   /*----------------------------------------------------------------------*/
   wxString s_language ;
   wxString s_country  ;
   wxString s_encoding ;

   /*----------------------------------------------------------------------*/
     s_language
   = wxGetApp().get_system_canonical_name().BeforeFirst( '_', &s_country ) ;
   s_encoding = wxLocale::GetSystemEncodingName() ;

   /*-----------------------------------------------------------------------+
   ! The suffixes that can be added to names.                               !
   ! Based on freedesktop specification:                                    !
   ! http://standards.freedesktop.org/desktop-entry-spec/latest/ar01s04.html!
   ! A name can have the form:                                              !
   ! name[lang_COUNTRY.ENCODING@MODIFIER]                                   !
   !                                                                        !
   ! LC_MESSAGES value   Possible keys in order of matching                 !
   ! -----------------   ----------------------------------                 !
   ! lang_COUNTRY        lang_COUNTRY, lang, default value                  !
   !                                                                        !
   ! The language used is the system one and not what is currently defined  !
   ! in Siren's option. This looks better based on the fact that the files  !
   ! content is set based on system's locale ...                            !
   +-----------------------------------------------------------------------*/
   if( !s_language.empty() )
   {
      /*-------------------------------------------------------------------*/
      if( !s_country.empty() )
      {
         /*--( Based on tests this one is used too )-----------------------*/
         if( !s_encoding.empty() )
         {  st_m_vec_suffix.push_back(   '[' + s_language
                                       + '_' + s_country
                                       + '.' + s_encoding
                                       + ']'
                                     ) ;
         }
         /*----------------------------------------------------------------*/
         st_m_vec_suffix.push_back( '['+ s_language + '_' + s_country +']' );
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      st_m_vec_suffix.push_back( '[' + s_language + ']' ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_desktop::read_best_val( const wxString &s_name,
                                      wxString &s_val
                                    )
{
  /*-----------------------------------------------------------------------*/
  size_t sz_num ;
  /*-----------------------------------------------------------------------*/
  for( sz_num = 0 ;
          sz_num < st_m_vec_suffix.size()
       && !m_fc.Read( s_name + st_m_vec_suffix[ sz_num ], &s_val ) ;
       ++sz_num
     )
  {  ; }
  /*--( Found ? )----------------------------------------------------------*/
  if( sz_num < st_m_vec_suffix.size() )
  {  return( 0 ) ; }
  /*--( Finally try with the default value (no suffix) )-------------------*/
  return( m_fc.Read( s_name, &s_val ) ? 0 : -1 ) ;
  /*-----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_desktop::read_val( const wxString &s_name, int i_col )
{
   /*----------------------------------------------------------------------*/
   wxString s_val ;

   /*--( If nothing has been found yet then quit )-------------------------*/
   if( read_best_val( s_name, s_val ) != 0 ) { return( -1 ) ; }

   /*--( Some content is "translated" )------------------------------------*/
   if( s_name.CmpNoCase( "Terminal" ) == 0 )
   {  /*-------------------------------------------------------------------*/
      if( s_val.CmpNoCase( "true" ) == 0 )
      {  s_val = "Terminal" ; }
      else
      {  s_val.clear() ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( m_fi.reserve_col( i_col ) && m_fi.prepare_string( s_val ) > 0 )
   {  m_f.val_s( i_col ) = s_val ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_desktop()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "desktop" ;
   /*-----------------------------------------------------------------------+
   ! Check the size: these files have to be not too large                   !
   +-----------------------------------------------------------------------*/
   if( m_f.get_size() > 50 * 1024 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   CFileInit_desktop::st_init_vec_suffix() ;
   /*----------------------------------------------------------------------*/
   CFileInit_desktop desktop( *this, m_f.get_full_path() ) ;

   /*--( The path that should be used is known )---------------------------*/
   const wxString co_s_path( "/Desktop Entry" ) ;
   desktop.m_fc.SetPath( co_s_path ) ;
   if( desktop.m_fc.GetPath() != co_s_path )
   {  return( -2 ) ; }

   /*--( Type doesn't seem interesting )-----------------------------------*/
   // desktop.read_val( "Type"       , COL_DOC_TYPE        ) ;
   /*----------------------------------------------------------------------*/
   desktop.read_val( "Name"       , COL_DOC_TITLE       ) ;
   desktop.read_val( "GenericName", COL_DOC_SUBJECT     ) ;
   desktop.read_val( "Exec"       , COL_DOC_APPLICATION ) ;
   desktop.read_val( "Terminal"   , COL_DOC_INFO        ) ;
   desktop.read_val( "Version"    , COL_DOC_VERSION     ) ;
   desktop.read_val( "Comment"    , COL_DOC_COMMENT     ) ;
   desktop.read_val( "Categories" , COL_DOC_KEYWORDS    ) ;
   desktop.read_val( "URL"        , COL_DOC_URL         ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                    End of file CFileInit_desktop.cpp                      !
+==========================================================================*/
