/*====================================+=====================================+
! File CFileInit_avi.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                          Video file: ".avi"                               !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! To avoid problems, generally in this process the end block address is     !
! saved and a jump to it is done after having read the subparts of this     !
! same block.                                                               !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_MainAVIHeader
{
   /*----------------------------------------------------------------------*/
   wxUint32 dwMicroSecPerFrame    ; // frame display rate (or 0L)
   wxUint32 dwMaxBytesPerSec      ; // max. transfer rate
   wxUint32 dwPaddingGranularity  ; // pad to multiples of this
                                    // size; normally 2K.
   wxUint32 dwFlags               ; // the ever-present flags
   wxUint32 dwTotalFrames         ; // # frames in file
   wxUint32 dwInitialFrames       ;
   wxUint32 dwStreams             ;
   wxUint32 dwSuggestedBufferSize ;

   wxUint32 dwWidth               ;
   wxUint32 dwHeight              ;
   wxUint32 dwReserved[ 4 ]       ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_AVIStreamHeader
{
   /*----------------------------------------------------------------------*/
   wxUint32 fccType               ;
   wxUint32 fccHandler            ;
   wxUint32 dwFlags               ; // Contains AVITF_* flags
   wxUint16 wPriority             ;
   wxUint16 wLanguage             ;
   wxUint32 dwInitialFrames       ;
   wxUint32 dwScale               ;
   wxUint32 dwRate                ; // dwRate / dwScale == samples/second
   wxUint32 dwStart               ;
   wxUint32 dwLength              ; // In units above...
   wxUint32 dwSuggestedBufferSize ;
   wxUint32 dwQuality             ;
   wxUint32 dwSampleSize          ;
   wxUint32 tb_dw_rect[ 4 ]       ; // 4 LONGs as defined in MSW
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! DV Information                                                            !
+--------------------------------------------------------------------------*/
typedef wxUint8 t_dv_vaux_pack[ 5 ] ;

/*-------------------------------------------------------------------------*/
struct str_dv_ssyb
{
   /*----------------------------------------------------------------------*/
   wxUint16       w_id    ;
   wxUint8        b_if_ff ;
   t_dv_vaux_pack pc      ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_dv_subcode_block
{
   /*----------------------------------------------------------------------*/
   wxUint8     tb_b_block_id[  3 ] ;
   str_dv_ssyb tb_ssyb      [  6 ] ;
   wxUint8     tb_b_dummy   [ 29 ] ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_dv_vaux_block
{
   /*----------------------------------------------------------------------*/
   wxUint8        tb_b_block_id[  3 ] ;
   t_dv_vaux_pack tb_b_pc      [ 15 ] ;
   wxUint8        tb_b_dummy   [  2 ] ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_dv_header
{
   /*----------------------------------------------------------------------*/
   wxUint8              tb_b_dif_header [ 80 ] ;
   str_dv_subcode_block tb_subcode_block[  2 ] ;
   str_dv_vaux_block    tb_vaux_block   [  3 ] ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*--------------------------------------------------------------------------+
! Data/Treatments used during the loading                                   !
+--------------------------------------------------------------------------*/
class CFileInit_avi : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_avi( CFileInit &parent )
                   : CFileInit_type_base( parent ),
                     m_dw_fcc( 0 ), m_dw_compress( 0 ), m_wFormatTag( 0 ),
                     m_boo_dv( false ), m_boo_aud_vid_interleaved( false ),
                     m_do_fps( 0 ), m_dw_nb_frames( 0 ), m_boo_end( false )
      {  ; }
   /*----------------------------------------------------------------------*/
   public :
      /*--( FourCCs ... )--------------------------------------------------*/
      wxUint32  m_dw_fcc                  ;
      wxUint32  m_dw_compress             ;
      wxUint32  m_wFormatTag              ;
      /*-------------------------------------------------------------------*/
      bool      m_boo_dv                  ;
      bool      m_boo_aud_vid_interleaved ; // Interleaved = 1 else 2
      double    m_do_fps                  ;
      wxUint32  m_dw_nb_frames            ;
      bool      m_boo_end                 ;
      /*-------------------------------------------------------------------*/
      wxString  m_s_audio_info            ;
      wxString  m_s_video_info            ;
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int  read_le_data( str_MainAVIHeader &mah ) ;
      int  read_le_data( str_AVIStreamHeader &ash,
                         size_t sz_size = sizeof( str_AVIStreamHeader )
                       ) ;
      int  read_one_info( wxUint32 dw_id, wxUint32 &dw_size ) ;
      void info_audio( wxString &s_info ) ;
      int  info_video( wxString &s_info ) ;
      int  read_strl(  wxUint32 &dw_size_list ) ;
      int  skip_odml(  wxUint32 &dw_size_list ) ;
      int  read_info_dv( wxUint32 &dw_size_list ) ;
      int  read_hdrl( wxUint32 &dw_size_list ) ;
      int  read_list() ;
      int  avi_read() ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
int CFileInit_avi::read_le_data( str_MainAVIHeader &mah )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( mah ), &mah ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   /*----------------------------------------------------------------------*/
   mah.dwMicroSecPerFrame = wxUINT32_SWAP_ALWAYS( mah.dwMicroSecPerFrame ) ;
   mah.dwMaxBytesPerSec   = wxUINT32_SWAP_ALWAYS( mah.dwMaxBytesPerSec   ) ;
     mah.dwPaddingGranularity
   = wxUINT32_SWAP_ALWAYS( mah.dwPaddingGranularity ) ;
   mah.dwFlags            = wxUINT32_SWAP_ALWAYS( mah.dwFlags            ) ;
   mah.dwTotalFrames      = wxUINT32_SWAP_ALWAYS( mah.dwTotalFrames      ) ;
   mah.dwInitialFrames    = wxUINT32_SWAP_ALWAYS( mah.dwInitialFrames    ) ;
   mah.dwStreams          = wxUINT32_SWAP_ALWAYS( mah.dwStreams          ) ;
     mah.dwSuggestedBufferSize
   = wxUINT32_SWAP_ALWAYS( mah.dwSuggestedBufferSize ) ;
   mah.dwWidth            = wxUINT32_SWAP_ALWAYS( mah.dwWidth            ) ;
   mah.dwHeight           = wxUINT32_SWAP_ALWAYS( mah.dwHeight           ) ;
   // wxUint32 dwReserved[ 4 ]       ;
   /*----------------------------------------------------------------------*/
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( In some cases only the beginning of the structure is needed )--------*/
int CFileInit_avi::read_le_data( str_AVIStreamHeader &ash, size_t sz_size )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sz_size, &ash ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
// ash.fccType         = wxUINT32_SWAP_ALWAYS( ash.fccType         ) ;
// ash.fccHandler      = wxUINT32_SWAP_ALWAYS( ash.fccHandler      ) ;
   ash.dwFlags         = wxUINT32_SWAP_ALWAYS( ash.dwFlags         ) ;
   ash.wPriority       = wxUINT16_SWAP_ALWAYS( ash.wPriority       ) ;
   ash.wLanguage       = wxUINT16_SWAP_ALWAYS( ash.wLanguage       ) ;
   ash.dwInitialFrames = wxUINT32_SWAP_ALWAYS( ash.dwInitialFrames ) ;
   ash.dwScale         = wxUINT32_SWAP_ALWAYS( ash.dwScale         ) ;
   ash.dwRate          = wxUINT32_SWAP_ALWAYS( ash.dwRate          ) ;
   ash.dwStart         = wxUINT32_SWAP_ALWAYS( ash.dwStart         ) ;
   ash.dwLength        = wxUINT32_SWAP_ALWAYS( ash.dwLength        ) ;
     ash.dwSuggestedBufferSize
   = wxUINT32_SWAP_ALWAYS( ash.dwSuggestedBufferSize ) ;
   ash.dwQuality       = wxUINT32_SWAP_ALWAYS( ash.dwQuality       ) ;
   ash.dwSampleSize    = wxUINT32_SWAP_ALWAYS( ash.dwSampleSize    ) ;
   //wxUint32 tb_dw_rect[ 4 ]       ; // 4 LONGs as defined in MSW
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_avi_conv_tag_col( wxUint32 dw_tag )
{
   /*----------------------------------------------------------------------*/
   switch( dw_tag )
   {
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'I','N','A','M' ) : return( COL_VIDTAG_TITLE      ) ;
      case SR_FOURCC( 'I','S','B','J' ) : return( COL_VIDTAG_SUBJECT    ) ;
      case SR_FOURCC( 'I','A','R','T' ) : return( COL_VIDTAG_ARTIST     ) ;
      case SR_FOURCC( 'I','C','M','T' ) : return( COL_VIDTAG_COMMENT    ) ;
      case SR_FOURCC( 'I','K','E','Y' ) : return( COL_VIDTAG_KEYWORDS   ) ;
      case SR_FOURCC( 'I','E','N','G' ) : return( COL_VIDTAG_ENGINEER   ) ;
      case SR_FOURCC( 'I','T','C','H' ) : return( COL_VIDTAG_TECHNICIAN ) ;
      case SR_FOURCC( 'I','G','N','R' ) : return( COL_VIDTAG_GENRE      ) ;
      case SR_FOURCC( 'I','C','R','D' ) : return( COL_VIDTAG_CRE_DATE   ) ;
      case SR_FOURCC( 'I','S','R','C' ) : return( COL_VIDTAG_SOURCE     ) ;
      case SR_FOURCC( 'I','C','O','P' ) : return( COL_VIDTAG_COPYRIGHT  ) ;
      case SR_FOURCC( 'I','S','F','T' ) : return( COL_VIDTAG_SOFTWARE   ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static bool st_avi_id_valid( wxUint32 dw_id_avi )
{
   /*----------------------------------------------------------------------*/
   return(    isalpha( *( wxUint8 * )&dw_id_avi     )
           && isalpha( *( wxUint8 * )&dw_id_avi + 1 )
           && isalpha( *( wxUint8 * )&dw_id_avi + 2 )
           && isalpha( *( wxUint8 * )&dw_id_avi + 3 )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Tag stored "uniquely": not in a LIST )-------------------------------*/
int CFileInit_avi::read_one_info( wxUint32 dw_id, wxUint32 &dw_size )
{
   /*----------------------------------------------------------------------*/
   int i_col ;
   /*----------------------------------------------------------------------*/
   i_col = st_avi_conv_tag_col( dw_id ) ;

   /*--( Unknown or already extracted ? )----------------------------------*/
   if( m_fi.reserve_col( i_col ) )
   {  /*-------------------------------------------------------------------*/
      wxString s_val ;
      /*-------------------------------------------------------------------*/
      if( m_fi.file_read_tb_c( dw_size, s_val ) == 0 && !s_val.empty() )
      {  m_f.val_s( i_col ) = s_val ; }
      /*-------------------------------------------------------------------*/
      dw_size = 0 ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit_avi::info_audio( wxString &s_info )
{
   /*----------------------------------------------------------------------*/
   switch( m_wFormatTag )
   {
      /*-------------------------------------------------------------------*/
      case 0x0001 : s_info = "PCM"                  ; break ;
      case 0x0002 : s_info = "MS ADPCM"             ; break ;
      case 0x0006 : s_info = "ALAW"                 ; break ;
      case 0x0007 : s_info = "MULAW"                ; break ;
      case 0x0008 : s_info = "DTS"                  ; break ;
      case 0x0010 : s_info = "OKI ADPCM"            ; break ;
      case 0x0011 : s_info = "IMA ADPCM"            ; break ;
      case 0x0022 : s_info = "DSP Group TrueSpeech" ; break ;
      case 0x0031 : s_info = "GSM 6.10"             ; break ;
      case 0x0050 : s_info = "Mpeg L2"              ; break ;
      case 0x0055 : s_info = "Mpeg L3"              ; break ;
      case 0x0161 : s_info = "WMA"                  ; break ;
      case 0x0162 : s_info = "WMA 9 Pro"            ; break ;
      case 0x0163 : s_info = "WMA 9 Lossless"       ; break ;
      case 0x2000 : s_info = "AC3 DVM"              ; break ;
      case 0x2001 : s_info = "AC3 DTS"              ; break ;
      /*-------------------------------------------------------------------*/
      default :
         s_info.Printf( "Audio (0x%04X)", m_wFormatTag ) ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_avi::info_video( wxString &s_info )
{
   /*----------------------------------------------------------------------*/
   const char *p_c_fcc      = ( char * )&m_dw_fcc      ;
   const char *p_c_compress = ( char * )&m_dw_compress ;
   const char *p_c_code                                ;

   /*--( Which fourcc is going to be used ? )------------------------------*/
   if(    isprint( p_c_compress[ 0 ] )
       && isprint( p_c_compress[ 1 ] )
       && isprint( p_c_compress[ 2 ] )
       && isprint( p_c_compress[ 3 ] )
     )
   {  p_c_code = p_c_compress ; }
   else
   if(    isprint( p_c_fcc[ 0 ] )
       && isprint( p_c_fcc[ 1 ] )
       && isprint( p_c_fcc[ 2 ] )
       && isprint( p_c_fcc[ 3 ] )
     )
   {  p_c_code = p_c_fcc ; }
   else
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   m_boo_dv = m_fi.info_fcc_video( p_c_code, s_info ) ;
   /*----------------------------------------------------------------------*/
   if( m_boo_dv )
   {  /*-------------------------------------------------------------------*/
      s_info += wxString::Format( " Type %d",
                                  ( m_boo_aud_vid_interleaved ? 1 : 2 )
                                ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_avi::read_strl( wxUint32 &dw_size_list )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset         fo_offset_end                      ;
   str_chunk            chunk                              ;
   str_AVIStreamHeader  avi_sh                             ;
   str_WAVEFORMATEX     wfx                                ;
   const int co_i_min_wave_size
           = offsetof( str_WAVEFORMATEX, nAvgBytesPerSec ) ;
   str_BITMAPINFOHEADER bmp_ih                             ;
   wxString             s_info                             ;
   bool                 boo_strh_video                     ;

   /*--( The stream header follows )---------------------------------------*/
   if(    m_fa.read_le_data( chunk ) != 0
       || chunk.dw_id != SR_FOURCC( 's','t','r','h' )
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   SR_SET_0( avi_sh ) ;

   /*--( Problem on RECT definition ... so the given size is read )--------*/
   if(    chunk.dw_size > sizeof( avi_sh )
       || read_le_data( avi_sh, chunk.dw_size ) != 0
     )
   {  return( -2 ) ; }

   /*--( The size has already been read )----------------------------------*/
   dw_size_list -= sizeof( chunk ) + chunk.dw_size ;

   /*--( Info about video stream ? )---------------------------------------*/
     boo_strh_video
   = (    avi_sh.fccType == SR_FOURCC( 'v','i','d','s' )
       || avi_sh.fccType == SR_FOURCC( 'i','a','v','s' )
     ) ;

   /*--( Interleaved audio & video => DV Video type 1 or 2 )---------------*/
     m_boo_aud_vid_interleaved
   = ( avi_sh.fccType == SR_FOURCC( 'i','a','v','s' ) ) ;

   /*----------------------------------------------------------------------*/
   if( boo_strh_video && avi_sh.dwScale != 0 )
   {
      /*--( Suppose that all video are encoded using the same fps ... )----*/
      if( m_do_fps == 0 )
      {  m_do_fps = ( double )avi_sh.dwRate / avi_sh.dwScale ; }
      /*--( The number frames is "accumulated" )---------------------------*/
      m_dw_nb_frames += avi_sh.dwLength ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Some chunks can be empty that the reason of the ">=" )------------*/
   while( ( int )dw_size_list >= ( int )sizeof( chunk ) )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.read_le_data( chunk ) != 0 )
      {  return( -3 ) ; }

      /*--( Some chunks can be empty ... )---------------------------------*/
      dw_size_list -= sizeof( chunk ) ;
      if( dw_size_list <= 0 ) { continue ; }
      /*-------------------------------------------------------------------*/
      fo_offset_end = m_fa.get_offset() + chunk.dw_size ;

      /*--------------------------------------------------------------------+
      ! Quit if invalid ! To quit without error the size is set to 0        !
      +--------------------------------------------------------------------*/
      if( !st_avi_id_valid( chunk.dw_id ) )
      {  dw_size_list = 0 ; continue ; }

      /*--( Extract video info )-------------------------------------------*/
      if( boo_strh_video && chunk.dw_id == SR_FOURCC( 's','t','r','f' ) )
      {
         /*----------------------------------------------------------------*/
         if(    avi_sh.fccType == SR_FOURCC( 'v','i','d','s' )
             && chunk.dw_size >= sizeof( bmp_ih )
           )
         {  /*-------------------------------------------------------------*/
            if( m_fa.read_le_data( bmp_ih ) != 0 )
            {  return( -4 ) ; }
            /*-------------------------------------------------------------*/
            m_dw_compress = bmp_ih.biCompression ;
            /*-------------------------------------------------------------*/
         }
         else
         {  m_dw_compress = 0 ; }
         /*----------------------------------------------------------------*/
         m_dw_fcc = avi_sh.fccHandler ;
         info_video( s_info ) ;
         /*----------------------------------------------------------------*/
      }
      else /*--( Extract audio info )--------------------------------------*/
      if(    avi_sh.fccType == SR_FOURCC( 'a','u','d','s' )
          && chunk.dw_id    == SR_FOURCC( 's','t','r','f' )
        )
      {  /*----------------------------------------------------------------*/
         if( ( int )chunk.dw_size >= co_i_min_wave_size )
         {  /*-------------------------------------------------------------*/
            if( m_fa.read_le_data( wfx, co_i_min_wave_size ) != 0 )
            {  return( -5 ) ; }
            /*-------------------------------------------------------------*/
            m_wFormatTag = wfx.wFormatTag ;
            /*-------------------------------------------------------------*/
            info_audio( s_info ) ;
            /*-------------------------------------------------------------*/
            m_fi.init_video_samprate( wfx.nSamplesPerSec ) ;
            m_fi.init_video_channel( wfx.nChannels ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else /*--( Data to skip )--------------------------------------------*/
      if( chunk.dw_id == SR_FOURCC( 'J','U','N','K' ) )
      {
         /*----------------------------------------------------------------*/
         if( m_fa.skip_nb_byte( chunk.dw_size ) != 0 )
         {  return( -6 ) ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( chunk.dw_id == SR_FOURCC( 'L','I','S','T' ) )
      {
         /*--( Size already read therefore a skip back is done )-----------*/
         if(    m_fa.skip_nb_byte( - ( int )sizeof( chunk.dw_size ) ) != 0
             || read_list() != 0
           )
         {  return( -7 ) ; }
         /*----------------------------------------------------------------*/
      }
      else /*--( GSPOT 2 problem )-----------------------------------------*/
      if( read_one_info( chunk.dw_id, chunk.dw_size ) != 0 )
      {  return( -8 ) ; }

      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_end ) != 0 )
      {  return( -9 ) ; }

      /*-------------------------------------------------------------------*/
      dw_size_list -= chunk.dw_size ;
      /*-------------------------------------------------------------------*/
   }

   /*--( The "info" will contain the "strn" name and if none, the handler )*/
   if( !boo_strh_video )
   {  m_fi.add_codec_info( s_info, m_s_audio_info ) ; }
   else
   {  m_fi.add_codec_info( s_info, m_s_video_info ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! For the multi RIFF AVIs, odml contains dmlh containing the total number   !
! of frames. This information is not useful here.                           !
+--------------------------------------------------------------------------*/
int CFileInit_avi::skip_odml( wxUint32 &dw_size_list )
{
   /*----------------------------------------------------------------------*/
   str_chunk chunk ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( chunk ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   dw_size_list -= sizeof( chunk ) + chunk.dw_size ;
   /*----------------------------------------------------------------------*/
   if( m_fa.skip_nb_byte( chunk.dw_size ) != 0 )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_avi::read_info_dv( wxUint32 &dw_size_list )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset  fo_offset_end                ;
   str_chunk     chunk                        ;
   str_dv_header dv_header                    ;
   bool          boo_frame_data_found = false ;
   wxDateTime    dt                           ;

   /*--( Looking for a data chunk )----------------------------------------*/
   while(    ( int )dw_size_list > ( int )sizeof( chunk )
          && !boo_frame_data_found
        )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.read_le_data( chunk ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      dw_size_list  -= sizeof( chunk ) + chunk.dw_size ;
      fo_offset_end  = m_fa.get_offset() + chunk.dw_size ;
      /*-------------------------------------------------------------------*/
        boo_frame_data_found
      = (    isdigit( *( ( char * )&chunk.dw_id + 0 ) )
          && isdigit( *( ( char * )&chunk.dw_id + 1 ) )
          && isprint( *( ( char * )&chunk.dw_id + 2 ) )
          && isprint( *( ( char * )&chunk.dw_id + 3 ) )
             /*--( Audio stream )------------------------------------------*/
          && (    *( ( char * )&chunk.dw_id + 2 ) != 'w'
               || *( ( char * )&chunk.dw_id + 3 ) != 'b'
             )
             /*--( Palette change )----------------------------------------*/
          && (    *( ( char * )&chunk.dw_id + 2 ) != 'p'
               || *( ( char * )&chunk.dw_id + 3 ) != 'c'
             )
        ) ;
      /*--( Read next )----------------------------------------------------*/
      if( !boo_frame_data_found && m_fa.set_offset( fo_offset_end ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! The interesting info are at the beginning of a frame                   !
   +-----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( dv_header ), &dv_header ) != 0 )
   {  return( -3 ) ; }
   dw_size_list -= sizeof( dv_header ) ;

   /*--( Minimal test )----------------------------------------------------*/
   if(    dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 11 ][ 0 ] != 'b'
       || dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 12 ][ 0 ] != 'c'
     )
   {  return( -4 ) ; }

   /*--( No need to swap, these are bytes )--------------------------------*/
   int i_second = dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 12 ][ 2 ] ;
   int i_minute = dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 12 ][ 3 ] ;
   int i_hour   = dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 12 ][ 4 ] ;
   int i_day    = dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 11 ][ 2 ] ;
   int i_month  = dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 11 ][ 3 ] ;
   int i_year   = dv_header.tb_vaux_block[ 2 ].tb_b_pc[ 11 ][ 4 ] ;
   /*----------------------------------------------------------------------*/
   i_second = ( i_second & 0x0F ) + ( ( i_second >> 4 ) & 0x07 ) * 10 ;
   i_minute = ( i_minute & 0x0F ) + ( ( i_minute >> 4 ) & 0x07 ) * 10 ;
   i_hour   = ( i_hour   & 0x0F ) + ( ( i_hour   >> 4 ) & 0x03 ) * 10 ;
   i_day    = ( i_day    & 0x0F ) + ( ( i_day    >> 4 ) & 0x03 ) * 10 ;
   i_month  = ( i_month  & 0x0F ) + ( ( i_month  >> 4 ) & 0x01 ) * 10 ;
   i_year   = ( i_year   & 0x0F ) + ( ( i_year   >> 4 ) & 0x0F ) * 10 ;
   /*----------------------------------------------------------------------*/
   i_year += ( i_year < 25 ? 2000 : 1900 ) ;

   /*----------------------------------------------------------------------*/
   if( sr::set_datetime( i_year, i_month , i_day,
                         i_hour, i_minute, i_second, 0, dt
                       ) != 0
     )
   {  return( -5 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fi.reserve_col( COL_VIDTAG_DV_DATE ) )
   {  m_fi.init_date( COL_VIDTAG_DV_DATE, dt ) ; }

   /*--( No need to swap, these are bytes )--------------------------------*/
   int i_tc_second = dv_header.tb_subcode_block[ 0 ].tb_ssyb[ 0 ].pc[ 2 ] ;
   int i_tc_minute = dv_header.tb_subcode_block[ 0 ].tb_ssyb[ 0 ].pc[ 3 ] ;
   int i_tc_hour   = dv_header.tb_subcode_block[ 0 ].tb_ssyb[ 0 ].pc[ 4 ] ;
   int i_tc_frame  = dv_header.tb_subcode_block[ 0 ].tb_ssyb[ 0 ].pc[ 1 ] ;
   /*----------------------------------------------------------------------*/
   i_tc_second = ( i_tc_second & 0x0F) + (( i_tc_second >> 4 ) & 0x07 ) * 10;
   i_tc_minute = ( i_tc_minute & 0x0F) + (( i_tc_minute >> 4 ) & 0x07 ) * 10;
   i_tc_hour   = ( i_tc_hour   & 0x0F) + (( i_tc_hour   >> 4 ) & 0x03 ) * 10;
   i_tc_frame  = ( i_tc_frame  & 0x0F) + (( i_tc_frame  >> 4 ) & 0x03 ) * 10;
   /*----------------------------------------------------------------------*/
   if( m_fi.reserve_col( COL_VIDTAG_DV_TIMECODE ) )
   {  m_f.val_s( COL_VIDTAG_DV_TIMECODE ).Printf(
                              "%02d.%02d.%02d.%02d",
                              i_tc_hour, i_tc_minute, i_tc_second, i_tc_frame
                                                 ) ;
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_avi::read_hdrl( wxUint32 &dw_size_list )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset      fo_offset_end ;
   str_chunk         chunk         ;
   str_MainAVIHeader avih          ;

   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( chunk ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   fo_offset_end = m_fa.get_offset() + chunk.dw_size ;

   /*--( It starts with a "avih" )-----------------------------------------*/
   if(    chunk.dw_id != SR_FOURCC( 'a','v','i','h' )
       || chunk.dw_size < sizeof( avih )
       || read_le_data( avih ) != 0
     )
   {  return( -2 ) ; }

   /*--( In some "iavs" avi the sizes can be 0 ... )-----------------------*/
   if( avih.dwWidth > 0 && avih.dwHeight > 0 )
   {  m_fi.init_video_x_y( avih.dwWidth, avih.dwHeight ) ; }
   /*----------------------------------------------------------------------*/
   dw_size_list -= sizeof( chunk ) + sizeof( avih ) ;
   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( fo_offset_end ) != 0 )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_avi::read_list()
{
   /*----------------------------------------------------------------------*/
   wxFileOffset fo_offset_end_list ;
   wxUint32     dw_id_list         ;
   wxUint32     dw_size            ;

   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( dw_size ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   fo_offset_end_list = m_fa.get_offset() + dw_size ;

   /*----------------------------------------------------------------------*/
   while( !m_boo_end && ( int )dw_size > 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.read_le_data( dw_id_list ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      dw_size -= sizeof( dw_id_list ) ;

      /*--------------------------------------------------------------------+
      ! Quit if invalid ! To quit without error the size is set to 0        !
      +--------------------------------------------------------------------*/
      if( !st_avi_id_valid( dw_id_list ) )
      {  dw_size = 0 ; continue ; }

      /*-------------------------------------------------------------------*/
      switch( dw_id_list )
      {
         /*----------------------------------------------------------------+
         ! It seems that the "movi" list is the last interesting one       !
         +----------------------------------------------------------------*/
         case SR_FOURCC( 'm','o','v','i' ) :
            /*------------------------------------------------------------*/
            if( m_boo_dv ) { read_info_dv( dw_size ) ; }
            /*------------------------------------------------------------*/
            m_boo_end = true ;
            /*------------------------------------------------------------*/
            break ;

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'h','d','r','l' ) :
            if( read_hdrl( dw_size ) != 0 ) { return( -3 ) ; }
            break ;

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 's','t','r','l' ) :
            if( read_strl( dw_size ) != 0 ) { return( -4 ) ; }
            break ;

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'o','d','m','l' ) :
            if( skip_odml( dw_size ) != 0 ) { return( -5 ) ; }
            break ;

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 'I','N','F','O' ) :
            if( m_fi.riff_read_serie_info( dw_size, st_avi_conv_tag_col
                                         ) != 0
              )
            {  return( -6 ) ; }
            break ;

         /*--( A sub-list ? )----------------------------------------------*/
         case SR_FOURCC( 'L','I','S','T' ) :
            if( read_list() != 0 ) { return( -7 ) ; }
            break ;

         /*--( By default jump over the data )-----------------------------*/
         default :
         {
            /*-------------------------------------------------------------*/
            wxUint32 dw_size_elmt ;
            /*-------------------------------------------------------------*/
            if(    m_fa.read_le_data( dw_size_elmt ) != 0
                || (     ( int )dw_size_elmt > 0
                     && m_fa.skip_nb_byte( dw_size_elmt ) != 0
                   )
              )
            {  return( -8 ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( fo_offset_end_list ) != 0 )
   {  return( -9 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_avi::avi_read()
{
   /*----------------------------------------------------------------------*/
   str_riff_header header    ;
   wxUint32        dw_id     ;
   wxUint32        dw_size   ;
   int             i_ret = 0 ;

   /*--( First, the header )-----------------------------------------------*/
   if( m_fa.read_le_data( header ) != 0 )
   {  return( -1 ) ; }

   /*--( It has to begin with ... )----------------------------------------*/
   if( header.dw_id != SR_FOURCC( 'A','V','I',' ' ) )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   while( !m_boo_end && m_fa.read_le_data( dw_id ) == 0 )
   {
      /*-------------------------------------------------------------------*/
      if( dw_id == SR_FOURCC( 'L','I','S','T' ) )
      {
         /*----------------------------------------------------------------*/
         if( read_list() != 0 )
         {  i_ret = -3 ; goto function_end ; }
         /*----------------------------------------------------------------*/
      }
      else /*--( Jump over the unamanaged type )---------------------------*/
      if(    m_fa.read_le_data( dw_size ) != 0
          || ( ( int )dw_size > 0 && m_fa.skip_nb_byte( dw_size ) != 0 )
        )
      {  i_ret = -4 ; goto function_end ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
function_end :
   /*----------------------------------------------------------------------*/
   if( m_do_fps != 0 )
   {  /*-------------------------------------------------------------------*/
      m_fi.init_video_fps( m_do_fps ) ;
      /*-------------------------------------------------------------------*/
      if( m_dw_nb_frames != 0 )
      {  m_fi.init_video_duration( ( int )( m_dw_nb_frames / m_do_fps ) ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !m_s_video_info.empty() )
   {    m_f.val_s( COL_VIDEO_INFO )
      = m_fi.concat_av_codecs( m_s_audio_info, m_s_video_info ) ;
   }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_avi()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "avi" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_avi( *this ).avi_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_avi.cpp                        !
+==========================================================================*/
