/*====================================+=====================================+
! File CFileInit_au.cpp               ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                         Audio file: ".au"                                 !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_au_header ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_au : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_au( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_be_data( str_au_header &ah ) ;
      int au_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_au_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_magic_number ; // Contains ".snd"
   wxUint32 dw_data_offset  ; // Minimal value: 24
   wxUint32 dw_data_size    ; // For "unknown" should be: 0xFFFFFFFF
   wxUint32 dw_encoding     ;
   wxUint32 dw_samprate     ; // Number of samples by second
   wxUint32 dw_channel      ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_au::read_be_data( str_au_header &ah )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ah ), &ah ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   ah.dw_magic_number = wxUINT32_SWAP_ALWAYS( ah.dw_magic_number ) ;
   ah.dw_data_offset  = wxUINT32_SWAP_ALWAYS( ah.dw_data_offset ) ;
   ah.dw_data_size    = wxUINT32_SWAP_ALWAYS( ah.dw_data_size ) ;
   ah.dw_encoding     = wxUINT32_SWAP_ALWAYS( ah.dw_encoding ) ;
   ah.dw_samprate     = wxUINT32_SWAP_ALWAYS( ah.dw_samprate ) ;
   ah.dw_channel      = wxUINT32_SWAP_ALWAYS( ah.dw_channel ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_au::au_read()
{
   /*----------------------------------------------------------------------*/
   str_au_header header                                  ;
   const char    *p_c_info                               ;
   wxFileOffset  fo_offset_data                          ;
   wxULongLong   ull_size                                ;
   int           i_bit_per_sample                        ;
   char          tb_c_annotation[ g_co_i_string_sz_max ] ;

   /*----------------------------------------------------------------------*/
   if( read_be_data( header ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if(    header.dw_magic_number
       != wxUINT32_SWAP_ON_LE( SR_FOURCC( '.','s','n','d' ) )
     )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   if( ( fo_offset_data = header.dw_data_offset ) < 24 )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   ull_size = header.dw_data_size ;
   if( header.dw_data_size == ( wxUint32 )~0 || ull_size >= m_f.get_size() )
   {  ull_size = m_f.get_size() - fo_offset_data ; }
   if( ull_size <= 0 ) { return( -4 ) ; }

   /*-----------------------------------------------------------------------+
   ! The annotation string follows the header. It may be composed of        !
   ! strings separated with 0x00. That's why a local treatment is made.     !
   +-----------------------------------------------------------------------*/
   if( fo_offset_data > ( wxFileOffset )sizeof( header ) )
   {
      /*-------------------------------------------------------------------*/
      size_t   sz_size_annotation ;
      size_t   sz_num             ;
      wxString s_val              ;
      /*-------------------------------------------------------------------*/
      sz_size_annotation = wxMin( fo_offset_data - sizeof( header ),
                                  sizeof( tb_c_annotation )
                                ) ;
      /*-------------------------------------------------------------------*/
      if( m_fa.read_buffer( sz_size_annotation, tb_c_annotation ) != 0 )
      {  return( -5 ) ; }
      /*--( Replacement of the 0x00 that in fact are only separators )-----*/
      for( sz_num = 0 ; sz_num < sz_size_annotation ; ++sz_num )
      {  /*----------------------------------------------------------------*/
         if( tb_c_annotation[ sz_num ] == '\0' )
         {  tb_c_annotation[ sz_num ] = ' ' ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( m_fi.prepare_string( tb_c_annotation, sz_size_annotation, s_val
                             ) > 0
        )
      {  m_f.val_s( COL_AUDTAG_COMMENT ) = s_val ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   switch( header.dw_encoding )
   {
      /*-------------------------------------------------------------------*/
      case  1 : p_c_info = "8-bit u-law" ;
                i_bit_per_sample = 8 ;
                break ;
      case  2 : p_c_info = "8-bit linear PCM" ;
                i_bit_per_sample = 8 ;
                break ;
      case  3 : p_c_info = "16-bit linear PCM" ;
                i_bit_per_sample = 16 ;
                break ;
      case  4 : p_c_info = "24-bit linear PCM" ;
                i_bit_per_sample = 24 ;
                break ;
      case  5 : p_c_info = "32-bit linear PCM" ;
                i_bit_per_sample = 32 ;
                break ;
      case  6 : p_c_info = "32-bit IEEE floating point" ;
                i_bit_per_sample = 32 ;
                break ;
      case  7 : p_c_info = "64-bit IEEE double precision float" ;
                i_bit_per_sample = 64 ;
                break ;
      case  8 : p_c_info = "Fragmented sample data" ;
                i_bit_per_sample = 0  ;
                break ;
      case 10 : p_c_info = "DSP program" ;
                i_bit_per_sample = 0 ;
                break ;
      case 11 : p_c_info = "8-bit fixed point" ;
                i_bit_per_sample = 8 ;
                break ;
      case 12 : p_c_info = "16-bit fixed point" ;
                i_bit_per_sample = 16 ;
                break ;
      case 13 : p_c_info = "24-bit fixed point" ;
                i_bit_per_sample = 24 ;
                break ;
      case 14 : p_c_info = "32-bit fixed point" ;
                i_bit_per_sample = 32 ;
                break ;
      case 18 : p_c_info = "16-bit linear with emphasis" ;
                i_bit_per_sample = 16 ;
                break ;
      case 19 : p_c_info = "16-bit linear compressed" ;
                i_bit_per_sample = 16 ;
                break ;
      case 20 :   p_c_info
                = "16-bit linear with emphasis and compression" ;
                i_bit_per_sample = 16 ;
                break ;
      case 21 : p_c_info = "Music kit DSP commands" ;
                i_bit_per_sample = 0 ;
                break ;
      case 23 : p_c_info = "4-bit CCITT G.721 ADPCM" ;
                i_bit_per_sample = 4 ;
                break ;
      case 24 : p_c_info = "CCITT G.722 ADPCM" ;
                i_bit_per_sample = 0 ;
                break ;
      case 25 : p_c_info = "CCITT G.723.3 ADPCM" ;
                i_bit_per_sample = 0 ;
                break ;
      case 26 : p_c_info = "CCITT G.723.5 ADPCM" ;
                i_bit_per_sample = 0 ;
                break ;
      case 27 : p_c_info = "8-bit A-law G.711" ;
                i_bit_per_sample = 8 ;
                break ;
      /*-------------------------------------------------------------------*/
      default : p_c_info = NULL ;
                i_bit_per_sample = 0 ;
                break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( p_c_info != NULL )
   {  m_f.val_s( COL_AUDIO_INFO ) = p_c_info ; }

   /*----------------------------------------------------------------------*/
   m_fi.init_audio_samprate( header.dw_samprate ) ;
   m_fi.init_audio_channel( header.dw_channel )   ;
   /*----------------------------------------------------------------------*/
   if(    header.dw_channel != 0
       && i_bit_per_sample != 0
       && header.dw_samprate != 0
     )
   {
      /*-------------------------------------------------------------------*/
      m_fi.init_audio_duration( ( int )(   ull_size.ToULong() * 8.0
                                         / header.dw_channel
                                         / i_bit_per_sample
                                         / header.dw_samprate
                                       )
                              ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_au()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "au" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_au( *this ).au_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_au.cpp                        !
+==========================================================================*/
