/*====================================+=====================================+
! File CFileInit_apk.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                       Android package: ".apk"                             !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! The extraction is done from two files belonging to the "apk" :            !
! - "AndroidManifest.xml" (AXML file)                                       !
!   Based on "AXMLPrinter2" java source code                                !
! - "META-INF/xxxx" with the extension "RSA" or "DSA"                       !
!   X.509 public key certificate                                            !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <memory>
#include <wx/wfstream.h>
#include <wx/zipstrm.h>
#include <wx/sstream.h>
#include <wx/regex.h>
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_apk_axml_header ;
struct str_apk_axml_prefix_uri ;
struct str_apk_axml_tag_header ;
struct str_apk_axml_attr ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_apk : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_apk( CFileInit &parent ) : CFileInit_type_base( parent ),
                                           m_p_input( NULL ) // for cppcheck
      {  ; }

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      wxInputStream *m_p_input ;
      /*-------------------------------------------------------------------*/
      typedef std::vector< str_apk_axml_attr > t_vec_attr ;
      typedef std::vector< wxUint32 > t_vec_offset ;
      t_vec_offset m_vec_offset_string ;
      /*-------------------------------------------------------------------*/
      wxMemoryBuffer m_mb_strings ;
      /*-------------------------------------------------------------------*/
      typedef std::vector< str_apk_axml_prefix_uri > t_vec_prefix_uri ;
      t_vec_prefix_uri m_vec_prefix_uri ;
      /*-------------------------------------------------------------------*/

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int read_buffer( size_t sz, void *p_buffer ) ;
      int read_buffer( size_t sz, wxMemoryBuffer &mb ) ;
      int read_data( wxUint32 &dw ) ;
      int read_data( str_chunk &chunk ) ;
      int skip_nb_byte( wxFileOffset val )
      {  return(    m_p_input->SeekI( val, wxFromCurrent )
                 != wxInvalidOffset ? 0 : -1
               ) ;
      }
      int read_data( str_apk_axml_header &aph ) ;
      int read_data( str_apk_axml_prefix_uri &prur ) ;
      int read_data( str_apk_axml_tag_header &asth ) ;
      int read_data( str_apk_axml_attr &aat ) ;
      wxString get_axml_string( size_t sz_num ) ;
      wxString get_axml_package( wxUint32 dw_id ) ;
      wxString get_axml_prefix_from_uri( wxUint32 dw_uri ) ;
      wxString get_axml_attr_val( const str_apk_axml_attr &attr ) ;
      int read_vec_dw( wxUint32 dw_nb, t_vec_offset &vec ) ;
      int read_vec_dw( wxUint32 dw_nb, t_vec_attr &vec ) ;
      int read_axml() ;
      int read_x509_string( wxString &s ) ;
      int read_x509_id_length( wxUint8 &b_type, size_t &sz_length ) ;
      int skip_x509_info( const char *p_c_skip ) ;
      int read_x509() ;
      int apk_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
const wxUint32 st_co_dw_CHUNK_AXML_FILE           = 0x00080003 ;
const wxUint32 st_co_dw_CHUNK_TYPE                = 0x001C0001 ;
const wxUint32 st_co_dw_CHUNK_RESOURCEIDS         = 0x00080180 ;
const wxUint32 st_co_dw_CHUNK_XML_START_NAMESPACE = 0x00100100 ;
const wxUint32 st_co_dw_CHUNK_XML_END_NAMESPACE   = 0x00100101 ;
const wxUint32 st_co_dw_CHUNK_XML_START_TAG       = 0x00100102 ;
const wxUint32 st_co_dw_CHUNK_XML_END_TAG         = 0x00100103 ;
const wxUint32 st_co_dw_CHUNK_XML_TEXT            = 0x00100104 ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_apk_axml_header
{
   /*----------------------------------------------------------------------*/
   str_chunk chunk_file            ;
   str_chunk chunk_string_block    ;
   wxUint32  dw_string_count       ;
   wxUint32  dw_style_offset_count ;
   wxUint32  __dw_dummy            ;
   wxUint32  dw_strings_offset     ;
   wxUint32  dw_styles_offset      ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_apk_axml_prefix_uri
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_prefix ;
   wxUint32 dw_uri    ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_apk_axml_tag_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_uri        ;
   wxUint32 dw_name       ;
   wxUint32 __dw_dummy    ;
   wxUint32 dw_attr_count ;
   wxUint32 dw_class_attr ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_apk_axml_attr
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_uri          ;
   wxUint32 dw_name         ;
   wxUint32 dw_value_string ;
   wxUint32 dw_value_type   ;
   wxUint32 dw_value_data   ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )



/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_buffer( size_t sz, void *p_buffer )
{
   /*----------------------------------------------------------------------*/
   if( !m_p_input->Read( p_buffer, sz ).IsOk() )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_p_input->LastRead() != sz )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_buffer( size_t sz, wxMemoryBuffer &mb )
{
   /*----------------------------------------------------------------------*/
   if( !m_p_input->Read( mb.GetWriteBuf( sz ), sz ).IsOk() )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   mb.UngetWriteBuf( sz ) ;
   /*----------------------------------------------------------------------*/
   if( m_p_input->LastRead() != sz )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_data( wxUint32 &dw )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( sizeof( dw ), &dw ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   dw = wxUINT32_SWAP_ALWAYS( dw ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_data( str_chunk &chunk )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( sizeof( chunk ), &chunk ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   chunk.dw_size = wxUINT32_SWAP_ALWAYS( chunk.dw_size ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_data( str_apk_axml_header &aph )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( sizeof( aph ), &aph ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   chunk_file.dw_size = wxUINT32_SWAP_ALWAYS( aph.chunk_file.dw_size ) ;
     aph.chunk_string_block.dw_size
   = wxUINT32_SWAP_ALWAYS( aph.chunk_string_block.dw_size ) ;
     aph.dw_string_count
   = wxUINT32_SWAP_ALWAYS( aph.dw_string_count ) ;
     aph.dw_style_offset_count
   = wxUINT32_SWAP_ALWAYS( aph.dw_style_offset_count ) ;
   aph.dw_strings_offset = wxUINT32_SWAP_ALWAYS( aph.dw_strings_offset ) ;
   aph.dw_styles_offset  = wxUINT32_SWAP_ALWAYS( aph.dw_styles_offset ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_data( str_apk_axml_prefix_uri &prur )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( sizeof( prur ), &prur ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   prur.dw_prefix = wxUINT32_SWAP_ALWAYS( prur.dw_prefix ) ;
   prur.dw_uri    = wxUINT32_SWAP_ALWAYS( prur.dw_uri ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_data( str_apk_axml_tag_header &ath )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( sizeof( ath ), &ath ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   ath.dw_uri        = wxUINT32_SWAP_ALWAYS( ath.dw_uri ) ;
   ath.dw_name       = wxUINT32_SWAP_ALWAYS( ath.dw_name ) ;
   ath.dw_attr_count = wxUINT32_SWAP_ALWAYS( ath.dw_attr_count ) ;
   ath.dw_class_attr = wxUINT32_SWAP_ALWAYS( ath.dw_class_attr ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_data( str_apk_axml_attr &aat )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( sizeof( aat ), &aat ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   aat.dw_uri          = wxUINT32_SWAP_ALWAYS( aat.dw_uri ) ;
   aat.dw_name         = wxUINT32_SWAP_ALWAYS( aat.dw_name ) ;
   aat.dw_value_string = wxUINT32_SWAP_ALWAYS( aat.dw_value_string ) ;
   aat.dw_value_type   = wxUINT32_SWAP_ALWAYS( aat.dw_value_type ) ;
   aat.dw_value_data   = wxUINT32_SWAP_ALWAYS( aat.dw_value_data ) ;
#endif
   /*--( Keep only the biggest byte of the type ... cf AXmlResourceParser )*/
   aat.dw_value_type >>= 24  ;
   aat.dw_value_type &= 0xFF ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_vec_dw( wxUint32 dw_nb, t_vec_offset &vec )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_num ;
   wxUint32 dw_val ;
   /*----------------------------------------------------------------------*/
   vec.clear() ;
   /*----------------------------------------------------------------------*/
   for( dw_num = 0 ; dw_num < dw_nb ; ++dw_num )
   {
      /*-------------------------------------------------------------------*/
      if( read_data( dw_val ) != 0 ) { return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      vec.push_back( dw_val ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_vec_dw( wxUint32 dw_nb, t_vec_attr &vec )
{
   /*----------------------------------------------------------------------*/
   wxUint32          dw_num ;
   str_apk_axml_attr attr   ;
   /*----------------------------------------------------------------------*/
   vec.clear() ;
   /*----------------------------------------------------------------------*/
   for( dw_num = 0 ; dw_num < dw_nb ; ++dw_num )
   {
      /*-------------------------------------------------------------------*/
      if( read_data( attr ) != 0 ) { return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      vec.push_back( attr ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFileInit_apk::get_axml_string( size_t sz_num )
{
   /*----------------------------------------------------------------------*/
   if( sz_num >= m_vec_offset_string.size() )
   {  return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   wxUint32 dw_offset = m_vec_offset_string[ sz_num ] ;
   /*----------------------------------------------------------------------*/
   if( dw_offset >= m_mb_strings.GetDataLen() )
   {  return( wxEmptyString ) ; }

   /*--( Extract length of string to extract )-----------------------------*/
   wxUint8  *p_b_length = ( wxUint8 * )m_mb_strings.GetData() + dw_offset ;
   wxUint16 w_string_length = SR_TWOCC( *p_b_length, *( p_b_length + 1 ) ) ;
   wxString s_val ;

   /*--( Prepare the string. Size is expressed in bytes )------------------*/
   sr::prepare_string( ( char * )p_b_length + 2, w_string_length * 2, s_val,
                       sr::CONV_FROM_UTF16LE
                     ) ;
   /*----------------------------------------------------------------------*/
   return( s_val ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFileInit_apk::get_axml_package( wxUint32 dw_id )
{
   /*----------------------------------------------------------------------*/
   if( ( ( dw_id >> 24 ) & 0xFF ) == 1 )
   {  return( "android:" ) ; }
   /*----------------------------------------------------------------------*/
   return( wxEmptyString ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFileInit_apk::get_axml_prefix_from_uri( wxUint32 dw_uri )
{
   /*----------------------------------------------------------------------*/
   t_vec_prefix_uri::const_reverse_iterator crit ;
   wxString s_res ;
   /*----------------------------------------------------------------------*/
   for( crit = m_vec_prefix_uri.rbegin() ;
        crit != m_vec_prefix_uri.rend() && crit->dw_uri != dw_uri ;
        ++crit
      )
   {  ; }
   /*----------------------------------------------------------------------*/
   if( crit != m_vec_prefix_uri.rend() )
   {  s_res = get_axml_string( crit->dw_prefix ) ;
      if( !s_res.empty() ) { s_res += ':' ; }
   }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFileInit_apk::get_axml_attr_val( const str_apk_axml_attr &attr )
{
   /*----------------------------------------------------------------------*/
   wxString s_val ;
   /*----------------------------------------------------------------------*/
   switch( attr.dw_value_type )
   {
      /*-------------------------------------------------------------------*/
      //case  0 : // TYPE_NULL
      case  1 : // TYPE_REFERENCE
         s_val.Printf( "@%s%08X", get_axml_package( attr.dw_value_data ),
                       attr.dw_value_data
                     ) ;
         break ;
      case  2 : // TYPE_ATTRIBUTE
         s_val.Printf( "?%s%08X", get_axml_package( attr.dw_value_data ),
                       attr.dw_value_data
                     ) ;
         break ;
      case  3 : // TYPE_STRING
         s_val = get_axml_string( attr.dw_value_data ) ;
         break ;
      case  4 : // TYPE_FLOAT
         s_val.Printf( "%f (float)", sr::conv_le_fl( attr.dw_value_data ) ) ;
         break ;
      // case  5 : // TYPE_DIMENSION
      // case  6 : // TYPE_FRACTION
      /*-------------------------------------------------------------------*/
      case 16 : // TYPE_INT_DEC
         s_val.Printf( "%ld", ( long )attr.dw_value_data ) ;
         break ;
      case 17 : // TYPE_INT_HEX
         s_val.Printf( "0x%08X", attr.dw_value_data ) ;
         break ;
      case 18 : // TYPE_INT_BOOLEAN
         s_val.Printf( attr.dw_value_data == 0 ? "false" : "true" ) ;
         break ;
      /*-------------------------------------------------------------------*/
      case 28 : // TYPE_INT_COLOR_ARGB8
      case 29 : // TYPE_INT_COLOR_RGB8
      case 30 : // TYPE_INT_COLOR_ARGB4
      case 31 : // TYPE_INT_COLOR_RGB4
         s_val.Printf( "#%08X", attr.dw_value_data ) ;
         break ;
      /*-------------------------------------------------------------------*/
      default :
         s_val.Printf( "Val:0x%08X Type:0x%02X",
                       attr.dw_value_data, attr.dw_value_type
                     ) ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( s_val ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_axml()
{
   /*----------------------------------------------------------------------*/
   str_apk_axml_header header      ;
   str_chunk           chunk       ;
   bool                boo_end     ;
   int                 i_level = 0 ;

   /*----------------------------------------------------------------------*/
   if( read_data( header ) != 0 )
   {  return( -1 ) ; }

   /*--( Checking the chunk ids will "validate" the file )-----------------*/
   if(    header.chunk_file.dw_id != st_co_dw_CHUNK_AXML_FILE
       || header.chunk_string_block.dw_id != st_co_dw_CHUNK_TYPE
     )
   {  return( -2 ) ; }

   /*--( Read the offsets of the strings )---------------------------------*/
   if( read_vec_dw( header.dw_string_count, m_vec_offset_string ) != 0 )
   {  return( -3 ) ; }
   /*--( Skip the styles offsets )-----------------------------------------*/
   if(    header.dw_style_offset_count != 0
       && skip_nb_byte( header.dw_style_offset_count * sizeof( wxUint32 ) )
     )
   {  return( -4 ) ; }

   /*--( Read the strings data )-------------------------------------------*/
   if( read_buffer( ( header.dw_styles_offset == 0
                      ? header.chunk_string_block.dw_size
                      : header.dw_styles_offset
                    ) - header.dw_strings_offset,
                    m_mb_strings
                  ) != 0
     )
   {  return( -5 ) ; }

   /*--( Skip the styles data )--------------------------------------------*/
   if(    header.dw_styles_offset != 0
       && skip_nb_byte( (   header.chunk_string_block.dw_size
                          - header.dw_strings_offset
                        ) * sizeof( wxUint32 )
                      ) != 0
     )
   {  return( -6 ) ; }

   /*----------------------------------------------------------------------*/
   for( boo_end = false ; !boo_end ; )
   {
      /*-------------------------------------------------------------------*/
      if( read_data( chunk ) != 0 ) { return( -7 ) ; }
      /*-------------------------------------------------------------------*/
      if( chunk.dw_id == st_co_dw_CHUNK_RESOURCEIDS )
      {
         /*----------------------------------------------------------------*/
         if( skip_nb_byte( chunk.dw_size - sizeof( chunk ) ) != 0 )
         {  return( -8 ) ; }
         /*----------------------------------------------------------------*/
      }
      else
      {
         /*--( Skip the line number and 0xFFFFFFFF )-----------------------*/
         if( skip_nb_byte( sizeof( wxUint32 ) * 2 ) != 0 )
         {  return( -9 ) ; }
         /*----------------------------------------------------------------*/
         switch( chunk.dw_id )
         {
            /*-------------------------------------------------------------*/
            case st_co_dw_CHUNK_XML_START_NAMESPACE :
            case st_co_dw_CHUNK_XML_END_NAMESPACE :
            {
               /*----------------------------------------------------------*/
               str_apk_axml_prefix_uri prur ;
               /*----------------------------------------------------------*/
               if( read_data( prur ) != 0 )
               {  return( -10 ) ; }
               /*----------------------------------------------------------*/
               if( chunk.dw_id == st_co_dw_CHUNK_XML_START_NAMESPACE )
               {  m_vec_prefix_uri.push_back( prur ) ; }
               else
               {  m_vec_prefix_uri.pop_back() ; }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }

            /*-------------------------------------------------------------*/
            case st_co_dw_CHUNK_XML_START_TAG :
            {
               /*----------------------------------------------------------*/
               str_apk_axml_tag_header ath            ;
               t_vec_attr              vec_attr       ;
               wxUint16                w_attr_count   ;
               wxString                s_tag          ;
               wxString                s_attr         ;
               wxString                s_info_version ;

               /*--( One level deeper )------------------------------------*/
               ++i_level ;
               /*----------------------------------------------------------*/
               if( read_data( ath ) != 0 )
               {  return( -11 ) ; }
               /*--( Read tag header )-------------------------------------*/
               s_tag =   get_axml_prefix_from_uri( ath.dw_uri )
                       + get_axml_string( ath.dw_name ) ;
               //w_id_attr = ( ath.dw_attr_count >> 16 ) & 0x00FF ;
               w_attr_count = ath.dw_attr_count & 0x00FF ;
               /*--( And the attributes associated to it )-----------------*/
               if( read_vec_dw( w_attr_count, vec_attr ) != 0 )
               {  return( -12 ) ; }

               /*--( Only interested in a few data )-----------------------*/
               if( s_tag.CmpNoCase( "manifest" ) == 0 )
               {
                  /*-------------------------------------------------------*/
                  size_t sz_num ;
                  /*-------------------------------------------------------*/
                  for( sz_num = 0 ; sz_num < w_attr_count ; ++sz_num )
                  {
                     /*----------------------------------------------------*/
                       s_attr
                     =   get_axml_prefix_from_uri( vec_attr[ sz_num ].dw_uri)
                       + get_axml_string( vec_attr[ sz_num ].dw_name ) ;
                     /*----------------------------------------------------*/
                     if( s_attr.CmpNoCase( "android:versionName" ) == 0 )
                     {  /*-------------------------------------------------*/
                        s_info_version.insert(
                                      0,
                                      get_axml_attr_val( vec_attr[ sz_num ] )
                                              ) ;
                        /*-------------------------------------------------*/
                     }
                     else
                     if( s_attr.CmpNoCase( "android:versionCode" ) == 0 )
                     {  /*-------------------------------------------------*/
                        s_info_version.append(
                                      " ("
                                    + get_axml_attr_val( vec_attr[ sz_num ] )
                                    + ')'
                                             ) ;
                        /*-------------------------------------------------*/
                     }
                     else
                     if( s_attr == "package" )
                     {  /*-------------------------------------------------*/
                        if( m_fi.reserve_col( COL_DOC_TITLE ) )
                        {    m_f.val_s( COL_DOC_TITLE )
                           = get_axml_attr_val( vec_attr[ sz_num ] ) ;
                        }
                        /*-------------------------------------------------*/
                     }
                     /*----------------------------------------------------*/
                  }
                  /*--( All what is "interesting" has been scanned )-------*/
                  boo_end = true ;
                  /*-------------------------------------------------------*/
                  if(    !s_info_version.empty()
                      && m_fi.reserve_col( COL_DOC_VERSION )
                    )
                  {  m_f.val_s( COL_DOC_VERSION ) = s_info_version ; }
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }

            /*-------------------------------------------------------------*/
            case st_co_dw_CHUNK_XML_END_TAG :
            {
               /*--( Skip uri and name )-----------------------------------*/
               if( skip_nb_byte( sizeof( wxUint32 ) * 2 ) != 0 )
               {  return( -13 ) ; }
               /*--( One level up. End tag at "surface" (0) means the end )*/
               boo_end = ( --i_level == 0 ) ;
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }

            /*-------------------------------------------------------------*/
            case st_co_dw_CHUNK_XML_TEXT :
            {
               /*--( Skip uri and name )-----------------------------------*/
               if( skip_nb_byte( sizeof( wxUint32 ) * 2 ) != 0 )
               {  return( -14 ) ; }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }

            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_apk_conv_oid_col( const wxMemoryBuffer &mb )
{
   /*--( This OID "family" seems to follow a scheme )----------------------*/
   if(    mb.GetDataLen() != 3
       || *( ( wxUint8 * )mb.GetData() ) != 0x55
       || *( ( wxUint8 * )mb.GetData() + 1 ) != 0x04
      )
   {  return( COL_NB ) ; }

   /*----------------------------------------------------------------------*/
   switch( *( ( wxUint8 * )mb.GetData() + 2 ) )
   {  /*-------------------------------------------------------------------*/
      case 0x0A : return( COL_DOC_COMPANY ) ; // OID O
      case 0x03 : return( COL_DOC_AUTHOR )  ; // OID CN
      /*-------------------------------------------------------------------*/
   }

   /* Other available (and known) DN OID
   C  country = 55 04 06
   ST state = 55 04 08
   L  locality = 55 04 07
   O  organisation name = 55 04 0A
   OU organisational unit = 55 04 0B
   CN common name = 55 04 03
   */

   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--( The file pointer is supposed to be at the start of the data )--------*/
int CFileInit_apk::read_x509_string( wxString &s )
{
   /*----------------------------------------------------------------------*/
   wxUint8         b_id                                 ;
   size_t          sz_size                              ;
   char            tb_c_val[ g_co_i_string_sz_max * 2 ] ;
   size_t          sz_size_to_read                      ;
   sr::e_conv_type conv_type                            ;

   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   if( read_x509_id_length( b_id, sz_size ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( sz_size == 0 ) { return( 0 ) ; }

   /*--( Different possible strings )--------------------------------------*/
   switch( b_id )
   {  /*-------------------------------------------------------------------*/
      case 0x04 : // Octet String
      case 0x13 : // Printable String
      case 0x16 : // IAS5 String
         conv_type = sr::CONV_FROM_8BIT ;
         break ;
      /*-------------------------------------------------------------------*/
      case 0x0C : // UTF8 String
         conv_type = sr::CONV_FROM_UTF8 ;
         break ;
      /*-------------------------------------------------------------------*/
      default :
         return( -2 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   sz_size_to_read = wxMin( sz_size, ( int )sizeof( tb_c_val ) ) ;
   /*----------------------------------------------------------------------*/
   if(    read_buffer( sz_size_to_read, tb_c_val ) != 0
       || (    sz_size_to_read < sz_size
            && skip_nb_byte( sz_size - sz_size_to_read ) != 0
          )
     )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   sr::prepare_string( tb_c_val, sz_size_to_read, s, conv_type ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( On return the file pointer is on the data start )--------------------*/
int CFileInit_apk::read_x509_id_length( wxUint8 &b_id, size_t &sz_length )
{
   /*--( Read the ID )-----------------------------------------------------*/
   if( !m_p_input->CanRead() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   b_id = m_p_input->GetC() ;
   /*----------------------------------------------------------------------*/
   if( !m_p_input->CanRead() ) { return( -2 ) ; }
   /*-----------------------------------------------------------------------+
   ! Read the DER length. If bit 7 is set this means that the length is     !
   ! specified in the N following bytes. N is indicated in the 0..6 bits    !
   +-----------------------------------------------------------------------*/
   sz_length = m_p_input->GetC() ;
   if( ( sz_length & 0x80 ) != 0 )
   {
      /*-------------------------------------------------------------------*/
      size_t sz_size = ( sz_length & 0x7F ) ;
      /*--( Minimal control )----------------------------------------------*/
      if( sz_size > sizeof( int ) ) { return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      for( sz_length = 0 ; sz_size > 0 ; --sz_size )
      {
         /*----------------------------------------------------------------*/
         if( !m_p_input->CanRead() ) { return( -4 ) ; }
         /*----------------------------------------------------------------*/
         sz_length += ( sz_length << 8 ) + m_p_input->GetC() ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function will skip DER information based on the parameter string     !
! content:                                                                  !
! h : only the header has to be skipped: id + length                        !
!     This is used for "SEQUENCE", "SET", "SET OF" ...                      !
! a : the complete next element is skipped: id + length + data              !
+--------------------------------------------------------------------------*/
int CFileInit_apk::skip_x509_info( const char *p_c_skip )
{
   /*----------------------------------------------------------------------*/
   wxUint8 b_id      ;
   size_t  sz_length ;
   /*----------------------------------------------------------------------*/
   if( p_c_skip == NULL || *p_c_skip == '\0' )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   for( ; *p_c_skip != '\0' ; ++p_c_skip )
   {
      /*-------------------------------------------------------------------*/
      if( read_x509_id_length( b_id, sz_length ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      switch( *p_c_skip )
      {
         /*--( Skip only the header )--------------------------------------*/
         case 'h' : break ;
         /*--( Skip all: header and data )---------------------------------*/
         case 'a' :
            if( skip_nb_byte( sz_length ) != 0 )
            {  return( -3 ) ; }
            break ;
         /*----------------------------------------------------------------*/
         default :
            return( -4 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::read_x509()
{
   /*----------------------------------------------------------------------*/
   wxUint8        b_id      ;
   size_t         sz_length ;
   wxMemoryBuffer mb_val    ;
   wxString       s_val     ;
   wxFileOffset   fo_end_dn ;
   int            i_col     ;

   /*--( Minimal check )---------------------------------------------------*/
   if( m_p_input->Peek() != 0x30 )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! First skip all the data just before where the Version number is        !
   ! supposed to be.                                                        !
   ! As it is optional, its presence has to be checked before skipping it   !
   +-----------------------------------------------------------------------*/
   if( skip_x509_info( "hahhaaahhh" ) != 0 )
   {  return( -2 ) ; }

   /*--( 0xA0 seems an accurate test: Class context, constructed, tag 0 )--*/
   if(    ( wxUint8 )m_p_input->Peek() == 0xA0
       && skip_x509_info( "a") != 0
     )
   {  return( -3 ) ; }

   /*--( Skip the rest before the Subject DN )-----------------------------*/
   if( skip_x509_info( "aaaa" ) != 0 )
   {  return( -4 ) ; }

   /*--( Just before the Subject DN )--------------------------------------*/
   if( read_x509_id_length( b_id, sz_length ) != 0 )
   {  return( -5 ) ; }
   /*----------------------------------------------------------------------*/
   fo_end_dn = m_p_input->TellI() + sz_length ;

   /*--( Scan the different fields )---------------------------------------*/
   while( m_p_input->TellI() < fo_end_dn )
   {
      /*--( First comes the SEQUENCE and the first element header )--------*/
      if( read_x509_id_length( b_id, sz_length ) != 0 )
      {  return( -6 ) ; }
      if( read_x509_id_length( b_id, sz_length ) != 0 )
      {  return( -7 ) ; }

      /*--( First comes an OID )-------------------------------------------*/
      if( read_x509_id_length( b_id, sz_length ) != 0 )
      {  return( -8 ) ; }
      /*-------------------------------------------------------------------*/
      if( b_id != 0x06 )
      {  return( -9 ) ; }
      /*--( Read the OID value )-------------------------------------------*/
      if( read_buffer( sz_length, mb_val ) != 0 )
      {  return( -10 ) ; }

      /*--( Then the string )----------------------------------------------*/
      i_col = st_apk_conv_oid_col( mb_val ) ;
      if( m_fi.reserve_col( i_col ) )
      {  /*----------------------------------------------------------------*/
         if( read_x509_string( s_val ) != 0 ) { return( -11 ) ; }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = s_val ;
         /*----------------------------------------------------------------*/
      }
      else
      if( skip_x509_info( "a" ) != 0 )
      {  return( -12 ) ; }

      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_apk::apk_read()
{
   /*--( The zip data should begin here )----------------------------------*/
   wxFileInputStream           finput( m_fa )        ;
   wxZipInputStream            zip_input( finput )   ;
   std::auto_ptr< wxZipEntry > ptr_zip_entry         ;
   bool                        boo_axml_done = false ;
   bool                        boo_x509_done = false ;

   /*----------------------------------------------------------------------*/
   if( !zip_input.IsOk() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   m_fi.init_doc_zip_info( zip_input ) ;
   m_p_input = &zip_input ;

   /*--( Looking for the files containing the metadata )-------------------*/
   do
   {  /*-------------------------------------------------------------------*/
      ptr_zip_entry.reset( zip_input.GetNextEntry() ) ;
      /*-------------------------------------------------------------------*/
      if( ptr_zip_entry.get() != NULL )
      {
         /*----------------------------------------------------------------*/
         if(    !boo_axml_done
             && ptr_zip_entry->GetInternalName().CmpNoCase(
                                                        "AndroidManifest.xml"
                                                          ) == 0
           )
         {  boo_axml_done = true ;
            if( read_axml() != 0 ) { return( -2 ) ; }
         }
         else
         if( !boo_x509_done )
         {
            /*--------------------------------------------------------------+
            ! The X.509 file base name is not specified                     !
            +--------------------------------------------------------------*/
            wxFileName fname( ptr_zip_entry->GetInternalName() ) ;
            /*-------------------------------------------------------------*/
            if(    fname.GetPath().CmpNoCase( "META-INF" ) == 0
                && (    fname.GetExt().CmpNoCase( "RSA" ) == 0
                     || fname.GetExt().CmpNoCase( "DSA" ) == 0
                   )
              )
            {  boo_x509_done = true ;
               if( read_x509() != 0 ) { return( -3 ) ; }
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   } while(    ptr_zip_entry.get() != NULL
            && ( !boo_axml_done || !boo_x509_done )
          ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_apk()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "ape" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_apk( *this ).apk_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_apk.cpp                        !
+==========================================================================*/
