/*====================================+=====================================+
! File CFileInit_Type.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <memory>
#include <wx/mimetype.h>
#include <wx/regex.h>
#include <wx/wfstream.h>
#include <wx/zipstrm.h>
#include <wx/thread.h>
#include "CFile.h"
#include "CFileInit.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! This class enables to find the right information based on the extension   !
! of the file :                                                             !
! - Type description                                                        !
! - function to extract metadata                                            !
+--------------------------------------------------------------------------*/
class CFileInit_ExtensionSupport
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_ExtensionSupport() ;

   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      typedef std::map< wxString, t_p_init_func > t_map_ext_init_func ;
      typedef t_map_ext_init_func::const_iterator t_map_ext_init_func_cit ;
      /*-------------------------------------------------------------------*/
      t_map_ext_init_func m_ext_init_func ; // Func to extract metadata
      /*--------------------------------------------------------------------+
      ! This map will be updated by the "get_type_desc" function.           !
      +--------------------------------------------------------------------*/
      sr::t_map_str_str   m_ext_type_desc ; // File type description string
      /*--------------------------------------------------------------------+
      ! A "file init", and so "get_type_desc", can be executed from the     !
      ! main and the loading threads. As the instance of this class will be !
      ! static updating it will need some precautions => critical section   !
      +--------------------------------------------------------------------*/
      wxCriticalSection   m_csec_ext ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      t_p_init_func get_init_func( const CFileInit &fi ) ;
      wxString get_type_desc( const CFileInit &fi ) ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
static CFileInit_ExtensionSupport st_extension_support ;

/*--------------------------------------------------------------------------+
! As the associated variable is declared as static the constructor will be  !
! called only once.                                                         !
+--------------------------------------------------------------------------*/
CFileInit_ExtensionSupport::CFileInit_ExtensionSupport()
{
   /*--( The file extensions have to be lowercase )------------------------*/
   m_ext_init_func[ "au"   ] = &CFileInit::init_au   ;
   m_ext_init_func[ "snd"  ] = &CFileInit::init_au   ;
   m_ext_init_func[ "mp3"  ] = &CFileInit::init_mp3  ;
   m_ext_init_func[ "mp2"  ] = &CFileInit::init_mp3  ;
   m_ext_init_func[ "kar"  ] = &CFileInit::init_kar  ;
   m_ext_init_func[ "mid"  ] = &CFileInit::init_kar  ;
   m_ext_init_func[ "ogg"  ] = &CFileInit::init_ogg  ;
   m_ext_init_func[ "ogm"  ] = &CFileInit::init_ogg  ;
   m_ext_init_func[ "oga"  ] = &CFileInit::init_ogg  ;
   m_ext_init_func[ "ogv"  ] = &CFileInit::init_ogg  ;
   m_ext_init_func[ "ape"  ] = &CFileInit::init_ape  ;
   m_ext_init_func[ "mpc"  ] = &CFileInit::init_mpc  ;
   m_ext_init_func[ "crw"  ] = &CFileInit::init_crw  ; // Raw Canon
   m_ext_init_func[ "mrw"  ] = &CFileInit::init_mrw  ; // Raw Minolta
   m_ext_init_func[ "raf"  ] = &CFileInit::init_raf  ; // Raw Fuji
   m_ext_init_func[ "x3f"  ] = &CFileInit::init_x3f  ; // Raw Sigma
   m_ext_init_func[ "tif"  ] = &CFileInit::init_tiff ;
   m_ext_init_func[ "tiff" ] = &CFileInit::init_tiff ;
   m_ext_init_func[ "nef"  ] = &CFileInit::init_tiff ; // Raw Nikon
   m_ext_init_func[ "nrw"  ] = &CFileInit::init_tiff ; // Raw Nikon
   m_ext_init_func[ "orf"  ] = &CFileInit::init_tiff ; // Raw Olympus
   m_ext_init_func[ "pef"  ] = &CFileInit::init_tiff ; // Raw Pentax
   m_ext_init_func[ "srf"  ] = &CFileInit::init_tiff ; // Raw Sony
   m_ext_init_func[ "sr2"  ] = &CFileInit::init_tiff ; // Raw Sony
   m_ext_init_func[ "arw"  ] = &CFileInit::init_tiff ; // Raw Sony
   m_ext_init_func[ "cr2"  ] = &CFileInit::init_tiff ; // Raw Canon
   m_ext_init_func[ "dng"  ] = &CFileInit::init_tiff ; // Raw Adobe
   m_ext_init_func[ "raw"  ] = &CFileInit::init_tiff ; // Raw Leica
   m_ext_init_func[ "erf"  ] = &CFileInit::init_tiff ; // Raw Epson
   m_ext_init_func[ "kdc"  ] = &CFileInit::init_tiff ; // Raw Kodak
   m_ext_init_func[ "dcr"  ] = &CFileInit::init_tiff ; // Raw Kodak
   m_ext_init_func[ "3fr"  ] = &CFileInit::init_tiff ; // Raw Hasselblat
   m_ext_init_func[ "mos"  ] = &CFileInit::init_tiff ; // Raw Creo/Leaf
   m_ext_init_func[ "rw2"  ] = &CFileInit::init_tiff ; // Raw Panasonic
   m_ext_init_func[ "mef"  ] = &CFileInit::init_tiff ; // Raw Mamiya
   m_ext_init_func[ "jpg"  ] = &CFileInit::init_jpg  ;
   m_ext_init_func[ "jpeg" ] = &CFileInit::init_jpg  ;
   m_ext_init_func[ "thm"  ] = &CFileInit::init_jpg  ; // Asso to CRW Canon
   m_ext_init_func[ "mpo"  ] = &CFileInit::init_jpg  ; // Stereoscopic JPG
   m_ext_init_func[ "gif"  ] = &CFileInit::init_gif  ;
   m_ext_init_func[ "png"  ] = &CFileInit::init_png  ;
   m_ext_init_func[ "pcx"  ] = &CFileInit::init_pcx  ;
   m_ext_init_func[ "bmp"  ] = &CFileInit::init_bmp  ;
   m_ext_init_func[ "dib"  ] = &CFileInit::init_bmp  ;
   m_ext_init_func[ "ico"  ] = &CFileInit::init_ico  ;
   m_ext_init_func[ "cur"  ] = &CFileInit::init_ico  ;
   m_ext_init_func[ "tga"  ] = &CFileInit::init_tga  ;
   m_ext_init_func[ "tpic" ] = &CFileInit::init_tga  ;
   m_ext_init_func[ "xpm"  ] = &CFileInit::init_xpm  ;
   m_ext_init_func[ "pnm"  ] = &CFileInit::init_pnm  ;
   m_ext_init_func[ "ppm"  ] = &CFileInit::init_pnm  ;
   m_ext_init_func[ "webp" ] = &CFileInit::init_webp ;
   m_ext_init_func[ "svg"  ] = &CFileInit::init_svg  ;
   m_ext_init_func[ "url"  ] = &CFileInit::init_url  ;
   m_ext_init_func[ "avi"  ] = &CFileInit::init_avi  ;
   m_ext_init_func[ "divx" ] = &CFileInit::init_avi  ;
   m_ext_init_func[ "mpg"  ] = &CFileInit::init_mpg  ;
   m_ext_init_func[ "mpeg" ] = &CFileInit::init_mpg  ;
   m_ext_init_func[ "mov"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "m4v"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "m4a"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "m4b"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "m4r"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "mp4"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "3gp"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "f4v"  ] = &CFileInit::init_qtf  ;
   m_ext_init_func[ "asf"  ] = &CFileInit::init_asf  ;
   m_ext_init_func[ "wmv"  ] = &CFileInit::init_asf  ;
   m_ext_init_func[ "wm"   ] = &CFileInit::init_asf  ;
   m_ext_init_func[ "wma"  ] = &CFileInit::init_asf  ;
   m_ext_init_func[ "mkv"  ] = &CFileInit::init_mkv  ;
   m_ext_init_func[ "webm" ] = &CFileInit::init_mkv  ;
   m_ext_init_func[ "ra"   ] = &CFileInit::init_real ;
   m_ext_init_func[ "rv"   ] = &CFileInit::init_real ;
   m_ext_init_func[ "rm"   ] = &CFileInit::init_real ;
   m_ext_init_func[ "rmvb" ] = &CFileInit::init_real ;
   m_ext_init_func[ "mka"  ] = &CFileInit::init_mkv  ;
   m_ext_init_func[ "wav"  ] = &CFileInit::init_wav  ;
   m_ext_init_func[ "torrent" ] = &CFileInit::init_torrent ;
   m_ext_init_func[ "pdf"  ] = &CFileInit::init_pdf  ;
   m_ext_init_func[ "html" ] = &CFileInit::init_html ;
   m_ext_init_func[ "htm"  ] = &CFileInit::init_html ;
   m_ext_init_func[ "zip"  ] = &CFileInit::init_zip  ;
   m_ext_init_func[ "jar"  ] = &CFileInit::init_zip  ; // Only zip data
   m_ext_init_func[ "war"  ] = &CFileInit::init_zip  ; // Only zip data
   m_ext_init_func[ "htb"  ] = &CFileInit::init_zip  ; // Only zip data
   m_ext_init_func[ "deb"  ] = &CFileInit::init_deb  ;
   m_ext_init_func[ "rpm"  ] = &CFileInit::init_rpm  ;
   m_ext_init_func[ "epub" ] = &CFileInit::init_epub ;
   m_ext_init_func[ "flv"  ] = &CFileInit::init_flv  ;
   m_ext_init_func[ "swf"  ] = &CFileInit::init_swf  ;
   m_ext_init_func[ "flac" ] = &CFileInit::init_flac ;
   m_ext_init_func[ "aac"  ] = &CFileInit::init_aac  ;
   m_ext_init_func[ "ttf"  ] = &CFileInit::init_ttf  ; // TrueType Font
   m_ext_init_func[ "otf"  ] = &CFileInit::init_ttf  ; // OpenType Font
   m_ext_init_func[ "odt"  ] = &CFileInit::init_odf  ; // OpenDoc Text
   m_ext_init_func[ "ods"  ] = &CFileInit::init_odf  ; // OpenDoc Spreadsheet
   m_ext_init_func[ "odp"  ] = &CFileInit::init_odf  ; // OpenDoc Pres
   m_ext_init_func[ "ott"  ] = &CFileInit::init_odf  ; // OpenDoc Text Modele
   m_ext_init_func[ "eml"  ] = &CFileInit::init_eml  ;
   m_ext_init_func[ "desktop" ] = &CFileInit::init_desktop ;
   m_ext_init_func[ "iso" ] = &CFileInit::init_iso   ;
   m_ext_init_func[ "xpi" ] = &CFileInit::init_xpi   ;
   m_ext_init_func[ "crx" ] = &CFileInit::init_crx   ;
   m_ext_init_func[ "apk" ] = &CFileInit::init_apk   ;
   m_ext_init_func[ "mo"  ] = &CFileInit::init_mo    ;
#ifdef __WXMSW__
   m_ext_init_func[ "lnk"  ] = &CFileInit::init_lnk  ;
   m_ext_init_func[ "exe"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "dll"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "com"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "ocx"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "cpl"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "tlb"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "vbx"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "olb"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "cnv"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "rll"  ] = &CFileInit::init_exe  ;
   m_ext_init_func[ "fon"  ] = &CFileInit::init_exe  ; // MSW Font
   m_ext_init_func[ "doc"  ] = &CFileInit::init_off  ; // Word
   m_ext_init_func[ "xls"  ] = &CFileInit::init_off  ; // Excel
   m_ext_init_func[ "xla"  ] = &CFileInit::init_off  ; // Macros Excel
   m_ext_init_func[ "xlt"  ] = &CFileInit::init_off  ; // Template Excel
   m_ext_init_func[ "pot"  ] = &CFileInit::init_off  ; // Powerpoint
   m_ext_init_func[ "pps"  ] = &CFileInit::init_off  ; // Powerpoint
   m_ext_init_func[ "ppt"  ] = &CFileInit::init_off  ; // Powerpoint
   m_ext_init_func[ "ppa"  ] = &CFileInit::init_off  ; // Add-in Power Point
   m_ext_init_func[ "mpp"  ] = &CFileInit::init_off  ; // Project
   m_ext_init_func[ "dot"  ] = &CFileInit::init_off  ; // Template
   m_ext_init_func[ "msi"  ] = &CFileInit::init_off  ; // Install
   m_ext_init_func[ "msp"  ] = &CFileInit::init_off  ; // Patch
   m_ext_init_func[ "mst"  ] = &CFileInit::init_off  ; // Transform
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
t_p_init_func CFileInit_ExtensionSupport::get_init_func( const CFileInit &fi)
{
   /*----------------------------------------------------------------------*/
   wxString s_ext_lc( fi.m_f.get_name().GetExt().Lower() ) ;
   /*----------------------------------------------------------------------*/
   if( s_ext_lc.empty() ) { return( NULL ) ; }
   /*-----------------------------------------------------------------------+
   ! Extension comparison test is done case insensitive because under       !
   ! Ubuntu/Gnome/Nautilus it seems to be the same ...                      !
   +-----------------------------------------------------------------------*/
   t_map_ext_init_func_cit   cit_ext_init_func
                           = m_ext_init_func.find( s_ext_lc ) ;
   /*----------------------------------------------------------------------*/
   return( cit_ext_init_func == m_ext_init_func.end()
           ? NULL : cit_ext_init_func->second
         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Returns the file type description associated to the file extension )-*/
wxString CFileInit_ExtensionSupport::get_type_desc( const CFileInit &fi )
{
   /*----------------------------------------------------------------------*/
   wxString s_ext_lc( fi.m_f.get_name().GetExt().Lower() ) ;
   /*----------------------------------------------------------------------*/
   if( s_ext_lc.empty() ) { return( wxEmptyString ) ; }
   /*--( Extension stored lower case )-------------------------------------*/
   sr::t_map_str_str_cit cit_ext = m_ext_type_desc.find( s_ext_lc ) ;
   /*----------------------------------------------------------------------*/
   if( cit_ext != m_ext_type_desc.end() )
   {  return( cit_ext->second ) ; }

   /*----------------------------------------------------------------------*/
   wxString s_type ;

   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   /*--( Get "msw" file type )---------------------------------------------*/
   SHFILEINFO sfi ;
   /*----------------------------------------------------------------------*/
   if( SHGetFileInfo( "." + s_ext_lc, FILE_ATTRIBUTE_NORMAL,
                      &sfi, sizeof( sfi ),
                      SHGFI_USEFILEATTRIBUTES | SHGFI_TYPENAME
                    ) != 0
     )
   {  s_type = sfi.szTypeName ; }
   /*----------------------------------------------------------------------*/
#else
   /*--( Get "wx" file type )----------------------------------------------*/
     wxFileType *p_file_type
   = wxTheMimeTypesManager->GetFileTypeFromExtension( s_ext_lc ) ;
   if( p_file_type != NULL )
   {  /*-------------------------------------------------------------------*/
      p_file_type->GetDescription( &s_type ) ;
      delete p_file_type ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
#endif // __WXMSW__

   /*--( If not found, a default description is built )--------------------*/
   if( s_type.empty() )
   {  s_type.Printf( _( "%s file" ), s_ext_lc ) ; }

   /*--( This part is "guarded". Cf class declaration )--------------------*/
   {
      wxCriticalSectionLocker enter_csec( m_csec_ext ) ;
      m_ext_type_desc[ s_ext_lc ] = s_type ;
   }
   /*----------------------------------------------------------------------*/
   return( s_type ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFileInit::get_type_desc()
{
   /*----------------------------------------------------------------------*/
   return( st_extension_support.get_type_desc( *this ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
t_p_init_func CFileInit::get_init_func_ext() const
{
   /*----------------------------------------------------------------------*/
   return( st_extension_support.get_init_func( *this ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Simplistic file content "analyze" to detect its type and call the right   !
! meta-data extraction function                                             !
+--------------------------------------------------------------------------*/
t_p_init_func CFileInit::get_init_func_detect()
{
   /*----------------------------------------------------------------------*/
   wxUint8 tb_b_data[ 3 * 1024 ] ;
   const size_t co_sz_data_1 = 24 ;

   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( 0 ) != 0 )
   {  return( NULL ) ; }
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( co_sz_data_1, tb_b_data ) != 0 )
   {  return( NULL ) ; }

   /*-----------------------------------------------------------------------+
   ! ID3 can be present at the beginning of different files types (mp3,     !
   ! aac ...).                                                              !
   +-----------------------------------------------------------------------*/
   if(    SR_THREECC( tb_b_data[ 0 ], tb_b_data[ 1 ], tb_b_data[ 2 ] )
       == SR_THREECC( 'I','D','3' )
     )
   {
      /*-------------------------------------------------------------------*/
      const wxUint32 co_dw_val = SR_FOURCC( tb_b_data[ 6 ], tb_b_data[ 7 ],
                                            tb_b_data[ 8 ], tb_b_data[ 9 ]
                                          ) ;
      wxUint32 dw_header_length = mp3_conv_id3v2_synch_size( co_dw_val ) ;
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( 10 + dw_header_length ) != 0 )
      {  return( NULL ) ; }
      /*-------------------------------------------------------------------*/
      if( m_fa.read_buffer( sizeof( tb_b_data ), tb_b_data ) != 0 )
      {  return( NULL ) ; }

      /*--( aac file ? )---------------------------------------------------*/
      if(    ( tb_b_data[ 0 ] & 0xFF ) == 0xFF
          && ( tb_b_data[ 1 ] & 0xF6 ) == 0xF0
        )
      {  return( &CFileInit::init_aac ) ; }

      /*--( By default assume it is a mp3 file ... )-----------------------*/
      return( &CFileInit::init_mp3 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! TGA has no "magic number", so let's try ...                            !
   +-----------------------------------------------------------------------*/
   if(    ( tb_b_data[ 1 ] == 0 || tb_b_data[ 1 ] == 1 )     // Color MapType
       && (    tb_b_data[ 2 ] == 1  || tb_b_data[ 2 ] == 2   // Image type
            || tb_b_data[ 2 ] == 3  || tb_b_data[ 2 ] == 9
            || tb_b_data[ 2 ] == 10 || tb_b_data[ 2 ] == 11
          )
       && (    tb_b_data[ 16 ] == 8  || tb_b_data[ 16 ] == 16 // Bit / pixel
            || tb_b_data[ 16 ] == 24 || tb_b_data[ 16 ] == 32
          )
     )
   {  return( &CFileInit::init_tga ) ; }

   /*----------------------------------------------------------------------*/
   switch( SR_TWOCC( tb_b_data[ 0 ], tb_b_data[ 1 ] ) )
   {
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'B','M' ) : case SR_TWOCC( 'B','A' ) :
      case SR_TWOCC( 'C','I' ) : case SR_TWOCC( 'C','P' ) :
      case SR_TWOCC( 'I','C' ) : case SR_TWOCC( 'P','T' ) :
         return( &CFileInit::init_bmp ) ;
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'I','I' ) : case SR_TWOCC( 'M','M' ) :
         /*----------------------------------------------------------------*/
         if( memcmp( &tb_b_data[ 6 ], "HEAPCCDR", 8 ) == 0 )
         {  return( &CFileInit::init_crw ) ; }
         /*----------------------------------------------------------------*/
         return( &CFileInit::init_tiff ) ;
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 0xFF,0xD8 ) : return( &CFileInit::init_jpg ) ;
      /*-------------------------------------------------------------------*/
#ifdef __WXMSW__
      case SR_TWOCC( 'M','Z' ) : return( &CFileInit::init_exe ) ;
#endif // __WXMSW__
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'P','K' ) :
      {
         /*-----------------------------------------------------------------+
         ! "zip" files have to be inspected to find an expected file        !
         ! (hoping the file names are unique to each type ...)              !
         +-----------------------------------------------------------------*/
         wxFileInputStream           finput( m_fa )      ;
         wxZipInputStream            zip_input( finput ) ;
         std::auto_ptr< wxZipEntry > ptr_zip_entry       ;
         wxString                    s_name              ;
         /*----------------------------------------------------------------*/
         do
         {
            /*-------------------------------------------------------------*/
            ptr_zip_entry.reset( zip_input.GetNextEntry() ) ;
            /*-------------------------------------------------------------*/
            if( ptr_zip_entry.get() != NULL )
            {
               /*----------------------------------------------------------*/
               s_name = ptr_zip_entry->GetInternalName() ;
               /*----------------------------------------------------------*/
               if( s_name.CmpNoCase( "AndroidManifest.xml" ) == 0 )
               {  return( &CFileInit::init_apk ) ; }
               /*----------------------------------------------------------*/
               if( s_name.CmpNoCase( "META-INF/container.xml" ) == 0 )
               {  return( &CFileInit::init_epub ) ; }
               /*----------------------------------------------------------*/
               if( s_name.CmpNoCase( "meta.xml" ) == 0 )
               {  return( &CFileInit::init_odf ) ; }
               /*----------------------------------------------------------*/
               if( s_name.CmpNoCase( "install.rdf" ) == 0 )
               {  return( &CFileInit::init_xpi ) ; }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         } while( ptr_zip_entry.get() != NULL ) ;

         /*--( Default see it at "zip" )-----------------------------------*/
         return( &CFileInit::init_zip ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   switch( SR_THREECC( tb_b_data[ 0 ], tb_b_data[ 1 ], tb_b_data[ 2 ] ) )
   {
      /*-------------------------------------------------------------------*/
      case SR_THREECC( 'M','P','+' ) : return( &CFileInit::init_mpc ) ;
      case SR_THREECC( 'F','L','V' ) : return( &CFileInit::init_flv ) ;
      case SR_THREECC( 'G','I','F' ) : return( &CFileInit::init_gif ) ;
      /*-------------------------------------------------------------------*/
      case SR_THREECC( 'C','W','S' ) :
      case SR_THREECC( 'F','W','S' ) : return( &CFileInit::init_swf ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   switch( SR_FOURCC( tb_b_data[ 0 ], tb_b_data[ 1 ],
                      tb_b_data[ 2 ], tb_b_data[ 3 ]
                    )
         )
   {
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( '.','s','n','d' )     : return( &CFileInit::init_au ) ;
      case SR_FOURCC( 0xED,0xAB,0xEE,0xDB ) : return( &CFileInit::init_rpm );
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 0x00,0x00,0x01,0x00 ) : // icon
      case SR_FOURCC( 0x00,0x00,0x02,0x00 ) : // cursor
         return( &CFileInit::init_ico ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 0x1A,0x45,0xDF,0xA3 ) : return( &CFileInit::init_mkv );
      case SR_FOURCC( 0xDE,0x12,0x04,0x95 ) : return( &CFileInit::init_mo ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 0x00,0x00,0x01,0xB3 ) :
      case SR_FOURCC( 0x00,0x00,0x01,0xBA ) :
         return( &CFileInit::init_mpg ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'C','r','2','4' )  : return( &CFileInit::init_crx ) ;
      case SR_FOURCC( 'f','L','a','C' )  : return( &CFileInit::init_flac ) ;
      case SR_FOURCC( 'M','T','h','d' )  : return( &CFileInit::init_kar ) ;
      case SR_FOURCC( 0x00,'M','R','M' ) : return( &CFileInit::init_mrw ) ;
      case SR_FOURCC( 'M','A','C', ' ')  : return( &CFileInit::init_ape ) ;
      case SR_FOURCC( 'O','g','g','S' )  : return( &CFileInit::init_ogg ) ;
      case SR_FOURCC( '%','P','D','F' )  : return( &CFileInit::init_pdf ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'R','I','F','F' ) :
         /*----------------------------------------------------------------*/
         switch( SR_FOURCC( tb_b_data[  8 ], tb_b_data[  9 ],
                            tb_b_data[ 10 ], tb_b_data[ 11 ]
                          )
               )
         {  /*-------------------------------------------------------------*/
            case SR_FOURCC( 'W','A','V','E' ) :
               return( &CFileInit::init_wav ) ;
            case SR_FOURCC( 'A','V','I',' ' ) :
               return( &CFileInit::init_avi ) ;
            case SR_FOURCC( 'W','E','B','P' ) :
               return( &CFileInit::init_webp ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'F','O','V','b' )     : return( &CFileInit::init_x3f );
      case SR_FOURCC( 0x30,0x26,0xB2,0x75 ) : return( &CFileInit::init_asf );
      case SR_FOURCC( 0x00,0x00,0x00,0x00 ) : return( &CFileInit::init_iso );
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 0x00,0x01,0x00,0x00 ) :
      case SR_FOURCC( 'O','T','T','O' ) :
         return( &CFileInit::init_ttf ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( '.','r','a',0xFD ) :
      case SR_FOURCC( '.','R','M','F' ) :
         return( &CFileInit::init_real ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( tb_b_data[ 0 ] == 'P' && tb_b_data[ 2 ] == '\n' )
   {  return( &CFileInit::init_pnm ) ; }
   /*----------------------------------------------------------------------*/
   if( tb_b_data[ 0 ] == 0x0A && tb_b_data[ 2 ] == 0x01 )
   {  return( &CFileInit::init_pcx ) ; }
   /*----------------------------------------------------------------------*/
   wxUint32 co_dw_val = SR_FOURCC( tb_b_data[ 0 ], tb_b_data[ 1 ],
                                   tb_b_data[ 2 ], tb_b_data[ 3 ]
                                 ) ;
   str_mp3_frame_info mp3_fi ;
   if( mp3_extr_frame_info( co_dw_val, mp3_fi ) == 0 )
   {  return( &CFileInit::init_mp3 ) ; }
   /*----------------------------------------------------------------------*/
   if(    ( tb_b_data[ 0 ] & 0xFF ) == 0xFF
       && ( tb_b_data[ 1 ] & 0xF6 ) == 0xF0
     )
   {  return( &CFileInit::init_aac ) ; }
   /*----------------------------------------------------------------------*/
   switch( SR_FOURCC( tb_b_data[ 4 ], tb_b_data[ 5 ],
                      tb_b_data[ 6 ], tb_b_data[ 7 ]
                    )
         )
   {  case SR_FOURCC( 'f','t','y','p' ) : case SR_FOURCC( 'm','o','o','v' ) :
      case SR_FOURCC( 'm','d','a','t' ) : case SR_FOURCC( 'w','i','d','e' ) :
      case SR_FOURCC( 'f','r','e','e' ) :
         return( &CFileInit::init_qtf ) ;
   }
   /*----------------------------------------------------------------------*/
   if( memcmp( tb_b_data, "d8:announce", 11 ) == 0 )
   {  return( &CFileInit::init_torrent ) ; }
   /*----------------------------------------------------------------------*/
   if( memcmp( tb_b_data, "!<arch>" "\x0A" "debian-", 15 ) == 0 )
   {  return( &CFileInit::init_deb ) ; }
   /*----------------------------------------------------------------------*/
   if( memcmp( tb_b_data, "/* XPM */", 9 ) == 0 )
   {  return( &CFileInit::init_xpm ) ; }
   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   if( memcmp( tb_b_data, "\xD0\xCF\x11\xE0\xA1\xB1\x1A\xE1", 8 ) == 0 )
   {  return( &CFileInit::init_off ) ; }
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
   if( memcmp( tb_b_data, "FUJIFILM", 8 ) == 0 )
   {  return( &CFileInit::init_raf ) ; }
   /*----------------------------------------------------------------------*/
   if( memcmp( tb_b_data, "\x89PNG\x0D\x0A\x1A\x0A", 8 ) == 0 )
   {  return( &CFileInit::init_png ) ; }

   /*-----------------------------------------------------------------------+
   ! From this point more information is needed to achieve the detection    !
   +-----------------------------------------------------------------------*/
   size_t sz_data_2 = sizeof( tb_b_data ) - co_sz_data_1 ;
   if( m_fa.read_buffer_max( sz_data_2, tb_b_data + co_sz_data_1 ) != 0 )
   {  return( NULL ) ; }

   /*--( Convert the buffer to string using html defaut )------------------*/
   wxString s_data( tb_b_data, wxCSConv( "ISO-8859-1" ), sz_data_2 ) ;

   /*----------------------------------------------------------------------*/
   if( s_data.find( "[Desktop Entry]" ) != wxString::npos )
   {  return( &CFileInit::init_desktop ) ; }
   /*----------------------------------------------------------------------*/
   if( s_data.find( "[InternetShortcut]" ) != wxString::npos )
   {  return( &CFileInit::init_url ) ; }
   /*----------------------------------------------------------------------*/
   if( s_data.find( "\x0ATo: " ) != wxString::npos )
   {  return( &CFileInit::init_eml ) ; }
   /*----------------------------------------------------------------------*/
   if( wxRegEx( "<\\s*?html", wxRE_ADVANCED|wxRE_ICASE ).Matches( s_data ) )
   {  return( &CFileInit::init_html ) ; }
   /*----------------------------------------------------------------------*/
   if( wxRegEx( "<\\s*?svg", wxRE_ADVANCED|wxRE_ICASE ).Matches( s_data ) )
   {  return( &CFileInit::init_svg ) ; }

   /*----------------------------------------------------------------------*/
   return( NULL ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Returns "true" if DV video )-----------------------------------------*/
bool CFileInit::info_fcc_video( const char * const p_c_fcc,
                                wxString &s_info
                              ) const
{
   /*----------------------------------------------------------------------*/
   bool boo_dv = false ;

   /*----------------------------------------------------------------------*/
   switch( SR_FOURCC( toupper( p_c_fcc[ 0 ] ),
                      toupper( p_c_fcc[ 1 ] ),
                      toupper( p_c_fcc[ 2 ] ),
                      toupper( p_c_fcc[ 3 ] )
                    )
         )
   {
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('D','I','V','X') :
         s_info = "DivX 4"    ; break ;
      case SR_FOURCC('D','I','V','3') :
         s_info = "DivX 3 LM" ; break ;
      case SR_FOURCC('D','I','V','4') :
         s_info = "DivX 3 FM" ; break ;
      case SR_FOURCC('D','I','V','5') :
         s_info = "DivX 5"    ; break ;
      case SR_FOURCC('D','X','5','0') :
         s_info = "DivX 5"    ; break ;
      case SR_FOURCC('X','V','I','D') :
         s_info = "XVID"     ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('M','P','G','4') : case SR_FOURCC('M','P','4','V') :
         s_info = "Mpeg4 Video" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('8','B','P','S') :
         s_info = "Planar RGB" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('Y','U','Y','2') :
      case SR_FOURCC('Y','U','V','2') :
      case SR_FOURCC('U','Y','V','Y') :
         s_info = "YUV" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('I','V','3','2') :
         s_info = "Indeo 3.2" ; break ;
      case SR_FOURCC('I','V','4','0') : case SR_FOURCC('I','V','4','1') :
      case SR_FOURCC('I','V','4','2') : case SR_FOURCC('I','V','4','3') :
      case SR_FOURCC('I','V','4','4') : case SR_FOURCC('I','V','4','5') :
      case SR_FOURCC('I','V','4','6') : case SR_FOURCC('I','V','4','7') :
      case SR_FOURCC('I','V','4','8') : case SR_FOURCC('I','V','4','9') :
      case SR_FOURCC('I','V','5','0') :
         s_info = "Indeo Interactive" ; break ;
      case SR_FOURCC('C','V','I','D') :
         s_info = "Cinepak" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('R','L','E',' ') :
         s_info = "Microsoft RLE" ; break ;
      case SR_FOURCC('M','S','V','C') : case SR_FOURCC('C','R','A','M') :
      case SR_FOURCC('W','H','A','M') :
         s_info = "Microsoft Video 1" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('M','P','4','1') :
         s_info = "Microsoft Mpeg4 v1" ; break ;
      case SR_FOURCC('M','P','4','2') :
         s_info = "Microsoft Mpeg4 v2" ; break ;
      case SR_FOURCC('M','P','4','3') :
         s_info = "Microsoft Mpeg4 v3" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('R','P','Z','A') :
         s_info = "Apple Video" ; break ;
      case SR_FOURCC('S','M','C',' ') :
         s_info = "Apple Graphics" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('R','A','W',' ') :
         s_info = "Raw RGB" ; break ;
      case SR_FOURCC('J','P','E','G') :
         s_info = "JPEG Still Image" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('R','V','1','0') :
         s_info = "RealVideo 1.0" ; break ;
      case SR_FOURCC('R','V','2','0') :
         s_info = "RealVideo 2.0" ; break ;
      case SR_FOURCC('R','V','3','0') :
         s_info = "RealVideo 3.0" ; break ;
      case SR_FOURCC('R','V','4','0') :
         s_info = "RealVideo 4.0" ; break ;
      case SR_FOURCC('M','J','P','A') :
      case SR_FOURCC('M','J','P','B') :
      case SR_FOURCC('M','J','P','G') :
         s_info = "Motion JPEG" ; break ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('A','V','C','1') :
         s_info = "H.264" ; break;
      case SR_FOURCC('S','2','6','3') :
         s_info = "H.263" ; break;
      case SR_FOURCC('S','V','Q','1') :
         s_info = "Sorenson Video 1" ; break ;
      case SR_FOURCC('S','V','Q','3') :
         s_info = "Sorenson Video 3" ; break ;

      /*-------------------------------------------------------------------*/
      case SR_FOURCC('D','V','S','D') : case SR_FOURCC('D','V','H','D') :
      case SR_FOURCC('D','V','S','L') :
      case SR_FOURCC('D','S','V','D') :
      case SR_FOURCC('C','D','V','C') :
         /*----------------------------------------------------------------*/
         boo_dv = true ;
         s_info = "DV Video" ;
         break ;

      /*-------------------------------------------------------------------*/
      default :
      {
         /*----------------------------------------------------------------*/
         s_info += "Video (" ;
         s_info += sr::byte_to_string( p_c_fcc[ 0 ] ) ;
         s_info += sr::byte_to_string( p_c_fcc[ 1 ] ) ;
         s_info += sr::byte_to_string( p_c_fcc[ 2 ] ) ;
         s_info += sr::byte_to_string( p_c_fcc[ 3 ] ) ;
         s_info += ')' ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( boo_dv ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_Type.cpp                       !
+==========================================================================*/
