/*====================================+=====================================+
! File CFile_Comp.cpp                 ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "CFile.h"
#include "CApp.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Only one sub-expression will be evaluated.                                !
! The returned pointer is the first character of the next expression to     !
! evaluate or "it_exp_end"                                                  !
+--------------------------------------------------------------------------*/
sr::wxString_cit CFile::compute_sub_exp( t_exp_comp_info &eci ) const
{
   /*----------------------------------------------------------------------*/
   bool boo_dot_passed = false ;

   /*----------------------------------------------------------------------*/
   eci.s_current_value.clear() ;
   /*----------------------------------------------------------------------*/
   while( eci.it_exp < eci.it_exp_end )
   {
      /*-------------------------------------------------------------------*/
      switch( ( *eci.it_exp ).GetValue() )
      {
         /*--( Escape next character )-------------------------------------*/
         case '\\' :
            /*-------------------------------------------------------------*/
            if( ++eci.it_exp < eci.it_exp_end )
            {  eci.s_current_value += *eci.it_exp ; }
            /*-------------------------------------------------------------*/
            break ;

         /*--( Base name or extension ? )----------------------------------*/
         case '*' :
         {
            /*-------------------------------------------------------------*/
            const wxFileName &co_name
                  = ( eci.boo_first_sub_exp ? get_name() : get_new_name() ) ;
            /*-------------------------------------------------------------*/
            if( boo_dot_passed )
            {  eci.s_current_value += get_comp_extension( co_name ) ; }
            else
            {  eci.s_current_value += get_comp_basename( co_name ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*--( What comes next is a variable ? )---------------------------*/
         case '%' :
         {
            /*-------------------------------------------------------------*/
            wxString s_val ;
            /*-------------------------------------------------------------*/
            compute_exp_variable( eci, s_val ) ;
            eci.s_current_value += s_val ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case ';' :
            /*--( Skip over "empty" expressions )--------------------------*/
            do
            {  sr::next_non_space( eci.it_exp_end, ++eci.it_exp ) ;
            } while( eci.it_exp != eci.it_exp_end && *eci.it_exp == ';' ) ;

            /*--( Another expression ? )-----------------------------------*/
            if( eci.it_exp != eci.it_exp_end )
            {  return( eci.it_exp ) ; }

            /*--( The end of the loop will do a "++" ... so ... )----------*/
            --eci.it_exp ;
            /*-------------------------------------------------------------*/
            break ;

         /*--( Normal character )------------------------------------------*/
         default :
            /*-------------------------------------------------------------*/
            eci.s_current_value += *eci.it_exp ;
            /*--( Is it the "dot" separating the base from extension ? )---*/
            if( *eci.it_exp == '.' )
            {  boo_dot_passed = true ; }
            else /*--( A directory name can contain "." characters ... )---*/
            if( *eci.it_exp == wxFileName::GetPathSeparator() )
            {  boo_dot_passed = false ; }
            /*-------------------------------------------------------------*/
            break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( eci.it_exp < eci.it_exp_end ) { ++eci.it_exp ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   /*-----------------------------------------------------------------------+
   ! Under Windows a cleaning of useless ':' is done because the way        !
   ! wxFileName handles it is not very user friendly under Siren.           !
   ! A ':' is only accepted when present at the second position.            !
   ! It is done here while the new name has not been converted to a         !
   ! wxFileName yet.                                                        !
   +-----------------------------------------------------------------------*/
   sr::wxString_it it ;
   /*----------------------------------------------------------------------*/
   for( it = eci.s_current_value.begin() ; it != eci.s_current_value.end() ;
        ++it
      )
   {  /*-------------------------------------------------------------------*/
      if( *it == ':' && it - eci.s_current_value.begin() != 1 )
      {  *it = wxGetApp().M_s_rep_char_inv.get()[ 0 ] ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
#endif // __WXMSW__

   /*----------------------------------------------------------------------*/
   return( eci.it_exp ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFile::name_global_modification( wxFileName &name ) const
{
   /*----------------------------------------------------------------------*/
   if( !name.IsOk() ) { return ; }

   /*----------------------------------------------------------------------*/
   wxString s_name( name.GetFullPath() ) ;

   /*--( Delete characters )-----------------------------------------------*/
   if( !wxGetApp().M_s_del_chars.empty() )
   {  sr::delete_chars( s_name, wxGetApp().M_s_del_chars.get() ) ; }

   /*--( Delete repeated characters )--------------------------------------*/
   if( !wxGetApp().M_s_del_repeat_chars.empty() )
   {  sr::delete_repeat_chars( s_name,
                               wxGetApp().M_s_del_repeat_chars.get()
                             ) ;
   }

   /*--( Replace characters )----------------------------------------------*/
   if( !wxGetApp().M_s_rep_chars_from.empty() )
   {  sr::replace_chars( s_name,
                         wxGetApp().M_s_rep_chars_from.get(),
                         wxGetApp().M_s_rep_chars_to.get()
                       ) ;
   }

   /*--( Forbidden chars )-------------------------------------------------*/
   sr::replace_filename_forbidden_chars( s_name,
                                         wxGetApp().M_s_rep_char_inv.get()
                                       ) ;

   /*--( Cut what is out of limit )----------------------------------------*/
   if(    wxGetApp().M_i_max_name_len.get() != g_co_i_undef_max_name_len
       && ( int )s_name.size() > wxGetApp().M_i_max_name_len.get()
     )
   {  /*-------------------------------------------------------------------*/
      wxString s_ext ;
      wxFileName::SplitPath( s_name, NULL, NULL, &s_ext ) ;
      /*--( Where to do the cut ? with or without the extension ? )--------*/
      if(    s_ext.empty()
          || 1 + ( int )s_ext.size() >= wxGetApp().M_i_max_name_len.get()
        )
      {  s_name.erase( wxGetApp().M_i_max_name_len.get() ) ; }
      else
      {  s_name.erase(   wxGetApp().M_i_max_name_len.get()
                       - 1 - s_ext.size()
                     ) ;
         s_name += '.' + s_ext ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Re-injection of the modified string )-----------------------------*/
   name.Assign( s_name ) ;

   /*-----------------------------------------------------------------------+
   ! If a volume is defined it is necessary (and clearer) to make the path  !
   ! absolute.                                                              !
   +-----------------------------------------------------------------------*/
   if( name.HasVolume() && !name.IsAbsolute() )
   {  /*--( Normalize can make a "SetSwd" and generate an error (Ex A:B:C))*/
      wxLogNull no_log ;
      /*-------------------------------------------------------------------*/
      name.Normalize( wxPATH_NORM_ABSOLUTE ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! If the new name has no path then it is supposed to be located at the   !
   ! same place as the name.                                                !
   +-----------------------------------------------------------------------*/
   if( name.GetDirCount() == 0 && m_name.GetDirCount() > 0 )
   {  name.SetPath( m_name.GetPath() ) ; }

   /*----------------------------------------------------------------------*/
}

/*--( Base name of a computed version of the name for this file )----------*/
wxString CFile::get_comp_basename( const wxFileName &name ) const
{
   /*--( Directories extension ? )-----------------------------------------*/
   if( is_dir() && !wxGetApp().M_boo_exp_dir_ext.get() )
   {  return( name.GetFullName() ) ; }
   /*----------------------------------------------------------------------*/
   return( name.GetName() ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Extension of a computed version of the name for this file )----------*/
wxString CFile::get_comp_extension( const wxFileName &name ) const
{
   /*--( Directories extension ? )-----------------------------------------*/
   if( is_dir() && !wxGetApp().M_boo_exp_dir_ext.get() )
   {  return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   return( name.GetExt() ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Computation of a specific expression on a specific file without touching  !
! the source file object.                                                   !
! The returned value is the resulting "new name"                            !
+--------------------------------------------------------------------------*/
wxFileName CFile::compute_exp( const wxString &s_exp_src )
{
   /*----------------------------------------------------------------------*/
   t_exp_comp_info eci      ;
   wxString        s_exp    ;
   wxFileName      new_name ;

   /*-----------------------------------------------------------------------+
   ! Computation initialisations                                            !
   ! On error an "empty" "wxFileName" is returned                           !
   +-----------------------------------------------------------------------*/
   if( CFileList::st_compute_init( s_exp_src, s_exp ) != 0 )
   {  return( new_name ) ; }

   /*----------------------------------------------------------------------*/
   new_name.Assign( get_name() ) ;
   eci.boo_first_sub_exp = true ;
   eci.it_exp = s_exp.begin() ;
   eci.it_exp_end = s_exp.end() ;

   /*-----------------------------------------------------------------------+
   ! Even if the expression is empty the re-computation is done to zero the !
   ! result                                                                 !
   +-----------------------------------------------------------------------*/
   do
   {
      /*-------------------------------------------------------------------*/
      compute_sub_exp( eci ) ;
      /*-------------------------------------------------------------------*/
      new_name.Assign( eci.s_current_value ) ;
      /*-------------------------------------------------------------------*/
   } while( eci.it_exp < eci.it_exp_end ) ;

   /*----------------------------------------------------------------------*/
   name_global_modification( new_name ) ;

   /*----------------------------------------------------------------------*/
   return( new_name ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Computation of a specific expression on a specific file without touching  !
! the source file object                                                    !
!                                                                           !
! Why this new function ?                                                   !
! It has been created to be used for the tools.                             !
! The problem with the "compute_exp" function is that it is intended to     !
! generate a new file name with all modifications associated to it:         !
!     - import/export in a "wxFileName" (=> no "//" lost)                   !
!       The conversion is done before "compute_next_exp" when in            !
!       "compute_exp" it is done after                                      !
!     - name_global_modification : remove forbidden chars (=> no '%')       !
!                                  cut after maximum size                   !
!     - ... etc ...                                                         !
+--------------------------------------------------------------------------*/
wxString CFile::compute_exp_tool( const wxString &s_exp_src )
{
   /*----------------------------------------------------------------------*/
   t_exp_comp_info eci      ;
   wxString        s_exp    ;
   wxFileName      new_name ;

   /*--( Computation initialisations )-------------------------------------*/
   if( CFileList::st_compute_init( s_exp_src, s_exp ) != 0 )
   {  return( wxEmptyString ) ; }

   /*----------------------------------------------------------------------*/
   new_name.Assign( get_name() ) ;
   eci.boo_first_sub_exp = true ;
   eci.it_exp = s_exp.begin() ;
   eci.it_exp_end = s_exp.end() ;

   /*-----------------------------------------------------------------------+
   ! Even if expression is empty the recomputation is done to zero the      !
   ! result                                                                 !
   +-----------------------------------------------------------------------*/
   do
   {
      /*-------------------------------------------------------------------*/
      compute_sub_exp( eci ) ;
      /*-------------------------------------------------------------------*/
      new_name.Assign( eci.s_current_value ) ;
      /*-------------------------------------------------------------------*/
   } while( eci.it_exp < eci.it_exp_end ) ;

   /*----------------------------------------------------------------------*/
   return( eci.s_current_value ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFile_Comp.cpp                          !
+==========================================================================*/
