/*====================================+=====================================+
! File CFile_Base.cpp                 ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFile.h"
#include "CApp.h"
#include "CFileInit.h"
#include "common/CChecksum.h"
#include "CFSWatcher.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CExternalData_Systime   CFile::st_m_ed_st ;
CExternalData_Clipboard CFile::st_m_ed_cp ;
CExternalData_File      CFile::st_m_ed_fi ;
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
void CFile::reset( bool boo_is_dir )
{
   /*----------------------------------------------------------------------*/
   m_boo_is_dir = boo_is_dir ;
   m_i_ico      = -1 ;
   /*----------------------------------------------------------------------*/
   m_s_val.clear()  ;
   m_ll_val.clear() ;
   m_do_val.clear() ;
   m_dt_val.clear() ;
   m_fo_image_offset     = -1 ;
   m_ull_image_size      =  0 ;
   m_fo_thumbnail_offset = -1 ;
   m_ull_thumbnail_size  =  0 ;
   m_i_duration          =  0 ;
   m_img_rotation        = IV_ROTATION_0 ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool CFile::is_fake() const
{  return( m_p_cdir->is_fake() ) ; }

/*-------------------------------------------------------------------------*/
bool CFile::checksum_info_empty() const
{  return( m_s_md5.empty() && m_s_sha1.empty() && m_s_crc32.empty() );}

/*-------------------------------------------------------------------------*/
void CFile::checksum_info_copy( const CFile &f )
{
   /*----------------------------------------------------------------------*/
   m_dt_md5   = f.m_dt_md5   ;
   m_s_md5    = f.m_s_md5    ;
   m_dt_sha1  = f.m_dt_sha1  ;
   m_s_sha1   = f.m_s_sha1   ;
   m_dt_crc32 = f.m_dt_crc32 ;
   m_s_crc32  = f.m_s_crc32  ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFile::init_s_size()
{
   /*----------------------------------------------------------------------*/
     m_s_val[ COL_BASE_SIZE ]
   = get_s_size( wxGetApp().M_boo_fsizeb_fl.get() ? 'b' : ' ' ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Execute program or the application associated to the file )----------*/
void CFile::exec_system_open() const
{
   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   ShellExecute( NULL, NULL, get_full_path().wx_str(), NULL, NULL, SW_SHOW );
#else
   /*----------------------------------------------------------------------*/
   wxString s_fullpath_filename ;
   /*--( To avoid any problem the full path is used )----------------------*/
     s_fullpath_filename
   = sr::absolute_path( m_name, wxGetApp().M_s_dir.get() ) + get_full_name();
   /*--( Exec of the file itself ? this "if" is mainly used for GNU/Linux )*/
   if( !m_name.HasExt() && m_name.IsFileExecutable() )
   {  wxExecute( s_fullpath_filename ) ; }
   else /*--( No, let wx decide )------------------------------------------*/
   {  wxLaunchDefaultApplication( s_fullpath_filename ) ; }
   /*----------------------------------------------------------------------*/
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFile::init_info_s_track()
{
   /*----------------------------------------------------------------------*/
   static int st_tb_i_col[ 3 ]
   = { COL_ID3V1_TRACK_NUM, COL_AUDTAG_TRACK_NUM, COL_AUDTAG_TRACK_NB } ;
   const int co_i_track_pad = wxGetApp().M_i_track_pad.get() ;
   t_ll_val_cit it_ll_val ;
   int i_num ;

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < ( int )WXSIZEOF( st_tb_i_col ) ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      it_ll_val = m_ll_val.find( st_tb_i_col[ i_num ] ) ;
      if( it_ll_val != m_ll_val.end() )
      {  /*----------------------------------------------------------------*/
           m_s_val[ st_tb_i_col[ i_num ] ]
         = sr::lpad_ll_with_0( it_ll_val->second, co_i_track_pad ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFile::init_s_sel_num()
{
   /*----------------------------------------------------------------------*/
   m_s_val[ COL_NONE_SEL_NUM ].Printf( "%0*d",
                                       wxGetApp().M_i_sel_num_pad.get(),
                                           wxGetApp().M_i_sel_num_beg.get()
                                       +   m_i_sel_num
                                         * wxGetApp().M_i_sel_num_inc.get()
                                     ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFile::set_sel_num( int i_num )
{
   /*----------------------------------------------------------------------*/
   m_i_sel_num = i_num ;
   init_s_sel_num()    ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFile::unsel()
{
   /*----------------------------------------------------------------------*/
   if( is_selected() )
   {  m_new_name.Clear() ; m_s_val[ COL_NONE_SEL_NUM ].clear() ; }
   /*----------------------------------------------------------------------*/
   m_i_sel_num = -1 ;
   /*----------------------------------------------------------------------*/
}

/*--( A directory can't be selected in "recursive" mode )------------------*/
bool CFile::is_selectable() const
{
   /*----------------------------------------------------------------------*/
   return( !is_dir() || !wxGetApp().get_fl()->m_dir.get_recurse() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::compute_abs_path( const wxString &s_path ) const
{
   /*----------------------------------------------------------------------*/
   if( !wxIsAbsolutePath( s_path ) )
   {  return( sr::append_sep_to_path( m_p_cdir->get_dir() ) + s_path ) ; }
   /*----------------------------------------------------------------------*/
   return( s_path ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::get_abs_full_path() const
{
   /*----------------------------------------------------------------------*/
   return( sr::append_sep_to_path( get_abs_path() ) + get_full_path() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::get_abs_new_full_path() const
{
   /*----------------------------------------------------------------------*/
   return( sr::append_sep_to_path( get_abs_new_path() + get_new_full_path() )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::get_s_val( int i_col ) const
{
   /*----------------------------------------------------------------------*/
   static wxString st_s_empty ;
   /*----------------------------------------------------------------------*/
   switch( i_col )
   {
      /*-------------------------------------------------------------------*/
      case COL_NONE_NAME     : return( m_name.GetFullPath() ) ;
      case COL_NONE_NEW_NAME : return( m_new_name.GetFullPath() ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   t_s_val_cit it_val = m_s_val.find( i_col ) ;
   /*----------------------------------------------------------------------*/
   return( it_val == m_s_val.end() ? st_s_empty : it_val->second ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxLongLong CFile::get_ll_val( int i_col ) const
{
   /*----------------------------------------------------------------------*/
   static const wxLongLong st_ll_min = INT_MIN  ;
   t_ll_val_cit it_val = m_ll_val.find( i_col ) ;
   /*----------------------------------------------------------------------*/
   return( it_val == m_ll_val.end() ? st_ll_min : it_val->second ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
double CFile::get_do_val( int i_col ) const
{
   /*----------------------------------------------------------------------*/
   static const double st_do_min = - 1 * ( double )LLONG_MAX ;
   t_do_val_cit it_val = m_do_val.find( i_col ) ;
   /*----------------------------------------------------------------------*/
   return( it_val == m_do_val.end() ? st_do_min : it_val->second ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxDateTime CFile::get_dt_val( int i_col ) const
{
   /*----------------------------------------------------------------------*/
   static const wxDateTime st_dt_empty ;
   t_dt_val_cit it_val = m_dt_val.find( i_col ) ;
   /*----------------------------------------------------------------------*/
   return( it_val == m_dt_val.end() ? st_dt_empty : it_val->second ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::id3v1aud_equiv( int i_col_1, int i_col_2 ) const
{
   /*----------------------------------------------------------------------*/
   wxString s_res ;
   /*----------------------------------------------------------------------*/
   s_res = get_s_val( i_col_1 ) ;
   if( s_res.empty() && wxGetApp().M_boo_id3v1aud_nvl.get() )
   {  s_res = get_s_val( i_col_2 ) ; }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::get_s_size( const wxUniChar &c_unit ) const
{
   /*----------------------------------------------------------------------*/
   if( is_dir() && !wxGetApp().M_boo_dir_size_fl.get() )
   {  return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   switch( c_unit.GetValue() )
   {
      /*-------------------------------------------------------------------*/
      case 'b' :
         /*----------------------------------------------------------------*/
         return( m_ull_size.ToString() ) ;
      /*-------------------------------------------------------------------*/
      case 'k' :
         /*----------------------------------------------------------------*/
         return( wxString::Format( "%.2lf",
                                   sr::round_01(   m_ull_size.ToDouble()
                                                 / 1024.0
                                               )
                                 )
               ) ;
      /*-------------------------------------------------------------------*/
      case 'm' :
         /*----------------------------------------------------------------*/
         return( wxString::Format( "%.2lf",
                                   sr::round_01(    m_ull_size.ToDouble()
                                                  / 1024.0 / 1024.0
                                               )
                                 )
               ) ;
      /*-------------------------------------------------------------------*/
      case 'g' :
         /*----------------------------------------------------------------*/
         return( wxString::Format( "%.2lf",
                                   sr::round_01(   m_ull_size.ToDouble()
                                                 / 1024.0 / 1024.0 / 1024.0
                                               )
                                 )
               ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( sr::size_in_most_appropriate_unit( m_ull_size ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFile::get_s_duration( int i_col, const wxUniChar &c_fmt ) const
{
   /*----------------------------------------------------------------------*/
   static wxString st_s_empty ;
   t_ll_val_cit    it_val = m_ll_val.find( i_col ) ;
   /*----------------------------------------------------------------------*/
   if( it_val == m_ll_val.end() )
   {  return( st_s_empty ) ; }
   /*----------------------------------------------------------------------*/
   return( sr::duration_to_string( it_val->second.ToLong(), c_fmt ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--( "const" because the associated variables are declared "mutable" )----*/
wxString CFile::get_md5() const
{
   /*--( If the file is not supposed to exist it is seen as empty )--------*/
   if( is_fake() )
   {  return( CChecksumMD5().calc( wxEmptyString ) ) ; }

   /*--( The name can be invalid during the expand_var call )--------------*/
   if( is_dir() || !m_name.IsOk() ) { return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   wxDateTime dt_mod = m_name.GetModificationTime() ;

   /*--( Computation needed ? )--------------------------------------------*/
   if( m_s_md5.empty() || dt_mod > m_dt_md5 )
   {  wxLogNull no_log ;
      CChecksumMD5().calc( m_name, m_s_md5, g_disp_info_yield ) ;
      m_dt_md5 = dt_mod ;
   }
   /*----------------------------------------------------------------------*/
   return( m_s_md5 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( "const" because the associated variables are declared "mutable" )----*/
wxString CFile::get_sha1() const
{
   /*--( If the file is not supposed to exist it is seen as empty )--------*/
   if( is_fake() )
   {  return( CChecksumSHA1().calc( wxEmptyString ) ) ; }

   /*--( The name can be invalid during the expand_var call )--------------*/
   if( is_dir() || !m_name.IsOk() ) { return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   wxDateTime dt_mod = m_name.GetModificationTime() ;

   /*--( Computation needed ? )--------------------------------------------*/
   if( m_s_sha1.empty() || dt_mod > m_dt_sha1 )
   {  wxLogNull no_log ;
      CChecksumSHA1().calc( m_name, m_s_sha1, g_disp_info_yield ) ;
      m_dt_sha1 = dt_mod ;
   }
   /*----------------------------------------------------------------------*/
   return( m_s_sha1 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( "const" because the associated variables are declared "mutable" )----*/
wxString CFile::get_crc32() const
{
   /*--( If the file is not supposed to exist it is seen as empty )--------*/
   if( is_fake() )
   {  return( CChecksumCRC32().calc( wxEmptyString ) ) ; }

   /*--( The name can be invalid during the expand_var call )--------------*/
   if( is_dir() || !m_name.IsOk() ) { return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   wxDateTime dt_mod = m_name.GetModificationTime() ;

   /*--( Computation needed ? )--------------------------------------------*/
   if( m_s_crc32.empty() || dt_mod > m_dt_crc32 )
   {  wxLogNull no_log ;
      CChecksumCRC32().calc( m_name, m_s_crc32, g_disp_info_yield ) ;
      m_dt_crc32 = dt_mod ;
   }
   /*----------------------------------------------------------------------*/
   return( m_s_crc32 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Undo/Redo names associated to an operation number                         !
+--------------------------------------------------------------------------*/
void CFile::names_undo_redo( int i_oper,
                             wxString &s_undo, wxString &s_redo
                           ) const
{  /*----------------------------------------------------------------------*/
   t_undo_cit it_undo ;

   /*--( The name that should be used for an undo operation )--------------*/
   if( ( it_undo = m_undo.find( i_oper ) ) == m_undo.end() )
   {  s_undo.clear() ; }
   else
   {  s_undo = it_undo->second.from.GetFullName() ; }

   /*--( The name that should be used for a redo operation )---------------*/
   if( ( it_undo = m_undo.find( i_oper + 1 ) ) == m_undo.end() )
   {  s_redo.clear() ; }
   else
   {  s_redo = it_undo->second.to.GetFullName() ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFile::reload_info( bool boo_full )
{
   /*----------------------------------------------------------------------*/
   wxDir     dir( get_abs_path() ) ;
   wxString  s_name ;
   CFileInit file_init( *this ) ;
   wxLogNull no_log ;

   /*-----------------------------------------------------------------------+
   ! "Reread" of the file name (GetFirst) because some "interpretation"     !
   ! and reassignment may be done by the operating system.                  !
   ! For example the "....." at the end of a name are removed under MSW.    !
   +-----------------------------------------------------------------------*/
   if( !dir.IsOpened() || !dir.GetFirst( &s_name, get_full_name() ) )
   {  return( -1 ) ; }
   /*--( Concatenate current "subpath" )-----------------------------------*/
   s_name = wxFileName( get_path(), s_name ).GetFullPath() ;
   /*----------------------------------------------------------------------*/
   if( !boo_full )
   {  file_init.init_mini_name( s_name ) ; }
   else
   {
      /*--------------------------------------------------------------------+
      ! Cumulative information may have change. For example, if the         !
      ! extension changes, the duration may appear or disappear.            !
      +--------------------------------------------------------------------*/
      if( is_selected() )
      {
         /*--( Access to the parent container is needed )------------------*/
         if( m_p_cdir == NULL ) { wxFAIL ; return( -2 ) ; }
         /*----------------------------------------------------------------*/
         m_p_cdir->sub_sel_cumul_info( *this ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      file_init.init_full( wxEmptyString, s_name, is_dir() ) ;
      /*--( The reload has removed the selection number string )-----------*/
      if( is_selected() )
      {  init_s_sel_num() ;
         m_p_cdir->add_sel_cumul_info( *this ) ;
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFile::do_rename( wxFileName &new_name )
{
   /*----------------------------------------------------------------------*/
   reset_oper_err() ;

   /*----------------------------------------------------------------------*/
   wxString s_current_full_path( get_full_path() )    ;
   wxString s_new_full_path( new_name.GetFullPath() ) ;
   /*-----------------------------------------------------------------------+
   ! Under GNU/Linux renaming with the same name will overwrite the         !
   ! destination !                                                          !
   ! More generally, a clear "collision" message is needed.                 !
   +-----------------------------------------------------------------------*/
   if(    !s_current_full_path.IsSameAs( s_new_full_path,
                                         wxFileName::IsCaseSensitive()
                                       )
       && sr::name_already_used_on_disk( s_new_full_path )
     )
   {  wxLogError( _( "Failed to rename \"%s\" to \"%s\" "
                     "because \"%s\" already exists"
                   ),
                  s_current_full_path, s_new_full_path, s_new_full_path
                ) ;
      return( -1 ) ;
   }
   /*----------------------------------------------------------------------*/
   if( sr::force_path_existence( new_name ) != 0 )
   {  return( -2 ) ; }

   /*-----------------------------------------------------------------------+
   ! The renaming between directories is generally seen from the            !
   ! filesystemwatcher with delete/create events. They are not easy/nice to !
   ! handle. Therefore "true" ones are manually created with a specific ID. !
   ! As the filesystemwatcher sets paths as absolute this will permit to    !
   ! set the new name path exactly as it has been chosen by the user.       !
   +-----------------------------------------------------------------------*/
   if( get_path() != new_name.GetPath() )
   {
      /*-------------------------------------------------------------------*/
      wxQueueEvent( wxGetApp().m_frame->m_fswatcher,
                    new wxFileSystemWatcherEvent( wxFSW_EVENT_RENAME,
                                                  get_name(), new_name,
                                                  FSW_WID_REN_DIFF_PATH
                                                )
                  ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! If the rename fails a copy/delete sequence is tried.                   !
   ! This is mainly useful under MSW where moving directories from one      !
   ! volume to another doesn't work.                                        !
   +-----------------------------------------------------------------------*/
   bool boo_ok ;
   /*----------------------------------------------------------------------*/
   if( wxRename( s_current_full_path, s_new_full_path ) == 0 )
   {  boo_ok = true ; }
   else
   {  /*--( No log to avoid error messages with "copy" or "del" in them )--*/
      wxLogNull no_log ;
      bool boo_copy_ok   = false ;
      bool boo_delete_ok = false ;
      /*--( As this looks like a rename the timestamps are copied too )----*/
        boo_copy_ok
      = ( sr::copy_file_dir( s_current_full_path, s_new_full_path,
                             sr::CFT_DO_COPY_FILE_TMS
                           ) == 0
        ) ;
      /*-------------------------------------------------------------------*/
      if( boo_copy_ok )
      {  /*----------------------------------------------------------------*/
           boo_delete_ok
         = ( sr::delete_file_dir( s_current_full_path ) == 0 ) ;
         /*--( If the "delete" failed the "copy" has to be removed )-------*/
         if( !boo_delete_ok )
         {  sr::delete_file_dir( s_new_full_path ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      boo_ok = boo_copy_ok && boo_delete_ok ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !boo_ok )
   {  /*-------------------------------------------------------------------*/
      wxLogError( _( "Failed to rename \"%s\" to \"%s\"" ),
                  s_current_full_path, s_new_full_path
                ) ;
      return( -3 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFile::do_copy( wxFileName &new_name )
{
   /*----------------------------------------------------------------------*/
   reset_oper_err() ;
   /*----------------------------------------------------------------------*/
   wxString s_new_full_path( new_name.GetFullPath() ) ;
   /*----------------------------------------------------------------------*/
   if( sr::force_path_existence( new_name ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( sr::copy_file_dir( get_full_path(), s_new_full_path,
                          wxGetApp().M_boo_copy_file_tms.get()
                          ? sr::CFT_DO_COPY_FILE_TMS
                          : sr::CFT_DONT_COPY_FILE_TMS
                        ) != 0
     )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Use to delete highlighted files and to undo a copy )-----------------*/
int CFile::do_delete( wxFileName &name )
{
   /*----------------------------------------------------------------------*/
   if( sr::delete_file_dir( name.GetFullPath() ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Do the requested operation on the file : rename, copy                     !
! Return:   0 : Ok no problem                                               !
!         < 0 : Ko during copy/rename                                       !
!         > 0 : Ko useless operation (nothing to change)                    !
!                                                                           !
! new_name is not passed as "const" because MkDir which is called in        !
! "force_path_existence" is not defined "const".                            !
+--------------------------------------------------------------------------*/
int CFile::oper_do( int i_num, e_otfile ofile, wxFileName &new_name )
{
   /*----------------------------------------------------------------------*/
   if( !new_name.IsOk() ) { return( 1 ) ; }
   /*-----------------------------------------------------------------------+
   ! The name has changed ?                                                 !
   ! Using the "wxFileName" "==" operator can be a problem if the user      !
   ! wants only to change the case. MSW is "file name case insensitive" and !
   ! the "==" operator, of course, follows this rule.                       !
   ! Therefore a "string" comparison is made.                               !
   +-----------------------------------------------------------------------*/
   if( m_name.GetFullPath() == new_name.GetFullPath() ) { return( 2 ) ; }

   /*----------------------------------------------------------------------*/
   m_undo[ i_num ].from = m_name   ;
   m_undo[ i_num ].to   = new_name ;
   /*----------------------------------------------------------------------*/
   if(    ( ofile == OFILE_RENAME && do_rename( new_name ) != 0 )
       || ( ofile == OFILE_COPY   && do_copy( new_name ) != 0 )
     )
   {  m_undo.erase( i_num ) ; return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Undo the requested operation on the file : rename, copy                   !
! Return:  0 : Ok no problem                                                !
!         -1 : Ko during undo                                               !
!         +1 : Ko oper not found                                            !
+--------------------------------------------------------------------------*/
int CFile::oper_undo( int i_num, e_otfile ofile )
{
   /*----------------------------------------------------------------------*/
   t_undo_it it_undo ;
   /*--( Known operation ? )-----------------------------------------------*/
   if( ( it_undo = m_undo.find( i_num ) ) == m_undo.end() )
   {  return( 1 ) ; }
   /*----------------------------------------------------------------------*/
   if(    ( ofile == OFILE_RENAME && do_rename( it_undo->second.from ) != 0 )
       || ( ofile == OFILE_COPY   && do_delete( it_undo->second.to ) != 0  )
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Redo the requested operation on the file : rename, copy                   !
! Return:  0 : Ok no problem                                                !
!         -1 : Ko during undo                                               !
!         +1 : Ko oper not found                                            !
+--------------------------------------------------------------------------*/
int CFile::oper_redo( int i_num, e_otfile ofile )
{
   /*----------------------------------------------------------------------*/
   t_undo_it it_undo ;
   /*--( Known operation ? )-----------------------------------------------*/
   if( ( it_undo = m_undo.find( i_num ) ) == m_undo.end() )
   {  return( 1 ) ; }
   /*----------------------------------------------------------------------*/
   if(    ( ofile == OFILE_RENAME && do_rename( it_undo->second.to ) != 0 )
       || ( ofile == OFILE_COPY   && do_copy( it_undo->second.to ) != 0 )
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                        End of File CFile_Base.cpp                         !
+==========================================================================*/
