/*====================================+=====================================+
! File sr_lib.cpp                     ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/wx.h>
#include <wx/clipbrd.h>
#include <wx/msgdlg.h>
#include <wx/dir.h>
#include <wx/dcclient.h>
#include <wx/sizer.h>
#include <wx/menu.h>
#include <wx/log.h>
#include "common/sr_lib.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
static bool st_end_of_subfilter( const wxChar *p )
{  return( *p == '\0' || *p == ';' ) ; }

/*--------------------------------------------------------------------------+
! File filter: *.cpp   *.h;*.cpp      ... etc ...                           !
!                                                                           !
! The format is the one of the UNIX sh (with the meta characters: *?[!-]\   !
! Many subfilters can be used "simultaneously" by separating them with ';'  !
!                                                                           !
! *   : any character string                                                !
! ?   : any character                                                       !
! [!-]: set of allowed characters. Or refused with '!'                      !
!       A '-' at the start or end specifies that the '-' that is searched   !
! \   : escape character                                                    !
!                                                                           !
! The third parameter is only used during recursion calls                   !
!                                                                           !
! The match is case insensitive.                                            !
!                                                                           !
+--------------------------------------------------------------------------*/
static bool st_match_filter( const wxChar *p_c_str, const wxChar *p_c_fil,
                             bool boo_stop_on_subfilter
                           )
{
   /*----------------------------------------------------------------------*/
   bool         boo_escape              = false   ;
   bool         boo_goto_next_subfilter = false   ;
   const wxChar *p_c_str_sav            = p_c_str ;

   /*--( No filter means ok )----------------------------------------------*/
   if( *p_c_fil == '\0' ) { return( true ) ; }
   /*----------------------------------------------------------------------*/
   while(    *p_c_fil != '\0'
          && ( *p_c_fil != ';' || !boo_stop_on_subfilter )
        )
   {
      /*--( Escape the next character ? )----------------------------------*/
      if( *p_c_fil == '\\' && !boo_escape )
      {  boo_escape = true ; ++p_c_fil ; }
      else /*--( End of subfilter ? )--------------------------------------*/
      if( *p_c_fil == ';' && !boo_escape )
      {
         /*----------------------------------------------------------------*/
         if( *p_c_str == '\0' ) { return( true ) ; }
         boo_goto_next_subfilter = true ;
         /*----------------------------------------------------------------*/
      }
      else /*--( Any string ? )--------------------------------------------*/
      if( *p_c_fil == '*' && !boo_escape )
      {
         /*--( '*' at the end of (sub)filter => ok )-----------------------*/
         ++p_c_fil ;
         if( st_end_of_subfilter( p_c_fil ) ) { return( true ) ; }

         /*--( '*.' => no extension. *.* => all )--------------------------*/
#ifdef __WXMSW__
         if(    *p_c_fil == '.'
             && (    (    st_end_of_subfilter( p_c_fil + 1 )
                       && wxStrchr( p_c_str, _T( '.' ) ) == NULL
                     )
                  || (    *( p_c_fil + 1 ) == '*'
                       && st_end_of_subfilter( p_c_fil + 2 )
                     )
                )
           )
         {  return( true ) ; }
#endif // __WXMSW__

         /*--( Search on all substrings )----------------------------------*/
         for( ; *p_c_str != '\0' ; ++p_c_str )
         {  /*-------------------------------------------------------------*/
            if( st_match_filter( p_c_str, p_c_fil, true ) )
            {  return( true ) ; }
            /*-------------------------------------------------------------*/
         }
         /*--( No result ... check the next subfilter )--------------------*/
         boo_goto_next_subfilter = true ;
         /*----------------------------------------------------------------*/
      }
      else /*--( Any character ? )-----------------------------------------*/
      if( *p_c_fil == '?' && !boo_escape )
      {
         /*----------------------------------------------------------------*/
         if( *p_c_str == '\0' )
         {  boo_goto_next_subfilter = true ; }
         else
         {  ++p_c_fil ; ++p_c_str ; }
         /*----------------------------------------------------------------*/
      }
      else /*--( Set of authorized/forbidden characters )------------------*/
      if( *p_c_fil == '[' && !boo_escape )
      {
         /*----------------------------------------------------------------*/
         bool  boo_expected_res ;
         bool  boo_found ;
         const wxUChar uc_chr_up = wxToupper( *p_c_str ) ;

         /*--( Invert the result ? )---------------------------------------*/
         if( *++p_c_fil == '!' )
         {  boo_expected_res = false ; ++p_c_fil ; }
         else
         {  boo_expected_res = true ; }
         /*----------------------------------------------------------------*/
         for( boo_found = false ;
              *p_c_fil != '\0' && *p_c_fil != ']' && !boo_found ;
              ++p_c_fil
            )
         {  /*-------------------------------------------------------------*/
            if( *p_c_fil != '-' )
            {    boo_found
               = ( uc_chr_up == ( wxUChar )wxToupper( *p_c_fil ) ) ;
            }
            else
            {  /*--( '-' at start or end => character to test )------------*/
               if(    *( p_c_fil - 1 ) == '['
                   || *( p_c_fil - 1 ) == '!'
                   || *( p_c_fil + 1 ) == ']'
                 )
               {  boo_found = ( uc_chr_up == '-' ) ; }
               else
               {  /*--( In the interval ? )--------------------------------*/
                    boo_found
                  = (    uc_chr_up >= ( wxUChar )wxToupper( *( p_c_fil-1 ) )
                      && uc_chr_up <= ( wxUChar )wxToupper( *( p_c_fil+1 ) )
                    ) ;
                  /*--( Jump after the interval )--------------------------*/
                  ++p_c_fil ;
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         /*--( Search of the end of the set )------------------------------*/
         while( *p_c_fil != ']' && !st_end_of_subfilter( p_c_fil ) )
         {  ++p_c_fil ; }
         /*----------------------------------------------------------------*/
         if( *p_c_fil == ']' ) { ++p_c_fil ; ++p_c_str ; }

         /*--( End of set reached means nothing found )--------------------*/
         if( boo_found != boo_expected_res )
         {  boo_goto_next_subfilter = true ; }
         /*----------------------------------------------------------------*/
      }
      else /*--( Else the comparison is done upper-case )------------------*/
      if( wxToupper( *p_c_str++ ) != wxToupper( *p_c_fil++ ) )
      {  boo_goto_next_subfilter = true ; }
      else
      {  boo_escape = false ; }

      /*-------------------------------------------------------------------*/
      if( boo_goto_next_subfilter )
      {
         /*--( The string has to be fully rechecked )----------------------*/
         p_c_str = p_c_str_sav ;
         /*----------------------------------------------------------------*/
         while( !st_end_of_subfilter( p_c_fil ) ) { ++p_c_fil ; }
         /*----------------------------------------------------------------*/
         if( *p_c_fil == ';' && !boo_stop_on_subfilter )
         {  ++p_c_fil ; }
         /*----------------------------------------------------------------*/
         boo_goto_next_subfilter = false ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( *p_c_str == '\0' ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool sr::filename_match_filter( const wxString &s_name,
                                const wxString &s_fil
                              )
{
   /*----------------------------------------------------------------------*/
   if( s_fil.empty() ) { return( true ) ; }
   /*----------------------------------------------------------------------*/
#ifdef __WINDOWS__
   if( s_fil == "*.*" ) { return( true ) ; }
#elif defined( __UNIX__ )
   if( s_fil == "*" ) { return( true ) ; }
#endif
   /*----------------------------------------------------------------------*/
   return( st_match_filter( s_name, s_fil, false ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::set_font_fix( wxWindow *p_win, bool boo_fix )
{
   /*----------------------------------------------------------------------*/
   wxFont cur_font( p_win->GetFont() ) ;
   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   wxFont win_font( wxSystemSettings::GetFont(
                    boo_fix ? wxSYS_ANSI_FIXED_FONT : wxSYS_DEFAULT_GUI_FONT
                                             )
                  ) ;
   win_font.SetPointSize( cur_font.GetPointSize() ) ;
   win_font.SetStyle( cur_font.GetStyle() ) ;
   win_font.SetWeight( cur_font.GetWeight() ) ;
   win_font.SetUnderlined( cur_font.GetUnderlined() ) ;
#else
   wxFont win_font( cur_font.GetPointSize(),
                    boo_fix ? wxFONTFAMILY_TELETYPE : wxFONTFAMILY_DEFAULT,
                    cur_font.GetStyle(), cur_font.GetWeight(),
                    cur_font.GetUnderlined(),
                    wxEmptyString, wxFONTENCODING_DEFAULT
                  ) ;
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
   p_win->SetFont( win_font ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function will return the nth directory (right counted) of the path   !
! to the file. The name can be relative or absolute. If it is relative, the !
! absolute path can be used as complement.                                  !
+--------------------------------------------------------------------------*/
wxString sr::dir_from_right( const wxFileName &name, int i_num_parent,
                             const wxString &s_absolute_path_to_name
                           )
{
   /*----------------------------------------------------------------------*/
   int i_ind ;
   /*----------------------------------------------------------------------*/
   if( i_num_parent <= 0 ) { return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   i_ind = name.GetDirs().GetCount() - i_num_parent ;
   /*--( Enough dirs in the name ? )---------------------------------------*/
   if( i_ind >= 0 ) { return( name.GetDirs()[ i_ind ] ) ; }
   /*--( Not ... ok if the path is absolute its useless to go further )----*/
   if( name.IsAbsolute() )
   {  return( wxEmptyString ) ; }
   /*--( Try to extract the dir from the path )----------------------------*/
   {
      /*--( Save some CPU by defining path in a block )--------------------*/
      wxFileName path( s_absolute_path_to_name, wxEmptyString ) ;
      /*-------------------------------------------------------------------*/
        i_ind
      = path.GetDirs().GetCount() + name.GetDirs().GetCount() - i_num_parent;
      if( i_ind >= 0 ) { return( path.GetDirs()[ i_ind ] ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( wxEmptyString ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::absolute_path( const wxFileName &name, const wxString &s_path )
{
   /*----------------------------------------------------------------------*/
   if( name.IsAbsolute() ) { return( name.GetPathWithSep() ) ; }
   /*----------------------------------------------------------------------*/
   wxFileName path( name ) ;
   path.Normalize( wxPATH_NORM_ABSOLUTE, s_path ) ;
   /*--( The calling function expects last char to be a path separator )---*/
   return( path.GetPathWithSep() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::size_in_most_appropriate_unit( const wxULongLong &ull_size )
{
   /*----------------------------------------------------------------------*/
   wxString s_res ;
   /*----------------------------------------------------------------------*/
   if( ull_size >= wxULL( 1024 * 1024 * 1024 ) )
   {  s_res.Printf( _( "%.2lf GiB" ),
                    sr::round_01(   ull_size.ToDouble()
                                  / 1024.0 / 1024.0 / 1024.0
                                )
                  ) ;
   }
   else
   if( ull_size >= wxULL( 1024 * 1024 ) )
   {  s_res.Printf( _( "%.2lf MiB" ),
                    sr::round_01( ull_size.ToDouble() / 1024.0 / 1024.0 )
                  ) ;
   }
   else
   if( ull_size >= wxULL( 1024 ) )
   {  s_res.Printf( _( "%.2lf KiB" ),
                    sr::round_01( ull_size.ToDouble() / 1024.0 )
                  ) ;
   }
   else
   {  s_res << ull_size ; }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Round 0 decimal )----------------------------------------------------*/
int sr::round_0( double do_val )
{
   /*----------------------------------------------------------------------*/
   return( do_val + 0.50 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Round 1 decimal (tenth) )--------------------------------------------*/
double sr::round_1( double do_val )
{
   /*----------------------------------------------------------------------*/
   return( floor( do_val * 10 + 0.50 ) / 10.0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Round 2 decimals (hundredth) )---------------------------------------*/
double sr::round_01( double do_val )
{
   /*----------------------------------------------------------------------*/
   return( floor( do_val * 100 + 0.50 ) / 100.0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Round 3 decimals (thousandth) )--------------------------------------*/
double sr::round_001( double do_val )
{
   /*----------------------------------------------------------------------*/
   return( floor( do_val * 1000 + 0.50 ) / 1000.0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::set_clipboard( const wxString &s )
{
   /*----------------------------------------------------------------------*/
   if( wxTheClipboard->Open() )
   {
      /*-------------------------------------------------------------------*/
      wxTheClipboard->SetData( new wxTextDataObject( s ) ) ;
      wxTheClipboard->Close() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Parsing a date string supposed to be "ISO" formatted:                     !
! "YYYY-MM-DDTHH:MM:SS".                                                    !
! Why not only use the "wxDateTime::ParseISOCombined" function ?            !
! Because if some characters are added to the string the parsing will fail. !
! For example: the "Z" or the number of ms (.131) at the end.               !
+--------------------------------------------------------------------------*/
int sr::init_datetime_wxiso( const wxString &s_date, wxDateTime &dt )
{
   /*----------------------------------------------------------------------*/
   wxString s_date_truncated( s_date ) ;
   /*--( Remove useless chars )--------------------------------------------*/
   if( s_date_truncated.length() > 19 )
   {  s_date_truncated.Remove( 19 ) ; }
   /*----------------------------------------------------------------------*/
   return( dt.ParseISOCombined( s_date_truncated ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Specific function to init a datetime because if the values are invalid    !
! the "Set" can trigger an ASSERT                                           !
! month is supposed to be expressed: 1 - 12                                 !
+--------------------------------------------------------------------------*/
int sr::set_datetime( size_t sz_year, size_t sz_month , size_t sz_day,
                      size_t sz_hour, size_t sz_minute, size_t sz_second,
                      size_t sz_millisec, wxDateTime &dt
                    )
{
   /*--( wxDateTime functions define months as an enum starting at 0 )-----*/
   --sz_month ;
   /*--( Check )-----------------------------------------------------------*/
   if(    sz_month > wxDateTime::Dec
       ||   sz_day
          > wxDateTime::GetNumberOfDays( ( wxDateTime::Month )sz_month,
                                         sz_year
                                       )
       || sz_hour > 23 || sz_minute > 59 || sz_second > 61
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   dt.Set( sz_day, ( wxDateTime::Month )sz_month, sz_year,
           sz_hour, sz_minute, sz_second, sz_millisec
         ) ;
   /*----------------------------------------------------------------------*/
   return( dt.IsValid() ? 0 : -2 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Parsing a date string following the order: year month day hour min sec    !
! Why not use "wxDateTime::ParseFormat" ?                                   !
! Mainly because if some elements are missing from the string, it fails.    !
+--------------------------------------------------------------------------*/
int sr::init_date_ymdhms( const wxString &s_date,
                          const char *p_c_format, wxDateTime &dt
                        )
{
   /*----------------------------------------------------------------------*/
   int i_year, i_month, i_day, i_hour, i_minute, i_second ;
   int i_nb ;
   /*----------------------------------------------------------------------*/
   i_nb = wxSscanf( s_date, p_c_format,
                    &i_year, &i_month , &i_day,
                    &i_hour, &i_minute, &i_second
                  ) ;
   /*----------------------------------------------------------------------*/
   switch( i_nb )
   {
      /*--( Not "breaking" cases will allow to init only what is needed )--*/
      case  1 : i_month  = 1 ;
      case  2 : i_day    = 1 ;
      case  3 : i_hour   = 0 ;
      case  4 : i_minute = 0 ;
      case  5 : i_second = 0 ;
      case  6 : break ;
      /*-------------------------------------------------------------------*/
      default : return( -1 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( sr::set_datetime( i_year, i_month , i_day,
                             i_hour, i_minute, i_second, 0, dt
                           )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::init_datetime( time_t t, wxDateTime &dt )
{
   /*----------------------------------------------------------------------*/
   dt.Set( t ) ;
   /*----------------------------------------------------------------------*/
   return( dt.IsValid() ? 0 : -1 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
#ifdef __WXMSW__
int sr::init_datetime( const FILETIME &ft, wxDateTime &dt )
{
   /*----------------------------------------------------------------------*/
   if( ft.dwLowDateTime == 0 && ft.dwHighDateTime == 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   FILETIME   ft_local ;
   SYSTEMTIME st ;
   /*----------------------------------------------------------------------*/
   if(    ::FileTimeToLocalFileTime( &ft, &ft_local ) != 0
       && ::FileTimeToSystemTime( &ft_local, &st ) != 0
     )
   {  dt.Set( st.wDay, ( wxDateTime::Month )( st.wMonth - 1 ), st.wYear,
              st.wHour, st.wMinute, st.wSecond, st.wMilliseconds
            ) ;
   }
   /*----------------------------------------------------------------------*/
   return( dt.IsValid() ? 0 : -2 ) ;
   /*----------------------------------------------------------------------*/
}
#endif // __WXMSW__

/*--------------------------------------------------------------------------+
! Converts a signed number of seconds to a string.                          !
! The second paramater defines the format to use :                          !
! h : HMS "forced"                                                          !
! m : MS "forced"                                                           !
! s : S "forced"                                                            !
! + : Same as 'h' but signed when positive                                  !
!                                                                           !
! By default positive values are not signed and "0" left parts are trimmed. !
+--------------------------------------------------------------------------*/
wxString sr::duration_to_string( int i_nb_sec, const wxUniChar &c_fmt )
{
   /*----------------------------------------------------------------------*/
   wxString s_res ;
   bool     boo_negative = ( i_nb_sec < 0 ) ;
   /*----------------------------------------------------------------------*/
   if( boo_negative ) { i_nb_sec = - i_nb_sec ; }

   /*--( Forced format ? )-------------------------------------------------*/
   switch( c_fmt.GetValue() )
   {
      /*-------------------------------------------------------------------*/
      case '+' :
      case 'h' :
         s_res.Printf( _( "%dh%02dm%02ds" ),
                       i_nb_sec / 3600, ( i_nb_sec / 60 ) % 60, i_nb_sec % 60
                     ) ;
         break ;
      /*-------------------------------------------------------------------*/
      case 'm' :
         s_res.Printf( _( "%dm%02ds" ), i_nb_sec / 60, i_nb_sec % 60 ) ;
         break ;
      /*-------------------------------------------------------------------*/
      case 's' :
         s_res.Printf( _( "%ds" ), i_nb_sec ) ;
         break ;
      /*-------------------------------------------------------------------*/
      default :
      {
         /*--( Default format : trimed on the left )-----------------------*/
         int i_h = i_nb_sec / 3600 ;
         int i_m = ( i_nb_sec / 60 ) % 60 ;
         int i_s = i_nb_sec % 60 ;
         /*----------------------------------------------------------------*/
         if( i_h > 0 )
         {  s_res.Printf( "%dh%02dm%02ds", i_h, i_m, i_s ) ; }
         else
         if( i_m > 0 )
         {  s_res.Printf( "%dm%02ds", i_m, i_s ) ; }
         else
         {  s_res.Printf( "%ds", i_s ) ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( boo_negative )
   {  s_res.insert( 0, '-' ) ; }
   else
   if( c_fmt == '+' )
   {  s_res.insert( 0, '+' ) ; }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::access_error_message( const wxString &s_element )
{  wxMessageBox( wxString::Format( _( "Impossible to access to \"%s\"" ),
                                   s_element
                                 ),
                 _( "Error" ), wxOK | wxICON_ERROR
               ) ;
}

/*-------------------------------------------------------------------------*/
void sr::error_message( const wxString &s_msg )
{  wxMessageBox( s_msg, _( "Error" ), wxOK | wxICON_ERROR ) ; }

/*-------------------------------------------------------------------------*/
void sr::info_message( const wxString &s_msg )
{  wxMessageBox( s_msg, _( "Information" ), wxOK | wxICON_INFORMATION ) ; }

/*-------------------------------------------------------------------------*/
bool sr::is_file_attr_hidden( const wxString &s_full_path )
{
   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   /*-----------------------------------------------------------------------+
   ! Under MSW the disk root directories are at least considered:           !
   ! Hidden and System. This sounds false from Siren's point of view ...    !
   +-----------------------------------------------------------------------*/
   return(    s_full_path.empty()
           || (    s_full_path.Last() != wxFileName::GetPathSeparator()
                && (    ::GetFileAttributes( s_full_path.fn_str() )
                      & ( FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM )
                   ) != 0
              )
         ) ;
#elif defined( __UNIX__ )
   /*--( Hidden filename/dir starts in path with "." or "/." )-------------*/
   return(    s_full_path.empty()
           || s_full_path[ 0 ] == '.'
           || s_full_path.find(   wxString( wxFileName::GetPathSeparator() )
                                + '.'
                              ) != wxString::npos
         ) ;
#else
#error Unknown platform to: test get file attribute
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::force_path_existence( wxFileName &name )
{
   /*----------------------------------------------------------------------*/
   if(    !name.DirExists()
       && !name.Mkdir( 0777, wxPATH_MKDIR_FULL )
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool sr::name_already_used_on_disk( const wxString &s_name )
{
   /*----------------------------------------------------------------------*/
   wxStructStat struct_stat ;
   /*--( wxStat doesn't seem to be an "official" function )----------------*/
   return( wxStat( s_name, &struct_stat ) == 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::append_sep_to_path( const wxString &s_name )
{
   /*--( Add the separator only if necessary )-----------------------------*/
   if( s_name.empty() || !wxFileName::IsPathSeparator( s_name.Last() ) )
   {  return( s_name + wxFileName::GetPathSeparator() ) ; }
   /*----------------------------------------------------------------------*/
   return( s_name ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This won't have the same effect depending on the platform :               !
! - MSW  : all field can be changed                                         !
! - UNIX : creation time won't be changed                                   !
+--------------------------------------------------------------------------*/
int sr::copy_file_timestamp( const wxString &s_from, const wxString &s_to )
{
   /*----------------------------------------------------------------------*/
   wxDateTime dt_acc ;
   wxDateTime dt_mod ;
   wxDateTime dt_cre ;
   /*----------------------------------------------------------------------*/
   if( !wxFileName( s_from ).GetTimes( &dt_acc, &dt_mod, &dt_cre ) )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( !wxFileName( s_to ).SetTimes( &dt_acc, &dt_mod, &dt_cre ) )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::delete_file_dir( const wxString &s_name )
{
   /*--( Directory ? Delete it recursively )-------------------------------*/
   if( wxDir::Exists( s_name ) )
   {  return( wxFileName::Rmdir( s_name,
                                 wxPATH_RMDIR_FULL | wxPATH_RMDIR_RECURSIVE
                               )
              ? 0 : -1
            ) ;
   }
   /*--( It's a file ... )-------------------------------------------------*/
   return( wxRemoveFile( s_name ) ? 0 : -1 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::copy_file_dir( const wxString &s_from, const wxString &s_to,
                       e_cft copy_tms
                     )
{
   /*--( File ? just copy it )---------------------------------------------*/
   if( !wxDir::Exists( s_from ) )
   {
      /*-------------------------------------------------------------------*/
      if( !wxCopyFile( s_from, s_to, false ) )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      if(    copy_tms == sr::CFT_DO_COPY_FILE_TMS
          && sr::copy_file_timestamp( s_from, s_to ) != 0
        )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      return( 0 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( It is a directory. Recreate the structure )-----------------------*/
   wxDir dir( s_from ) ;

   /*----------------------------------------------------------------------*/
   if( !dir.IsOpened() ) { return( -3 ) ; }

   /*--( First create the destination directory )--------------------------*/
   if( !wxMkdir( s_to ) ) { return( -4 ) ; }

   /*----------------------------------------------------------------------*/
   wxString s_from_path ;
   wxString s_to_path   ;
   wxString s_elmt      ;

   /*----------------------------------------------------------------------*/
   s_from_path = sr::append_sep_to_path( s_from ) ;
   s_to_path   = sr::append_sep_to_path( s_to ) ;

   /*--( Then recursively copy the dir/files )-----------------------------*/
   if( dir.GetFirst( &s_elmt ) )
   {  /*-------------------------------------------------------------------*/
      do
      {  /*----------------------------------------------------------------*/
         if( sr::copy_file_dir( s_from_path + s_elmt, s_to_path + s_elmt,
                                copy_tms
                              ) != 0
           )
         {  return( -5 ) ; }
         /*----------------------------------------------------------------*/
      } while( dir.GetNext( &s_elmt ) ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( MSW only. Uniform Naming Convention: \\10.237.222.222\dir ... )------*/
bool sr::is_unc_path( const wxString &s_path )
{  /*----------------------------------------------------------------------*/
   return(    s_path.length() > 3
           && s_path[ 0 ] == '\\' && s_path[ 1 ] == '\\'
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::expand_path( const wxString &s_path )
{
   /*--( Normalize can display error messages. For ex: ".." when in root )-*/
   wxLogNull  no_log ;
   wxFileName fn( s_path, wxEmptyString ) ;
   /*-----------------------------------------------------------------------+
   ! Associate "DOTS" to "ABSOLUTE" is necessary. It adds the current       !
   ! directory and so enables to treat ".." for example                     !
   +-----------------------------------------------------------------------*/
   if( !fn.Normalize(   wxPATH_NORM_ENV_VARS
                      | wxPATH_NORM_TILDE
                      | wxPATH_NORM_ABSOLUTE
                      | wxPATH_NORM_DOTS
                    )
     )
   {  return( wxEmptyString ) ; }
   /*----------------------------------------------------------------------*/
   return( fn.GetPath() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::attach_win_to_panel( wxWindow *p_win, wxPanel *p_panel )
{
   /*----------------------------------------------------------------------*/
   wxBoxSizer *box_sizer = new wxBoxSizer( wxVERTICAL ) ;
   box_sizer->Add( p_win, 1, wxEXPAND ) ;
   p_panel->SetSizer( box_sizer ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! wxMemoryBuffer is not "copy on write". This function will make a          !
! complete copy of the object including the data part.                      !
+--------------------------------------------------------------------------*/
wxMemoryBuffer sr::memorybuffer_copy( const wxMemoryBuffer &mb )
{
   /*----------------------------------------------------------------------*/
   wxMemoryBuffer mb_copy ;
   /*----------------------------------------------------------------------*/
   memcpy( mb_copy.GetWriteBuf( mb.GetDataLen() ),
           mb.GetData(), mb.GetDataLen()
         ) ;
   mb_copy.UngetWriteBuf( mb.GetDataLen() ) ;
   /*----------------------------------------------------------------------*/
   return( mb_copy ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                           End of file sr_lib.cpp                          !
+==========================================================================*/
