/*====================================+=====================================+
! File sr_ctxmnu.cpp                  ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifdef __WXMSW__
/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "common/msw/sr_ctxmnu.h"
#include <wx/msw/private.h>
#include <shlobj.h>
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
static LRESULT CALLBACK st_contextual_menu_shell_wndproc( HWND hWnd,
                                                          UINT message,
                                                          WPARAM wParam,
                                                          LPARAM lParam
                                                        )
{
   /*----------------------------------------------------------------------*/
   LPCONTEXTMENU2 p_cm2
   = reinterpret_cast<LPCONTEXTMENU2>( wxGetWindowUserData( hWnd ) ) ;
   /*----------------------------------------------------------------------*/
   if( p_cm2 == NULL )
   {  return( DefWindowProc( hWnd, message, wParam, lParam ) ) ; }
   /*----------------------------------------------------------------------*/
   switch( message )
   {
      /*-------------------------------------------------------------------*/
      case WM_MENUCHAR :
      case WM_DRAWITEM :
      case WM_MEASUREITEM :
      case WM_INITMENUPOPUP :
         /*----------------------------------------------------------------*/
         p_cm2->HandleMenuMsg( message, wParam, lParam ) ;
         /*----------------------------------------------------------------*/
         return( message == WM_INITMENUPOPUP ? 0 : TRUE ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( DefWindowProc( hWnd, message, wParam, lParam ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_exec_shell_contextual_menu( const LPSHELLFOLDER p_dir  ,
                                          const LPITEMIDLIST  *p_pidl,
                                          int i_nb_pidl
                                        )
{
   /*----------------------------------------------------------------------*/
   LPCONTEXTMENU       pcm              = NULL   ;
   LPCONTEXTMENU2      pcm2             = NULL   ;
   HMENU               hmenu            = NULL   ;
   HRESULT             hres                      ;
   UINT                u_choice                  ;
   CMINVOKECOMMANDINFO inv_cmd                   ;
   const int           ci_min_id_shell  =      1 ;
   const int           ci_max_id_shell  = 0x7FFF ;
   POINT               pt                        ;
   HWND                hwnd_menu_parent = NULL   ;
   int                 i_ret            = 0      ;

   /*--( Creation of the menu in which the options will be inserted )------*/
   if( ( hmenu = CreatePopupMenu() ) == NULL ) { return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! Get the contextual menu interface                                      !
   ! The "3" and the "2" can handle the filling of the "Send To" menu       !
   ! and maybe other things ...                                             !
   +-----------------------------------------------------------------------*/
   hres = p_dir->GetUIObjectOf( NULL, i_nb_pidl, ( LPCITEMIDLIST * )p_pidl,
                                IID_IContextMenu, NULL,
                                ( void ** )&pcm
                              ) ;
   if( FAILED( hres ) ) { i_ret = -2 ; goto func_end ; }
   /*----------------------------------------------------------------------*/
   pcm2 = NULL ;
   /*--( The 3 or the 2 ? )------------------------------------------------*/
   if(   SUCCEEDED( pcm->QueryInterface( IID_IContextMenu3, (void **)&pcm2 ))
      || SUCCEEDED( pcm->QueryInterface( IID_IContextMenu2, (void **)&pcm2 ))
     )
   {  pcm->Release() ; pcm = pcm2 ; }

   /*--( Fill the menu )---------------------------------------------------*/
   hres = pcm->QueryContextMenu( hmenu, 0, ci_min_id_shell, ci_max_id_shell,
                                 CMF_EXPLORE
                               ) ;
   if( FAILED( hres ) ) { i_ret = -3 ; goto func_end ; }

   /*--( The "parent" window )---------------------------------------------*/
     hwnd_menu_parent
   = CreateWindow( _T( "STATIC" ), NULL, 0, 0, 0, 0, 0, NULL, NULL,
                   wxGetInstance(), NULL
                 ) ;

   /*--( Intercept the messages specific to contextual menu 2 and 3 )------*/
   if( pcm2 != NULL )
   {
      /*-------------------------------------------------------------------*/
      wxSetWindowUserData( hwnd_menu_parent, pcm2 ) ;
      wxSetWindowProc( hwnd_menu_parent, st_contextual_menu_shell_wndproc ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( The menu is displayed at the position of the mouse cursor )-------*/
   GetCursorPos( &pt ) ;
   /*----------------------------------------------------------------------*/
   u_choice = TrackPopupMenu( hmenu, // handle to shortcut menu
                              TPM_RETURNCMD | TPM_LEFTALIGN,
                              pt.x,   // horizontal pos, in screen coord
                              pt.y,   // vertical pos, in screen coord
                              0,      // reserved, must be zero
                              hwnd_menu_parent, // handle to owner window
                              NULL    // ignored
                            ) ;

   /*--( Quit if no choice made )------------------------------------------*/
   if( u_choice == 0 ) { goto func_end ; }

   /*-----------------------------------------------------------------------+
   ! A problem that SEEMS to happen only under Win2000                      !
   ! Calling PASTE may cause a problem, I've noticed it only with this      !
   ! command.                                                               !
   ! An "addref" seems enough !                                             !
   +-----------------------------------------------------------------------*/
   if( u_choice - ci_min_id_shell == 26 /*IDM_PASTE*/ ) { pcm->AddRef() ; }

   /*----------------------------------------------------------------------*/
   ZeroMemory( &inv_cmd, sizeof( inv_cmd ) ) ;
   inv_cmd.cbSize = sizeof( inv_cmd ) ;
   inv_cmd.hwnd   = hwnd_menu_parent ;
   inv_cmd.lpVerb = ( char * )MAKEINTRESOURCE( u_choice - ci_min_id_shell ) ;
   inv_cmd.nShow  = SW_SHOWNORMAL ;
   /*--( Run it ! )--------------------------------------------------------*/
   hres = pcm->InvokeCommand( ( CMINVOKECOMMANDINFO * )&inv_cmd ) ;
   /*----------------------------------------------------------------------*/
func_end :
   /*----------------------------------------------------------------------*/
   if( hwnd_menu_parent != NULL ) { DestroyWindow( hwnd_menu_parent ) ; }
   if( hmenu != NULL ) { DestroyMenu( hmenu ) ; }
   if( pcm   != NULL ) { pcm->Release()       ; }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Conversion of a file name to its PIDL )------------------------------*/
static int st_conv_name_pidl( const wxString &s_name,
                              LPSHELLFOLDER p_folder, LPITEMIDLIST  &pidl
                            )
{
   /*----------------------------------------------------------------------*/
   ULONG   u_lg ;
   HRESULT hres ;
   /*----------------------------------------------------------------------*/
   if( s_name.empty() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   hres = p_folder->ParseDisplayName( NULL, NULL, ( WCHAR * )s_name.wc_str(),
                                      &u_lg, &pidl, NULL
                                    ) ;
   if( FAILED( hres ) ) { return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The file names in the array can contain relative or absolute paths.       !
! When relative, the default path is used.                                  !
! As all the files have to belong to the same directory, specific           !
! ".._dir_menu" variables are used.                                         !
+--------------------------------------------------------------------------*/
int sr::shell_contextual_menu( const wxString &s_dir_def,
                               const wxArrayString &as_file
                             )
{
   /*----------------------------------------------------------------------*/
   LPMALLOC      pMalloc       = NULL ;
   LPSHELLFOLDER pDesktop      = NULL ;
   LPSHELLFOLDER p_dir_menu    = NULL ;
   wxString      s_dir_menu           ;
   LPITEMIDLIST  pidl_dir_menu = NULL ;
   LPITEMIDLIST  *p_tb_pidl    = NULL ;
   int           i_ret         = 0    ;
   HRESULT       hres                 ;
   int           i_num                ;

   /*--( Nothing ? )-------------------------------------------------------*/
   if( as_file.GetCount() == 0 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   hres = SHGetMalloc( &pMalloc ) ;
   if( FAILED( hres ) ) { i_ret = -11 ; goto func_end ; }
   /*----------------------------------------------------------------------*/
   hres = SHGetDesktopFolder( &pDesktop ) ;
   if( FAILED( hres ) ) { i_ret = -12 ; goto func_end ; }

   /*--( Create an array of IDs )------------------------------------------*/
   p_tb_pidl = new LPITEMIDLIST [ as_file.GetCount() ] ;
   if( p_tb_pidl == NULL ) { i_ret = -13 ; goto func_end ; }

   /*--( Store the IDs of the files into this array )----------------------*/
   for( i_num = 0 ; i_num < ( int )as_file.GetCount() ; ++i_num )
   {
      /*--( Initialize directory information on the first file )-----------*/
      if( i_num == 0 )
      {
         /*----------------------------------------------------------------*/
         s_dir_menu = sr::absolute_path( as_file[ i_num ], s_dir_def ) ;
         /*----------------------------------------------------------------*/
         if( st_conv_name_pidl( s_dir_menu, pDesktop, pidl_dir_menu ) != 0 )
         {  i_ret = -15 ; goto func_end ; }
         hres = pDesktop->BindToObject( pidl_dir_menu, NULL,
                                        IID_IShellFolder,
                                        ( void ** )&p_dir_menu
                                      ) ;
         if( FAILED( hres ) ) { i_ret = -16 ; goto func_end ; }
         /*----------------------------------------------------------------*/
      }
      else /*--( All the files have to be in the same directory )----------*/
      if( sr::absolute_path( as_file[ i_num ], s_dir_def ) != s_dir_menu )
      {  /*----------------------------------------------------------------*/
         sr::info_message( _( "The shell contextual menu can only be called "
                              "if the highlighted files/directories belong "
                              "to the same directory"
                            )
                         ) ;
         /*----------------------------------------------------------------*/
         i_ret = -17 ; goto func_end ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! Add the ID of the "filename" without the path.                      !
      ! The array element can contain a path.                               !
      +--------------------------------------------------------------------*/
      if( st_conv_name_pidl( wxFileName( as_file[ i_num ] ).GetFullName(),
                             p_dir_menu, p_tb_pidl[ i_num ]
                           ) != 0
        )
      {  i_ret = -18 ; goto func_end ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( GO ! )------------------------------------------------------------*/
   st_exec_shell_contextual_menu( p_dir_menu, p_tb_pidl, as_file.GetCount());
   /*----------------------------------------------------------------------*/
func_end :
   /*----------------------------------------------------------------------*/
   if( pMalloc != NULL )
   {
      /*-------------------------------------------------------------------*/
      if( p_tb_pidl != NULL )
      {  /*----------------------------------------------------------------*/
         for( i_num = 0 ; i_num < ( int )as_file.GetCount() ; ++i_num )
         {  pMalloc->Free( p_tb_pidl[ i_num ] ) ; }
         /*----------------------------------------------------------------*/
         delete [] p_tb_pidl ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( pidl_dir_menu != NULL ) { pMalloc->Free( pidl_dir_menu ) ; }
      if( p_dir_menu != NULL ) { p_dir_menu->Release() ; }
      if( pDesktop != NULL ) { pDesktop->Release() ; }
      pMalloc->Release() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::shell_contextual_menu( const wxString &s_full_path )
{
   /*----------------------------------------------------------------------*/
   LPMALLOC      pMalloc         = NULL ;
   LPSHELLFOLDER pDesktop        = NULL ;
   LPSHELLFOLDER p_dir_path      = NULL ;
   LPITEMIDLIST  pidl_path       = NULL ;
   LPITEMIDLIST  pidl_name_start = NULL ;
   LPITEMIDLIST  pidl_name_end   = NULL ;
   int           i_ret           = 0    ;
   HRESULT       hres                   ;
   USHORT        us_cb_save             ;

   /*--( Useful ? )--------------------------------------------------------*/
   if( s_full_path.empty() ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   hres = SHGetMalloc( &pMalloc ) ;
   if( FAILED( hres ) ) { i_ret = -2 ; goto func_end ; }

   /*----------------------------------------------------------------------*/
   hres = SHGetDesktopFolder( &pDesktop ) ;
   if( FAILED( hres ) ) { i_ret = -3 ; goto func_end ; }

   /*--( Conversion of the name to an ID )---------------------------------*/
   if( st_conv_name_pidl( s_full_path.wx_str(), pDesktop, pidl_path ) != 0 )
   {  i_ret = -4 ; goto func_end ; }
   hres = pDesktop->BindToObject( pidl_path, NULL, IID_IShellFolder,
                                  ( void ** )&p_dir_path
                                ) ;
   if( FAILED( hres ) ) { i_ret = -5 ; goto func_end ; }

   /*--( Searching the last ID )-------------------------------------------*/
   pidl_name_start = pidl_path ;
   while( (   pidl_name_end
            = ( LPITEMIDLIST )(   ( BYTE * )pidl_name_start
                                + pidl_name_start->mkid.cb
                              )
          )->mkid.cb != 0
        )
   {  pidl_name_start = pidl_name_end ; }

   /*--( Separate the path name from the name with a '\0' )----------------*/
   us_cb_save = pidl_name_start->mkid.cb ;
   pidl_name_start->mkid.cb = 0          ;
   /*--( Get the directory )-----------------------------------------------*/
   hres = pDesktop->BindToObject( pidl_path, NULL, IID_IShellFolder,
                                  ( void ** )&p_dir_path
                                ) ;
   if( FAILED( hres ) ) { i_ret = -6 ; goto func_end ; }
   /*--( Restore the modified data )---------------------------------------*/
   pidl_name_start->mkid.cb = us_cb_save ;

   /*--( GO ! )------------------------------------------------------------*/
   st_exec_shell_contextual_menu( p_dir_path, &pidl_name_start, 1 ) ;
   /*----------------------------------------------------------------------*/
func_end :
   /*----------------------------------------------------------------------*/
   if( pMalloc != NULL )
   {
      /*-------------------------------------------------------------------*/
      if( pidl_path != NULL ) { pMalloc->Free( pidl_path ) ; }
      if( p_dir_path != NULL ) { p_dir_path->Release() ; }
      if( pDesktop != NULL ) { pDesktop->Release() ; }
      pMalloc->Release() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
#endif // __WXMSW__
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                        End of file sr_ctxmnu.cpp                          !
+==========================================================================*/
