/*====================================+=====================================+
! File CLoadDir.cpp                   ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/stopwatch.h>
#include "CApp.h"
#include "CLoadDir.h"
#include "CFileInit.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CLoadDirParam::CLoadDirParam()
{
   /*----------------------------------------------------------------------*/
   init() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CLoadDirParam::~CLoadDirParam()
{  ; }

/*-------------------------------------------------------------------------*/
void CLoadDirParam::init()
{
   /*----------------------------------------------------------------------*/
   raz_repos_name()            ;
   raz_sel_filter()            ;
   set_add_dir_to_hist( true ) ;
   set_keep_oper_err( false )  ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CLoadDir::CLoadDir()
         : m_boo_same_dir( false ),
           m_boo_list_dir( true ),
           m_boo_filter_dir( true ),
           m_boo_list_hidden( true ),
           m_i_nb_disp( 0 ),
           m_ll_time_last_info( 0 ),
           m_boo_thread_stopping( false )
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
   m_p_thread = NULL ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CLoadDir::~CLoadDir()
{  ; }

/*-------------------------------------------------------------------------*/
void CLoadDir::display_run_info()
{
   /*----------------------------------------------------------------------*/
   if( !m_boo_display_run_info ) { return ; }

   /*--( Limit the display of information )--------------------------------*/
   if( wxGetLocalTimeMillis() - m_ll_time_last_info < 500 )
   {  return ; }

   /*----------------------------------------------------------------------*/
   wxString s_message ;
   /*----------------------------------------------------------------------*/
   if( m_dir.size() == 0 )
   {  s_message.Printf( _( "Loading %s ..." ), m_s_path ) ; }
   else
   {  /*-------------------------------------------------------------------*/
      s_message.Printf( _( "Loading (%d) %s ..." ),
                        ( int )m_dir.size(), m_dir.back().get_path()
                      ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! To write the info on the status an event is used. Updating GUI from a  !
   ! thread (even with "wxMutexGuiEnter/Leave") is not a good idea ...      !
   +-----------------------------------------------------------------------*/
   wxCommandEvent event_disp( EVT_SR_DISP_INFO, wxID_ANY ) ;
   /*--( To avoid thread safe issue, a deep copy of the string is made )---*/
   event_disp.SetString( s_message.wc_str() ) ;
   if( m_p_thread == NULL )
   {  wxGetApp().m_frame->ProcessWindowEvent( event_disp ) ; }
   else
   {  wxQueueEvent( wxGetApp().m_frame, new wxCommandEvent( event_disp ) ) ;}

   /*-----------------------------------------------------------------------+
   ! As "m_ll_time_last_info" has been reset to 0 during start, the file    !
   ! list clear is done during the second "time" this code is executed.     !
   +-----------------------------------------------------------------------*/
   if( m_i_nb_disp == 1 )
   {
      /*--------------------------------------------------------------------+
      ! Before clearing the list, its "display state" is kept: focus ...    !
      ! to put it back after loading. Of course this is done only if the    !
      ! directory being loaded is the same as the previously loaded one.    !
      ! The "extraction" is not done during the start because the user      !
      ! can interact with the list before it is cleared.                    !
      ! It is only done only once to avoid resetting during repeated load   !
      ! starts (refresh constantly pressed ...)                             !
      +--------------------------------------------------------------------*/
      if( get_same_dir() && m_fl_display_state.empty() )
      {  wxGetApp().get_fl()->extract_display_state( m_fl_display_state ) ; }

      /*-------------------------------------------------------------------*/
      wxCommandEvent event_clear( EVT_SR_CLEAR_FILE_LIST, wxID_ANY ) ;
      /*-------------------------------------------------------------------*/
      if( m_p_thread == NULL )
      {  wxGetApp().m_frame->ProcessWindowEvent( event_clear ) ; }
      else
      {  wxQueueEvent( wxGetApp().m_frame,
                       new wxCommandEvent( event_clear )
                     ) ;
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   ++m_i_nb_disp ;
   m_ll_time_last_info = wxGetLocalTimeMillis() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CLoadDir::add_name( const wxString &s_name, bool boo_is_dir )
{
   /*-----------------------------------------------------------------------+
   ! The element has to be added ?                                          !
   ! The filter is applied only on the file name part, not the path         !
   +-----------------------------------------------------------------------*/
   if(    (    boo_is_dir
            && (    !m_boo_list_dir
                 || (    m_boo_filter_dir
                      && !m_s_filter.empty()
                      && !sr::filename_match_filter( s_name, m_s_filter )
                    )
               )
          )
       || (    !boo_is_dir
            && !m_s_filter.empty()
            && !sr::filename_match_filter( s_name, m_s_filter )
          )
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   m_dir.push_back( CFile() ) ;
   /*-----------------------------------------------------------------------+
   ! Initialisation done after the creation to (maybe) avoid useless data   !
   ! transfer.                                                              !
   +-----------------------------------------------------------------------*/
   CFileInit( m_dir.back() ).init_full( m_s_path, s_name, boo_is_dir ) ;
   /*----------------------------------------------------------------------*/
   display_run_info() ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxDirTraverseResult CLoadDir::OnDir( const wxString &s_dirname )
{
   /*--( Stop request ? )--------------------------------------------------*/
   if( m_p_thread != NULL && m_p_thread->TestDestroy() )
   {  m_boo_display_run_info = false ; return( wxDIR_STOP ) ; }
   /*----------------------------------------------------------------------*/
   add_name( s_dirname, true ) ;
   /*----------------------------------------------------------------------*/
   return( m_dir.get_recurse() ? wxDIR_CONTINUE : wxDIR_IGNORE ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxDirTraverseResult CLoadDir::OnFile( const wxString &s_filename )
{
   /*--( Stop request ? )--------------------------------------------------*/
   if( m_p_thread != NULL && m_p_thread->TestDestroy() )
   {  m_boo_display_run_info = false ; return( wxDIR_STOP ) ; }
   /*----------------------------------------------------------------------*/
   add_name( s_filename, false ) ;
   /*----------------------------------------------------------------------*/
   return( wxDIR_CONTINUE ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CLoadDir::exec()
{
   /*--( Don't want any error message during directory load ... )----------*/
   wxLogNull no_log ;

   /*--( Directory Ok ? )--------------------------------------------------*/
   wxDir dir( m_s_path ) ;
   /*----------------------------------------------------------------------*/
   if( !dir.IsOpened() ) { return ; }
   /*----------------------------------------------------------------------*/
   m_dir.zero() ;
   m_i_nb_disp = 0 ;
   m_ll_time_last_info = 0 ;
   m_boo_display_run_info = true ;
   /*--( First display: 0 file )-------------------------------------------*/
   display_run_info() ;
   /*----------------------------------------------------------------------*/
   dir.Traverse( *this, wxEmptyString,
                   wxDIR_FILES | wxDIR_DIRS | wxDIR_NO_FOLLOW
                 | ( m_boo_list_hidden ? wxDIR_HIDDEN : 0 )
               ) ;
   /*--( Nothing needs to be displayed any more )--------------------------*/
   m_boo_display_run_info = false ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CLoadDir::send_end_event( bool boo_synchronized )
{
   /*--( Loading terminated (stopped or not). It is signalled )------------*/
   wxCommandEvent event( EVT_SR_LOADDIR_END, wxID_ANY ) ;
   /*--( Wait for the end of event processing ? )--------------------------*/
   if( boo_synchronized )
   {  wxGetApp().m_frame->ProcessWindowEvent( event ) ; }
   else
   {  wxQueueEvent( wxGetApp().m_frame, new wxCommandEvent( event ) ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CLoadDir::non_threaded_exec()
{
   /*----------------------------------------------------------------------*/
   exec() ;
   send_end_event( true ) ;
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
void CLoadDir::thread_start()
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
   /*----------------------------------------------------------------------*/
   if( m_p_thread != NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   m_p_thread = new CThreadLoadDir( this ) ;
   /*----------------------------------------------------------------------*/
   if( m_p_thread->Create() != wxTHREAD_NO_ERROR )
   {  wxFAIL ; return ; }
   /*----------------------------------------------------------------------*/
   if( m_p_thread->Run() != wxTHREAD_NO_ERROR )
   {  wxFAIL ; return ; }
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
void CLoadDir::thread_stop()
{
   /*----------------------------------------------------------------------*/
   m_boo_thread_stopping = true ;
   /*----------------------------------------------------------------------*/
   {  wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
      if(    m_p_thread != NULL
          && m_p_thread->Delete() != wxTHREAD_NO_ERROR
        )
      {  wxFAIL ; return ; }
   }
   /*--( Wait until thread real death ! )----------------------------------*/
   while( true )
   {  /*-------------------------------------------------------------------*/
      {  wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
         if( m_p_thread == NULL ) { break ; }
      }
      /*-------------------------------------------------------------------*/
      wxThread::This()->Sleep( 1 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_boo_thread_stopping = false ;
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
bool CLoadDir::thread_running()
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
   /*----------------------------------------------------------------------*/
   return( m_p_thread != NULL && !m_boo_thread_stopping ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CThreadLoadDir::CThreadLoadDir( CLoadDir *p_ld )
               : wxThread( wxTHREAD_DETACHED ),
                 m_p_ld( p_ld ),
                 m_seevt( SEEVT_DO_SEND_END_EVENT )
{  ; }

/*-------------------------------------------------------------------------*/
CThreadLoadDir::~CThreadLoadDir()
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_p_ld->m_csec_p_thread ) ;
   m_p_ld->m_p_thread = NULL ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CThreadLoadDir::gui_enable_stop_loading( bool boo_on )
{
   /*----------------------------------------------------------------------*/
   wxCommandEvent *p_event ;
   /*----------------------------------------------------------------------*/
   p_event = new wxCommandEvent( EVT_SR_SET_BT_STOP_LOADING, wxID_ANY ) ;
   p_event->SetInt( boo_on ? 1 : 0 ) ;
   wxQueueEvent( wxGetApp().m_frame, p_event ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxThread::ExitCode CThreadLoadDir::Entry()
{
   /*----------------------------------------------------------------------*/
   gui_enable_stop_loading( true ) ;
   m_p_ld->exec() ;
   gui_enable_stop_loading( false ) ;

   /*-----------------------------------------------------------------------+
   ! By sending the event from this location it will be executed during:    !
   ! - "normal" ending                                                      !
   ! - "stopped" ending                                                     !
   +-----------------------------------------------------------------------*/
   if( get_send_end_event() == SEEVT_DO_SEND_END_EVENT )
   {  m_p_ld->send_end_event() ; }
   /*----------------------------------------------------------------------*/
   return( ( wxThread::ExitCode )0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CLoadDir.cpp                          !
+==========================================================================*/
