/*====================================+=====================================+
! File CImageVisu.cpp                 ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/mstream.h>
#include <wx/dcclient.h>
#include <wx/settings.h>
#include "CApp.h"
#include "CImageVisu.h"
#include "CFileAccess.h"
#include "img_rotation.xpm"
#include "thumbnail_off.xpm"
#include "thumbnail_on.xpm"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
static const int st_co_i_margin = 2 * 4 ;
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE( CImageVisu, wxWindow )
   EVT_ERASE_BACKGROUND( CImageVisu::OnEraseBackground )
   EVT_PAINT( CImageVisu::OnPaint )
   EVT_SIZE( CImageVisu::OnResize )
END_EVENT_TABLE()
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CImageVisu::CImageVisu( wxWindow *parent, wxWindowID id )
          : wxWindow( parent, id, wxDefaultPosition, wxDefaultSize,
                      wxSUNKEN_BORDER
                    )
{  /*----------------------------------------------------------------------*/
   m_boo_imagevisu_possible = false ;
   reset() ;
   SetBackgroundColour( wxSystemSettings::GetColour( wxSYS_COLOUR_WINDOW ) );
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CImageVisu::~CImageVisu()
{  ; }

/*--( To limit flickering the erase is done in the display function )------*/
void CImageVisu::OnEraseBackground( wxEraseEvent & WXUNUSED( event ) )
{
   /*----------------------------------------------------------------------*/
   ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CImageVisu::OnPaint( wxPaintEvent & WXUNUSED( event ) )
{
   /*----------------------------------------------------------------------*/
   wxPaintDC dc( this ) ;
   wxColour  colour( dc.GetBackground().GetColour() ) ;

   /*--( Dark a bit the background to make standard colour appear )--------*/
   colour.Set( colour.Red() - 25, colour.Green() - 25, colour.Blue() - 25 ) ;
   dc.SetBackground( wxBrush( colour ) ) ;
   /*----------------------------------------------------------------------*/
   if( display_image( dc ) != 0 )
   {  /*--( In case of problem, all the space is cleared )-----------------*/
      dc.DestroyClippingRegion() ;
      dc.SetClippingRegion( wxRect( GetClientSize() ) ) ;
      dc.Clear() ;
      /*-------------------------------------------------------------------*/
   }
   /*-----------------------------------------------------------------------+
   ! Even in case of error some info can be displayed. For example the      !
   ! thumbnail icon if the "full" image can't be displayed: CRW ... etc ... !
   +-----------------------------------------------------------------------*/
   display_info( dc ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CImageVisu::OnResize( wxSizeEvent &event )
{
   /*----------------------------------------------------------------------*/
   if( wxGetApp().m_frame == NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   wxSize size_display( event.GetSize() ) ;
   /*----------------------------------------------------------------------*/
   bool boo_imagevisu_possible
        =    size_display.GetWidth()  > 2 * st_co_i_margin
          && size_display.GetHeight() > 2 * st_co_i_margin ;
   /*----------------------------------------------------------------------*/
   if( boo_imagevisu_possible != m_boo_imagevisu_possible )
   {
      /*-------------------------------------------------------------------*/
      m_boo_imagevisu_possible = boo_imagevisu_possible ;
      /*-------------------------------------------------------------------*/
      if( m_boo_imagevisu_possible )
      {  wxGetApp().m_frame->visu_file() ; }
      else
      {  reset() ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   Refresh( false ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CImageVisu::reset()
{
   /*----------------------------------------------------------------------*/
   m_s_file_abs_full_path.clear() ;
   m_dt_file_mod = wxInvalidDateTime ;
   m_ull_file_size = 0 ;
   /*----------------------------------------------------------------------*/
   m_iv_thumbnail = IV_THUMBNAIL_NONE ;
   m_iv_rotation  = IV_ROTATION_0     ;
   /*----------------------------------------------------------------------*/
   m_image.Destroy() ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The rotation value contained in the instance is supposed to indicate the  !
! rotation to apply to have an "upright" image.                             !
+--------------------------------------------------------------------------*/
int CImageVisu::rotate_image()
{
   /*----------------------------------------------------------------------*/
   if( !m_image.IsOk() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   switch( m_iv_rotation )
   {
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_0_HM :
         m_image = m_image.Mirror() ;
         break ;
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_90 :
         m_image = m_image.Rotate90() ;
         break ;
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_90_HM :
         m_image = m_image.Rotate90( true ).Mirror() ;
         break ;
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_180 :
         m_image = m_image.Rotate180() ;
         break ;
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_180_HM :
         m_image = m_image.Rotate180().Mirror() ;
         break ;
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_270 :
         m_image = m_image.Rotate90( false ) ;
         break ;
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_270_HM :
         m_image = m_image.Rotate90( false ).Mirror() ;
         break ;
      /*-------------------------------------------------------------------*/
      default : break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CImageVisu::visu( const CFile *p_file )
{
   /*--( Temporarily no image load wanted ? )------------------------------*/
   if( !wxGetApp().m_frame->get_disp_image() ) { return( -1 ) ; }
   /*--( Enough space ? )--------------------------------------------------*/
   if( !m_boo_imagevisu_possible ) { return( -2 ) ; }

   /*--( Save the previous info to detect a modification )-----------------*/
   e_iv_thumbnail iv_thumbnail_prev = m_iv_thumbnail ;
   e_iv_rotation  iv_rotation_prev  = m_iv_rotation  ;
   bool           boo_img_hq_manip  = m_boo_img_hq_manip ;
   bool           boo_img_stretch   = m_boo_img_stretch ;
   wxFileOffset   fo_img_offset     = -1 ;
   wxULongLong    ull_img_size ;
   /*----------------------------------------------------------------------*/
   m_iv_thumbnail = IV_THUMBNAIL_NONE ;
   m_iv_rotation  = IV_ROTATION_0     ;
   m_boo_img_hq_manip = wxGetApp().M_boo_img_hq_manip.get() ;
   m_boo_img_stretch  = wxGetApp().M_boo_img_stretch.get() ;

   /*--( What will be displayed ? )----------------------------------------*/
   if( p_file != NULL )
   {
      /*-------------------------------------------------------------------*/
      if( p_file->has_thumbnail() )
      {
         /*----------------------------------------------------------------*/
         if( wxGetApp().M_boo_thumbnail.get() )
         {  /*-------------------------------------------------------------*/
            fo_img_offset = p_file->get_thumbnail_offset() ;
            ull_img_size  = p_file->get_thumbnail_size()   ;
            /*-------------------------------------------------------------*/
            m_iv_thumbnail = IV_THUMBNAIL_ON ;
            /*-------------------------------------------------------------*/
         }
         else
         {  /*-------------------------------------------------------------*/
            fo_img_offset = p_file->get_image_offset() ;
            ull_img_size  = p_file->get_image_size()   ;
            /*-------------------------------------------------------------*/
            m_iv_thumbnail = IV_THUMBNAIL_OFF ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else
      {
         /*--( No image => default values and erase the space )------------*/
         fo_img_offset = p_file->get_image_offset() ;
         ull_img_size  = p_file->get_image_size()   ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! The "rotation" symbol is not displayed if no image or no rotation   !
      ! is requested.                                                       !
      ! Example: Canon 10D\78625d18.dng.tif                                 !
      +--------------------------------------------------------------------*/
      if(     wxGetApp().M_boo_img_rotation.get()
          && ( p_file->has_image() || p_file->has_thumbnail() )
        )
      {  m_iv_rotation = p_file->get_img_rotation() ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( No file or file without image ? )---------------------------------*/
   if( fo_img_offset < 0 )
   {
      /*--( Refresh if an image was already displayed )--------------------*/
      if( m_image.IsOk() )
      {  reset() ;
         Refresh( false ) ;
      }
      /*-------------------------------------------------------------------*/
      return( -3 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   wxString    s_file_abs_full_path ;
   wxDateTime  dt_file_mod ;
   wxULongLong ull_file_size ;

   /*--( Keep track of the file to limit reloads )-------------------------*/
   if( p_file != NULL )
   {  s_file_abs_full_path = p_file->get_abs_full_path() ;
      dt_file_mod = p_file->get_dt_val( COL_BASE_MOD_DATE ) ;
      ull_file_size = p_file->get_size() ;
   }
   /*----------------------------------------------------------------------*/
   if(    m_s_file_abs_full_path == s_file_abs_full_path
       && m_dt_file_mod.IsValid()
       && m_dt_file_mod      == dt_file_mod
       && m_ull_file_size    == ull_file_size
       && m_iv_thumbnail     == iv_thumbnail_prev
       && m_iv_rotation      == iv_rotation_prev
       && m_boo_img_hq_manip == boo_img_hq_manip
       && m_boo_img_stretch  == boo_img_stretch
     )
   {  return( 0 ) ; }

   /*----------------------------------------------------------------------*/
   g_disp_info( wxString::Format( _( "Loading image from \"%s\" ..." ),
                                  p_file->get_full_path()
                                )
              ) ;
   int i_ret = 0 ;

   /*-----------------------------------------------------------------------+
   ! Full file or only a part of it ?                                       !
   +-----------------------------------------------------------------------*/
   {
      /*--( No error message )---------------------------------------------*/
      wxLogNull no_log ;
      bool      boo_load_ok ;

      /*-------------------------------------------------------------------*/
      if( fo_img_offset == 0 )
      {  boo_load_ok = m_image.LoadFile( p_file->get_full_path() ) ; }
      else
      {
         /*-----------------------------------------------------------------+
         ! "wxMemoryInputStream" is used instead of "wxFileInputStream".    !
         ! "wxFileInputStream" doesn't take a size limit and in some cases  !
         ! (image data rotten) "LoadImage" tries to get much more than it   !
         ! should and the loading fails.                                    !
         +-----------------------------------------------------------------*/
         CFileAccess    fa ;
         size_t         co_sz_img_size = ull_img_size.ToULong() ;
         wxMemoryBuffer mb_image ;
         /*----------------------------------------------------------------*/
         if(    fa.open_to_read( p_file->get_full_path() ) != 0
             || fa.set_offset( fo_img_offset ) != 0
             || fa.read_buffer( co_sz_img_size, mb_image ) != 0
           )
         {  i_ret = -1 ; goto func_end ; }
         /*----------------------------------------------------------------*/
         wxMemoryInputStream mem_input( mb_image.GetData(), co_sz_img_size );
         boo_load_ok = m_image.LoadFile( mem_input ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( !boo_load_ok )
      {  reset() ;
         i_ret = -2 ; goto func_end ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   rotate_image() ;
   /*----------------------------------------------------------------------*/
func_end :
   /*--( Refresh / Delete )------------------------------------------------*/
   Refresh( false ) ;
   /*----------------------------------------------------------------------*/
   if( i_ret == 0 )
   {  m_s_file_abs_full_path = s_file_abs_full_path ;
      m_dt_file_mod = dt_file_mod ;
      m_ull_file_size = ull_file_size ;
   }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CImageVisu::display_image( wxDC &dc )
{
   /*----------------------------------------------------------------------*/
   if( !m_boo_imagevisu_possible ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( !m_image.IsOk() ) { return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   wxSize               size_image        ;
   wxRect               rec_display       ;
   wxRect               rec_image_display ;
   wxImageResizeQuality ir_quality        ;

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! Computation of the dimensions                                          !
   !                                                                        !
   +-----------------------------------------------------------------------*/

   /*--( The physical size of the image )----------------------------------*/
   size_image.Set( m_image.GetWidth(), m_image.GetHeight() ) ;

   /*--( What is the destination available space ? )-----------------------*/
   rec_display = GetClientSize() ;

   /*--( Start with the "source" sizes )-----------------------------------*/
   rec_image_display.width  = size_image.GetWidth()  ;
   rec_image_display.height = size_image.GetHeight() ;

   /*-----------------------------------------------------------------------+
   ! Computation of the image size based on the available space:            !
   ! Of course this operation is done based on the data after rotation.     !
   ! If the image is smaller: it can be stretched if asked via the option.  !
   ! If the image is bigger: it is reduced keeping its proportions.         !
   +-----------------------------------------------------------------------*/
   if(    wxGetApp().M_boo_img_stretch.get()
       || rec_image_display.width > rec_display.GetWidth() - st_co_i_margin
     )
   {  /*-------------------------------------------------------------------*/
      rec_image_display.width  = rec_display.GetWidth() - st_co_i_margin ;
      rec_image_display.height =   rec_image_display.width
                                 * size_image.GetHeight()
                                 / size_image.GetWidth() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( rec_image_display.height > rec_display.GetHeight() - st_co_i_margin )
   {  /*-------------------------------------------------------------------*/
      rec_image_display.height = rec_display.GetHeight() - st_co_i_margin ;
      rec_image_display.width  =   rec_image_display.height
                                 * size_image.GetWidth()
                                 / size_image.GetHeight() ;
      /*-------------------------------------------------------------------*/
   }
   /*--( All is ok ? )-----------------------------------------------------*/
   if( rec_image_display.width  <= 0 || rec_image_display.height <= 0 )
   {  return( -3 ) ; }

   /*--( The position where to display the image: upper left corner )------*/
     rec_image_display.x
   = ( rec_display.GetWidth() - rec_image_display.width ) / 2 ;
     rec_image_display.y
   = ( rec_display.GetHeight() - rec_image_display.height ) / 2 ;

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! If the image has an alpha channel or some transparency, a simple draw  !
   ! may let some information of the previous image in the "clear" parts of !
   ! the new one. This is why "all" the dc is cleared.                      !
   !                                                                        !
   ! Of course this will generate flickering for "alphaed/masked" images    !
   ! during a resize.                                                       !
   !                                                                        !
   +-----------------------------------------------------------------------*/
   if( m_image.HasAlpha() || m_image.HasMask() )
   {
      /*-------------------------------------------------------------------*/
      dc.Clear() ;
      /*--( In this case, high quality would give a blurry result ... )----*/
      ir_quality = wxIMAGE_QUALITY_NORMAL ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*-------------------------------------------------------------------*/
      wxRegion reg_to_clean ;
      /*-------------------------------------------------------------------*/
      reg_to_clean.Union( rec_display ) ;
      reg_to_clean.Subtract( rec_image_display ) ;
      dc.DestroyClippingRegion() ;
      dc.SetDeviceClippingRegion( reg_to_clean ) ;
      dc.Clear() ;
      dc.DestroyClippingRegion() ;
      dc.SetDeviceClippingRegion( rec_display ) ;
      /*-------------------------------------------------------------------*/
      ir_quality = wxGetApp().M_boo_img_hq_manip.get()
                   ? wxIMAGE_QUALITY_HIGH
                   : wxIMAGE_QUALITY_NORMAL ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Display the image                                                      !
   +-----------------------------------------------------------------------*/
   {
      /*-------------------------------------------------------------------*/
      wxLogNull no_log ;
      /*-------------------------------------------------------------------*/
      dc.DrawBitmap( wxBitmap( m_image.Scale( rec_image_display.width,
                                              rec_image_display.height,
                                              ir_quality
                                            )
                             ),
                     rec_image_display.x, rec_image_display.y,
                     m_image.HasMask()
                   ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CImageVisu::display_info( wxDC &dc ) const
{
   /*----------------------------------------------------------------------*/
   if( !m_boo_imagevisu_possible ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   int i_col_aff = st_co_i_margin ;

   /*----------------------------------------------------------------------*/
   if( m_iv_thumbnail != IV_THUMBNAIL_NONE )
   {
      /*-------------------------------------------------------------------*/
      wxIcon thm_ico ;
      /*-------------------------------------------------------------------*/
      switch( m_iv_thumbnail )
      {
         /*----------------------------------------------------------------*/
         case IV_THUMBNAIL_ON : thm_ico = wxIcon( thumbnail_on_xpm ) ; break;
         case IV_THUMBNAIL_OFF: thm_ico = wxIcon( thumbnail_off_xpm ); break;
         default: return( -2 ) ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      dc.DrawIcon( thm_ico, i_col_aff, st_co_i_margin ) ;
      /*-------------------------------------------------------------------*/
      i_col_aff += thm_ico.GetWidth() + st_co_i_margin ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( m_iv_rotation != IV_ROTATION_0 )
   {
      /*-------------------------------------------------------------------*/
      dc.DrawIcon( wxIcon( img_rotation_xpm ), i_col_aff, st_co_i_margin ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CImageVisu.cpp                        !
+==========================================================================*/
