/*====================================+=====================================+
! File CDirTree.cpp                   ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/menu.h>
#include "common/sr_lib.h"
#include "CDirTree.h"
#include "CCombo.h"
#include "CApp.h"
#ifdef __WXMSW__
#include <shlobj.h>
#include <shlwapi.h>
#include <wx/msw/registry.h>
#include "common/msw/sr_ctxmnu.h"
#endif // __WXMSW__
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CDirTree::CDirTree( wxWindow *parent, wxWindowID id )
{
   /*-----------------------------------------------------------------------+
   ! A few styles have to be added to the tree via the virtual function     !
   ! "CreateTreeCtrl". A the virtual table will only be "active" once the   !
   ! objects are created, "Create" is used.                                 !
   +-----------------------------------------------------------------------*/
   wxGenericDirCtrl::Create( parent, id, wxEmptyString,
                             wxDefaultPosition, wxDefaultSize,
                             wxDIRCTRL_DIR_ONLY
                           ) ;
   /*----------------------------------------------------------------------*/
   init() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CDirTree::~CDirTree()
{
   /*----------------------------------------------------------------------*/
   setup_events( false ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Under MSW, to have a more "Explorer" look, a few styles have to be added  !
+--------------------------------------------------------------------------*/
wxTreeCtrl *CDirTree::CreateTreeCtrl( wxWindow *parent, wxWindowID treeid,
                                      const wxPoint &pos, const wxSize &size,
                                      long treeStyle
                                    )
{  /*----------------------------------------------------------------------*/
   return( new wxTreeCtrl( parent, treeid, pos, size,
                           treeStyle | wxTR_NO_LINES | wxTR_TWIST_BUTTONS
                         )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
#ifdef __WXMSW__
/*--------------------------------------------------------------------------+
! Under MSW the default "system" icons are stored in SHELL32.DLL.           !
! This default can be changed via the registry:                             !
! <expected id> => file,id in file                                          !
+--------------------------------------------------------------------------*/
wxIcon static st_get_shell_icon( int i_shell_icon )
{
   /*----------------------------------------------------------------------*/
   wxLogNull no_log ;
   HICON     hico ;
   wxIcon    ico  ;
   wxRegKey  key( wxRegKey::HKLM,
                  "SOFTWARE\\Microsoft\\Windows\\"
                  "CurrentVersion\\Explorer\\Shell Icons"
                ) ;
   /*--( Try to find icon customization defined through the registry )-----*/
   if( key.Exists() )
   {
      /*-------------------------------------------------------------------*/
      wxString s_val      ;
      wxString s_val_ico  ;
      wxString s_filename ;
      wxString s_ico_num  ;
      /*-------------------------------------------------------------------*/
      s_val << i_shell_icon ;
      /*-------------------------------------------------------------------*/
      if(    key.QueryValue( s_val, s_val_ico )
          && !(   s_filename
                = s_val_ico.BeforeLast( wxUniChar(','), &s_ico_num )
              ).empty()
        )
      {  /*----------------------------------------------------------------*/
         long l_ico_num ;
         /*----------------------------------------------------------------*/
         if(    s_ico_num.ToLong( &l_ico_num )
             && ExtractIconEx( s_filename.fn_str(), l_ico_num, NULL, &hico, 1
                             ) == 1
           )
         {  ico.SetHICON( hico ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Standard way ? )--------------------------------------------------*/
   if(    !ico.IsOk()
       && ExtractIconEx( wxString( "SHELL32.DLL" ).fn_str(),
                         i_shell_icon, NULL, &hico, 1
                       ) == 1
     )
   {  ico.SetHICON( hico ) ; }

   /*----------------------------------------------------------------------*/
   return( ico ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static void st_replace_tfit_icon( int i_tfit, int i_shell_icon )
{
   /*----------------------------------------------------------------------*/
   wxIcon ico( st_get_shell_icon( i_shell_icon ) ) ;

   /*----------------------------------------------------------------------*/
   if( ico.IsOk() )
   {  wxTheFileIconsTable->GetSmallImageList()->Replace( i_tfit, ico ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_add_tfit_icon( int i_shell_icon )
{
   /*----------------------------------------------------------------------*/
   return( wxTheFileIconsTable->GetSmallImageList()->Add(
                                            st_get_shell_icon( i_shell_icon )
                                                        )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Based on msdn code ... )---------------------------------------------*/
static int st_get_specialfolder_info( int i_folder,
                                      wxString &s_name, wxString &s_path
                                    )
{
   /*----------------------------------------------------------------------*/
   IMalloc      *pShellMalloc = NULL ;
   IShellFolder *psfParent ;
   LPITEMIDLIST pidlItem = NULL ;
   LPITEMIDLIST pidlRelative = NULL ;
   STRRET       str ;
   WCHAR        sz_info[ MAX_PATH ] ;
   int          i_ret = 0 ;

   /*----------------------------------------------------------------------*/
   s_name.clear() ;
   s_path.clear() ;
   /*----------------------------------------------------------------------*/
   HRESULT hres = SHGetMalloc( &pShellMalloc ) ;
   if( FAILED( hres ) ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   hres = SHGetSpecialFolderLocation( NULL, i_folder, &pidlItem ) ;
   if( FAILED( hres ) )
   {  i_ret = -2 ; }
   else
   {  /*-------------------------------------------------------------------*/
      hres = SHBindToParent( pidlItem, IID_IShellFolder,
                             ( void ** )&psfParent,
                             ( LPCITEMIDLIST * )&pidlRelative
                           ) ;
      if( FAILED( hres ) )
      {  i_ret = -3 ; }
      else
      {  /*--( Retrieve the display name )---------------------------------*/
         memset( &str, 0, sizeof( str ) ) ;
           hres
         = psfParent->GetDisplayNameOf( pidlRelative, SHGDN_NORMAL, &str ) ;
         if( SUCCEEDED( hres ) )
         {  StrRetToBuf( &str, pidlItem, sz_info, WXSIZEOF( sz_info ) ) ;
            s_name = sz_info ;
         }
         /*--( Retrieve the path )-----------------------------------------*/
         memset( &str, 0, sizeof( str ) ) ;
           hres
         = psfParent->GetDisplayNameOf( pidlRelative,
                                        SHGDN_NORMAL | SHGDN_FORPARSING, &str
                                      ) ;
         if( SUCCEEDED( hres ) )
         {  StrRetToBuf( &str, pidlItem, sz_info, WXSIZEOF( sz_info ) ) ;
            s_path = sz_info ;
         }
         /*----------------------------------------------------------------*/
         psfParent->Release() ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Clean up allocated memory                                              !
   ! pidlRelative, despite what is done in msdn example, is not released    !
   ! because it generates a crash (the doc says it has not to be done)      !
   +-----------------------------------------------------------------------*/
   if( pidlItem != NULL ) { pShellMalloc->Free( pidlItem ) ; }
   pShellMalloc->Release();
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CDirTree::add_special_folder( size_t sz_before,
                                  int i_folder, int i_tfit
                                )
{
   /*----------------------------------------------------------------------*/
   wxString s_name ;
   wxString s_path ;

   /*--( First get information about the "special folder" )----------------*/
   if( st_get_specialfolder_info( i_folder, s_name, s_path ) != 0 )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! Add the "section" (directory tree item)                                !
   ! AddSection adds the tree item at the bottom. It has to be moved to top !
   +-----------------------------------------------------------------------*/
   wxTreeItemId ti_bottom  ;
   wxTreeItemId ti_top     ;

   /*--( Create the "section" at the bottom )------------------------------*/
   ti_bottom = wxGenericDirCtrl::AddSection( s_path, s_name, i_tfit ) ;

   /*--( Insert the same item at top )-------------------------------------*/
     ti_top
   = m_p_tree->InsertItem( m_p_tree->GetRootItem(), sz_before,
                           m_p_tree->GetItemText( ti_bottom ),
                           m_p_tree->GetItemImage( ti_bottom,
                                                   wxTreeItemIcon_Normal
                                                 ),
                           m_p_tree->GetItemImage( ti_bottom,
                                                   wxTreeItemIcon_Selected
                                                 ),
                           m_p_tree->GetItemData( ti_bottom )
                         ) ;
   m_p_tree->SetItemHasChildren( ti_top ) ;

   /*--( And delete the first created one )--------------------------------*/
   m_p_tree->SetItemData( ti_bottom, NULL ) ;
   m_p_tree->Delete( ti_bottom ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::add_special_folders()
{
   /*--( Desktop )---------------------------------------------------------*/
   add_special_folder( 0, CSIDL_DESKTOPDIRECTORY, st_add_tfit_icon( 34 ) ) ;
   /*--( My Document )-----------------------------------------------------*/
   add_special_folder( 1, CSIDL_PERSONAL, st_add_tfit_icon( -235 ) ) ;
   /*----------------------------------------------------------------------*/
}

#endif // __WXMSW__

/*-------------------------------------------------------------------------*/
void CDirTree::setup_events( bool boo_connect )
{
   /*----------------------------------------------------------------------*/
   m_p_tree = GetTreeCtrl() ;
   /*-----------------------------------------------------------------------+
   ! Set the events associated to the directory tree.                       !
   ! A simple "SetPath" will fire an event ...                              !
   +-----------------------------------------------------------------------*/
   if( boo_connect )
   {
      /*-------------------------------------------------------------------*/
      m_p_tree->Bind( wxEVT_COMMAND_TREE_SEL_CHANGING,
                      &CDirTree::OnSelChanging, this
                    ) ;
      m_p_tree->Bind( wxEVT_COMMAND_TREE_SEL_CHANGED,
                      &CDirTree::OnSelChanged, this
                    ) ;
      /*-------------------------------------------------------------------*/
      m_p_tree->Bind( wxEVT_COMMAND_TREE_ITEM_MENU,
                      &CDirTree::OnItemMenu, this
                    ) ;
      /*-------------------------------------------------------------------*/
      m_p_tree->Bind( wxEVT_COMMAND_TREE_KEY_DOWN,
                      &CDirTree::OnKeyDown, this
                    ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   {
      /*-------------------------------------------------------------------*/
      m_p_tree->Unbind( wxEVT_COMMAND_TREE_SEL_CHANGING,
                        &CDirTree::OnSelChanging, this
                      ) ;
      m_p_tree->Unbind( wxEVT_COMMAND_TREE_SEL_CHANGED,
                        &CDirTree::OnSelChanged, this
                      ) ;
      /*-------------------------------------------------------------------*/
      m_p_tree->Unbind( wxEVT_COMMAND_TREE_ITEM_MENU,
                        &CDirTree::OnItemMenu, this
                      ) ;
      /*-------------------------------------------------------------------*/
      m_p_tree->Unbind( wxEVT_COMMAND_TREE_KEY_DOWN,
                        &CDirTree::OnKeyDown, this
                      ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_boo_treat_sel_change_events = boo_connect ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::init()
{
   /*----------------------------------------------------------------------*/
   setup_events( true ) ;

   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   /*-----------------------------------------------------------------------+
   ! Replace the default "artprovider"'s icons (blue) with MSW's ones.      !
   ! The icon indices are "known" ...                                       !
   ! Add the special folders at stop of tree.                               !
   +-----------------------------------------------------------------------*/
   if( wxGetApp().m_frame->get_disp_image() )
   {
      /*-------------------------------------------------------------------*/
      static bool st_init_done = false ;
      /*--( Operation useful only once )-----------------------------------*/
      if( !st_init_done )
      {
         /*----------------------------------------------------------------*/
         st_replace_tfit_icon( wxFileIconsTable::folder     ,  3 ) ;
         st_replace_tfit_icon( wxFileIconsTable::folder_open,  4 ) ;
         st_replace_tfit_icon( wxFileIconsTable::computer   , 15 ) ;
         st_replace_tfit_icon( wxFileIconsTable::drive      ,  8 ) ;
         st_replace_tfit_icon( wxFileIconsTable::cdrom      , 11 ) ;
         st_replace_tfit_icon( wxFileIconsTable::floppy     ,  6 ) ;
         st_replace_tfit_icon( wxFileIconsTable::removeable ,  7 ) ;
         /*----------------------------------------------------------------*/
         add_special_folders() ;
         /*----------------------------------------------------------------*/
         st_init_done = true ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CDirTree::get_item_full_path( wxTreeItemId ti )
{
   /*----------------------------------------------------------------------*/
   if( !ti.IsOk() ) { return( wxEmptyString ) ; }
   /*-----------------------------------------------------------------------+
   ! The data associated to the item is maintained by wxGenericDirCtrl      !
   +-----------------------------------------------------------------------*/
   wxDirItemData *p_did = ( wxDirItemData *)m_p_tree->GetItemData( ti ) ;
   return( p_did->m_path ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::ensurevisible_current_path()
{
   /*----------------------------------------------------------------------*/
   m_p_tree->EnsureVisible( m_p_tree->GetSelection() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::OnSelChanging( wxTreeEvent &event )
{
   /*----------------------------------------------------------------------*/
   if( !m_boo_treat_sel_change_events ) { return ; }

   /*----------------------------------------------------------------------*/
   wxString s_new_path( get_item_full_path( event.GetItem() ) ) ;
   /*----------------------------------------------------------------------*/
   if( !wxGetApp().is_path_ok_to_read( s_new_path ) )
   {  sr::access_error_message( s_new_path ) ;
      event.Veto() ;
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::OnSelChanged( wxTreeEvent & WXUNUSED( event ) )
{
   /*----------------------------------------------------------------------*/
   if( !m_boo_treat_sel_change_events ) { return ; }
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->change_directory( GetPath() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::OnKeyDown( wxTreeEvent &event )
{
   /*--( Under MSW "backspace" is directly managed by the tree widget )----*/
#ifndef __WXMSW__
   if( event.GetKeyCode() == WXK_BACK )
   {  wxGetApp().m_frame->ProcessCommand( ID_TB_DIR_UP ) ; }
   else
#endif // ! __WXMSW__
   {  event.Skip() ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::OnItemMenu( wxTreeEvent &event )
{
   /*----------------------------------------------------------------------*/
   wxMenu       menu    ;
   int          i_flags ;
   wxTreeItemId ti_item ;

   /*----------------------------------------------------------------------*/
   ti_item = m_p_tree->HitTest( event.GetPoint(), i_flags ) ;
   if( !ti_item.IsOk() ) { return ; }

   /*----------------------------------------------------------------------*/
   menu.Append( IDM_DT_REFRESH,
                _( "&Refresh tree and file list\tF5" )
              ) ;
   menu.Append( IDM_DT_INS_PATH, _( "Insert full path into expression" ) ) ;
   menu.Append( IDM_EXPLORE_DIR, _( "Open in file &explorer" ) ) ;
   /*----------------------------------------------------------------------*/
#ifdef __WXMSW__
   menu.Append( IDM_SHELL_CONTEXTUAL_MENU, _( "S&hell contextual menu" ) ) ;
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
   const wxPoint pt( ScreenToClient( wxGetMousePosition() ) ) ;
   /*----------------------------------------------------------------------*/
   switch( GetPopupMenuSelectionFromUser( menu, pt ) )
   {
      /*-------------------------------------------------------------------*/
      case IDM_DT_REFRESH :
         recreate_and_refresh() ;
         break ;
      /*-------------------------------------------------------------------*/
      case IDM_DT_INS_PATH :
      {
         /*----------------------------------------------------------------*/
         wxString s_full_path = get_item_full_path( ti_item ) ;
         /*----------------------------------------------------------------*/
         if( s_full_path.empty() ) { return ; }
         /*-----------------------------------------------------------------+
         ! Automatically append the last separator else the user will       !
         ! surely have to do it                                             !
         +-----------------------------------------------------------------*/
         if( s_full_path.Last() != wxFileName::GetPathSeparator() )
         {  s_full_path.append( wxFileName::GetPathSeparator() ) ; }
         /*----------------------------------------------------------------*/
#ifdef __WXMSW__
         sr::double_backslash( s_full_path ) ;
#endif // __WXMSW__
         /*----------------------------------------------------------------*/
         wxGetApp().m_frame->m_cb_exp->WriteText( s_full_path ) ;
         wxGetApp().get_fl()->compute_new_names() ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      case IDM_EXPLORE_DIR :
         wxLaunchDefaultApplication( get_item_full_path( ti_item ) ) ;
         break ;
      /*-------------------------------------------------------------------*/
#ifdef __WXMSW__
      case IDM_SHELL_CONTEXTUAL_MENU :
         sr::shell_contextual_menu( get_item_full_path( ti_item ) ) ;
         break ;
      /*-------------------------------------------------------------------*/
#endif // __WXMSW__
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The directory tree won't be refreshed in case of a UNC path               !
+--------------------------------------------------------------------------*/
int CDirTree::set_path( const wxString &s_path_abs )
{
   /*--( Useful to go further ? )------------------------------------------*/
   if( s_path_abs.IsSameAs( GetPath(), wxFileName::IsCaseSensitive() ) )
   {  /*-------------------------------------------------------------------*/
      ensurevisible_current_path() ;
      return( 0 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( !wxGetApp().is_path_ok_to_read( s_path_abs ) )
   {  sr::access_error_message( s_path_abs ) ;
      return( -1 ) ;
   }

   /*--( Nothing done for UNC paths expect not using the directory tree )--*/
   if( sr::is_unc_path( s_path_abs ) )
   {  /*-------------------------------------------------------------------*/
      m_boo_treat_sel_change_events = false ;
      m_p_tree->Unselect() ;
      m_boo_treat_sel_change_events = true ;
      /*-------------------------------------------------------------------*/
      return( 0 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! During a "SetPath" "wxGenericDirCtrl" gets the focus (at least under   !
   ! MSW) so it has to be set manually ...                                  !
   ! "SetPath" expects an absolute path                                     !
   +-----------------------------------------------------------------------*/
   wxWindow *p_win( wxWindow::FindFocus() ) ;
   int      i_ret = 0 ;
   /*----------------------------------------------------------------------*/
   m_boo_treat_sel_change_events = false ;
   SetPath( s_path_abs ) ;
   m_boo_treat_sel_change_events = true ;

   /*----------------------------------------------------------------------*/
   if( !s_path_abs.IsSameAs( GetPath(), wxFileName::IsCaseSensitive() ) )
   {
      /*--( The directory tree may need to be recreated (refreshed) )------*/
      recreate() ;
      /*-------------------------------------------------------------------*/
      m_boo_treat_sel_change_events = false ;
      SetPath( s_path_abs ) ;
      m_boo_treat_sel_change_events = true ;
      /*-------------------------------------------------------------------*/
      if( !s_path_abs.IsSameAs( GetPath(), wxFileName::IsCaseSensitive() ) )
      {  sr::access_error_message( s_path_abs ) ;
         i_ret = -2 ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( p_win != NULL ) { p_win->SetFocus() ; }

   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::apply_hidden()
{
   /*----------------------------------------------------------------------*/
   m_boo_treat_sel_change_events = false ;
   ShowHidden( wxGetApp().M_boo_list_hidden.get() ) ;
   m_boo_treat_sel_change_events = true ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::recreate()
{
   /*----------------------------------------------------------------------*/
   m_boo_treat_sel_change_events = false ;
   Freeze() ;
   /*----------------------------------------------------------------------*/
   ReCreateTree() ;
#ifdef __WXMSW__
   add_special_folders() ;
#endif // __WXMSW__
   /*----------------------------------------------------------------------*/
   Thaw() ;
   m_boo_treat_sel_change_events = true ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::refresh()
{
   /*----------------------------------------------------------------------*/
   bool boo_same_dir ;

   /*--( Current dir may not be accessible any more ... )------------------*/
   boo_same_dir = ( set_path( wxGetApp().M_s_dir.get() ) == 0 ) ;

   /*-----------------------------------------------------------------------+
   ! Load again of current directory because it may not be the one expected !
   +-----------------------------------------------------------------------*/
   wxGetApp().m_frame->load_directory( boo_same_dir
                                       ? wxString() : GetPath()
                                     ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDirTree::recreate_and_refresh()
{
   /*----------------------------------------------------------------------*/
   recreate() ;
   refresh() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of File CDirTree.cpp                          !
+==========================================================================*/
