/*====================================+=====================================+
! File CConfig.cpp                    ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/fileconf.h>
#include <wx/log.h>
/*-------------------------------------------------------------------------*/
#include "CConfig.h"
#include "siren.h"
#include "common/sr_lib.h"
#include "common/sr_string.h"
/*-------------------------------------------------------------------------*/




/*-------------------------------------------------------------------------*/
static wxFileConfig *st_new_file_config( const wxString &s_name,
                                         int i_param_to_add
                                       )
{
   /*-----------------------------------------------------------------------+
   ! Version <= 2.00 worked only under windows and may contain '\' as       !
   ! path separator. With version 3 the '\' are escaped.                    !
   ! The first time a v2 file is opened and if a path refers to a root      !
   ! (path is terminated with '\'), an error will be raised.                !
   ! So I disable their display.                                            !
   +-----------------------------------------------------------------------*/
   wxLogNull    no_log  ;
   wxFileConfig *p_conf ;
   /*----------------------------------------------------------------------*/
   p_conf = new wxFileConfig( wxEmptyString, wxEmptyString, s_name,
                                wxEmptyString,
                                wxCONFIG_USE_LOCAL_FILE
                              | wxCONFIG_USE_RELATIVE_PATH
                              | i_param_to_add
                            ) ;
   /*-----------------------------------------------------------------------+
   ! By default the strings written are analysed and the environment        !
   ! variables are replaced. This behaviour is removed because environment  !
   ! variables contained in expressions have to be kept untouched.          !
   +-----------------------------------------------------------------------*/
   p_conf->SetExpandEnvVars( false ) ;
   /*----------------------------------------------------------------------*/
   return( p_conf ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Why this function and not using the "RenameEntry" function ?              !
! Because internally wx uses a sorted array to store all these strings and  !
! when changing the name, the position can be lost:                         !
! For example:                                                              !
!    mot_001 => word_001                                                    !
!    mot_002 => word_002                                                    !
!    mot_003 => word_003                                                    !
! When renaming "mot_001" to "word_001" the "current" (index) entry in the  !
! array becomes "mot_002" (alphabetical order) and the current is "mot_003" !
! and so the second one (mot_002) is not renamed because moved during first !
! operation.                                                                !
+--------------------------------------------------------------------------*/
void static st_rename_entries( wxFileConfig *p_conf,
                               const wxString &s_from_format,
                               const wxString &s_to_format
                             )
{
   /*----------------------------------------------------------------------*/
   wxString s_from          ;
   wxString s_to            ;
   bool     boo_end = false ;
   int      i_num           ;
   /*----------------------------------------------------------------------*/
   for( i_num = 1 ; !boo_end ; ++i_num )
   {
       /*------------------------------------------------------------------*/
       s_from.Printf( s_from_format, i_num ) ;
       /*------------------------------------------------------------------*/
       if( p_conf->HasEntry( s_from ) )
       {  s_to.Printf( s_to_format, i_num ) ;
          p_conf->RenameEntry( s_from, s_to ) ;
       }
       else
       {  boo_end = true ; }
       /*------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
#ifndef __WXMSW__
void static st_delete_entries( wxFileConfig *p_conf,
                               const wxString &s_format
                             )
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*----------------------------------------------------------------------*/
   for( i_num = 1 ;
        p_conf->DeleteEntry( wxString::Format( s_format, i_num ) ) ;
        ++i_num
      )
   {  ; }
   /*----------------------------------------------------------------------*/
}
#endif // ! __WXMSW__

/*-------------------------------------------------------------------------*/
void static st_convert_from_v2( const wxString &s_file )
{
   /*----------------------------------------------------------------------*/
   wxFileConfig *p_conf       ;
   wxString     s_group       ;
   long         l_index_group ;

   /*-----------------------------------------------------------------------+
   ! One of the first action is to double the "\" in the file because the   !
   ! storing method of the v3 is to let wx handle it (he doubles them).     !
   +-----------------------------------------------------------------------*/
   p_conf = st_new_file_config( s_file, wxCONFIG_USE_NO_ESCAPE_CHARACTERS ) ;

   /*----------------------------------------------------------------------*/
   if( p_conf->GetFirstGroup( s_group, l_index_group ) )
   {
      /*-------------------------------------------------------------------*/
      wxString s_entry       ;
      long     l_index_entry ;
      wxString s_val         ;
      /*-------------------------------------------------------------------*/
      do
      {
         /*----------------------------------------------------------------*/
         p_conf->SetPath( s_group ) ;
         /*----------------------------------------------------------------*/
         if( p_conf->GetFirstEntry( s_entry, l_index_entry ) )
         {
            /*-------------------------------------------------------------*/
            do
            {  /*----------------------------------------------------------*/
               s_val = p_conf->Read( s_entry, s_val ) ;
               sr::double_backslash( s_val ) ;
               p_conf->Write( s_entry, s_val ) ;
               /*----------------------------------------------------------*/
            } while( p_conf->GetNextEntry( s_entry, l_index_entry ) ) ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         if( s_group == "favoris" )
         {  st_rename_entries( p_conf, "nom_%03d", "name_%03d" ) ; }
         else
         if( s_group == "combo" )
         {
#ifdef __WXMSW__
            st_rename_entries( p_conf, "rep_%02d", "dir_%02d" ) ;
#else
            /*--( It is useless to keep paths on non MSW OS )--------------*/
            st_delete_entries( p_conf, "rep_%02d" ) ;
            /*-------------------------------------------------------------*/
#endif // __WXMSW__
         }
         else
         if( s_group == "cas_mot" )
         {  st_rename_entries( p_conf, "mot_%03d", "word_%03d" ) ; }
         else
         if( s_group == "date" )
         {  p_conf->RenameEntry( "ajo", "add" ) ; }

         /*----------------------------------------------------------------*/
         p_conf->SetPath( ".." ) ;

         /*----------------------------------------------------------------*/
      } while( p_conf->GetNextGroup( s_group, l_index_group ) ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! For the same reason as for "RenameEntry" this is done outside of the   !
   ! loop.                                                                  !
   +-----------------------------------------------------------------------*/
   p_conf->RenameGroup( "favoris", "favourite" ) ;
   p_conf->RenameGroup( "cas_mot", "force_case_word" ) ;

   /*----------------------------------------------------------------------*/
   delete p_conf ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CConfig_init( const wxString &s_file, const int i_version,
                   bool &boo_conv_from_v2
                 )
{
   /*----------------------------------------------------------------------*/
   wxFileConfig *p_conf    ;
   CConfigInt   pi_version ;

   /*----------------------------------------------------------------------*/
   p_conf = st_new_file_config( s_file, 0 ) ;
   wxConfigBase::Set( p_conf ) ;
   boo_conv_from_v2 = false ;

   /*----------------------------------------------------------------------*/
   pi_version.init( "/profile/version", -1 ) ;

   /*--( If it contains the current version, all is OK )-------------------*/
   if( pi_version.get() == i_version )
   {  return ; }

   /*--( A previous version ? Try to keep info )---------------------------*/
   if( pi_version.get() != -1 )
   {
      /*--( Make a backup of the file )------------------------------------*/
      if( !wxCopyFile( s_file, s_file + ".bak" ) )
      {  return ; }

      /*--------------------------------------------------------------------+
      ! Siren prior v3 was completely french (even config group/entries).   !
      +--------------------------------------------------------------------*/
      if( pi_version.get() > 0 && pi_version.get() <= 11 )
      {
         /*--( The ini is going to be reopened so I close it first )-------*/
         wxConfigBase::Set( NULL ) ;
         delete p_conf ;
         /*----------------------------------------------------------------*/
         st_convert_from_v2( s_file ) ;
         boo_conv_from_v2 = true ;
         /*--( And then reopen it )----------------------------------------*/
         p_conf = st_new_file_config( s_file, 0 ) ;
         wxConfigBase::Set( p_conf ) ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! The config file is here but with a "wrong" version value.           !
      ! It has to be "cleaned".                                             !
      +--------------------------------------------------------------------*/
      wxArrayString as_group_to_keep ;
      wxString      s_group ;
      long          l_index ;

      /*-------------------------------------------------------------------*/
      as_group_to_keep.Add( "combo" ) ;
      as_group_to_keep.Add( "favourite" ) ;
      as_group_to_keep.Add( "tool" ) ;

      /*--( Reset file content after copy keeping some groups )------------*/
      if( p_conf->GetFirstGroup( s_group, l_index ) )
      {
         /*----------------------------------------------------------------*/
         do
         {  /*--( Delete the groups that have not to be kept )-------------*/
            if( as_group_to_keep.Index( s_group ) == wxNOT_FOUND )
            {  p_conf->DeleteGroup( s_group ) ; }
            /*-------------------------------------------------------------*/
         } while( p_conf->GetNextGroup( s_group, l_index ) ) ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      sr::info_message( _( "The configuration file doesn't correspond to "
                           "the current version of Siren.\n"
                           "It has been reinitialized (after a backup "
                           "copy).\n"
                           "Only the content of the combos, the favourites "
                           "and the tools have been kept."
                         )
                      ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Write new current version value )---------------------------------*/
   pi_version.set( i_version ) ;
   pi_version.write_val() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                            End of file CConfig.cpp                        !
+==========================================================================*/
