/*====================================+=====================================+
! File CFileList.h                    ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifndef CFILELIST_H
#define CFILELIST_H
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#include "CDir.h"
#include "CImageFileList.h"
#include "CColumn.h"
#include "common/CListCtrlBase.h"
#include "CFile.h"
#include "CDragAndDrop.h"
#include "common/CLogGuiGrouped.h"
/*-------------------------------------------------------------------------*/
class CLoadDir ;
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! This enables to keep the display state of the list: file with focus and   !
! highlighted files. This will mainly be used after a reload or a sort.     !
+--------------------------------------------------------------------------*/
class CFileListDisplayState
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      wxString      m_s_focus ; // Focused file name
      wxArrayString m_as_hil  ; // Highlighted file names
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void clear()
      {  m_s_focus.clear() ; m_as_hil.clear() ; }
      /*-------------------------------------------------------------------*/
      bool empty()
      {  return( m_s_focus.size() == 0 && m_as_hil.size() == 0 ) ; }
   /*----------------------------------------------------------------------*/
} ;

/*--( Search filter flags )------------------------------------------------*/
enum e_search_filter
{ SEARCH_FILTER_MATCH = 1 << 0, SEARCH_FILTER_DIR = 1 << 1 } ;

/*-------------------------------------------------------------------------*/
class CFileList : public CListCtrlBase
{
   /*----------------------------------------------------------------------*/
   public :
      /*--( Defined in CFileList_Base.cpp )--------------------------------*/
      CFileList( wxWindow *parent, wxWindowID id = -1 ) ;
      ~CFileList() ;
   /*----------------------------------------------------------------------*/
   private :
      /*--------------------------------------------------------------------+
      ! Subitem associated to each column created. This is useful for       !
      ! functions/events referencing "real" columns as they have been       !
      ! created in the file list                                            !
      +--------------------------------------------------------------------*/
      sr::t_vec_int m_vec_i_col_subitem ;
      /*--------------------------------------------------------------------+
      ! File operation history                                              !
      +--------------------------------------------------------------------*/
      std::vector< e_otfile > m_vec_otfile    ; // Ren, Copy ...
      int                     m_i_num_otfile  ; // Current operation
      /*-------------------------------------------------------------------*/
      CImageFileList          *m_p_image_list ;
      /*--------------------------------------------------------------------+
      ! Search by typing character after character in the list              !
      ! Repositioning is done after each char                               !
      +--------------------------------------------------------------------*/
      wxLongLong m_ll_last_search_char ; // Last a search has been done (ms)
      wxString   m_s_search_by_char    ; // Search string : concatenation
                                         // of struked chars

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CDir m_dir ;
      /*--( Columns sorted following display order: left to right )--------*/
      CColumn  m_tb_col              [ COL_NB ] ;
      int      m_tb_subitem_in_tb_col[ COL_NB ] ; // Reverse: SI => COL

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CImageFileList *get_image_list() const
      {  return( m_p_image_list ) ; }
      CFile *get_p_file_focus()
      {  return( m_dir.get_p_file( get_item_focus() ) ) ; }
      /*-------------------------------------------------------------------*/
      int get_nb_col() const { return( m_vec_i_col_subitem.size() ) ; }
      bool is_disp_col_ok( int i_col ) const
      {  return( i_col >= 0 && i_col < get_nb_col() ) ; }
      /*-------------------------------------------------------------------*/
      int get_subitem_in_col( size_t sz_col ) const
      {  return( m_vec_i_col_subitem[ sz_col ] ) ; }
      /*--( Internal column subitem based on a displayed one )-------------*/
      int get_ind_col_disp_col( size_t sz_disp_col ) const
      {  return( m_tb_subitem_in_tb_col[ m_vec_i_col_subitem[ sz_disp_col ] ]
               ) ;
      }
      /*-------------------------------------------------------------------*/
      const CColumn &get_col_disp_col( size_t sz_disp_col ) const
      {  return( m_tb_col[ get_ind_col_disp_col( sz_disp_col ) ] ) ; }
      /*-------------------------------------------------------------------*/
      const wxString &get_subitem_name( int i_si ) const
      {  return( m_tb_col[ m_tb_subitem_in_tb_col[ i_si ] ].m_s_name ) ; }
      /*-------------------------------------------------------------------*/
      int get_subitem_grp( int i_si ) const
      {  return( m_tb_col[ m_tb_subitem_in_tb_col[ i_si ] ].m_grp ) ; }
      /*-------------------------------------------------------------------*/
      int get_subitem_sort( int i_si ) const
      {  return( m_tb_col[ m_tb_subitem_in_tb_col[ i_si ] ].m_sort ) ; }
      /*-------------------------------------------------------------------*/
      int get_subitem_col_disp( int i_si ) const
      {  return( m_tb_col[ m_tb_subitem_in_tb_col[ i_si ] ].m_i_col_disp ) ;}

      /*--( The operations are numbered starting at 1 )--------------------*/
      e_otfile get_otfile_num( int i_num ) const
      {  return( i_num > 0 && i_num <= ( int )m_vec_otfile.size()
                 ? m_vec_otfile[ i_num - 1 ] : OFILE_NB
               ) ;
      }
      /*-------------------------------------------------------------------*/
      int get_nb_otfile() { return( m_vec_otfile.size() ) ; }
      int get_num_otfile() { return( m_i_num_otfile ) ; }
      /*-------------------------------------------------------------------*/
      void init_drop_target()
      {  SetDropTarget( new CDropOnFileList ) ; }
      void reset_drop_target()
      {  SetDropTarget( NULL ) ; }

      /*--------------------------------------------------------------------+
      ! CFileList_Init.cpp                                                  !
      +--------------------------------------------------------------------*/
      int init() ;

      /*--------------------------------------------------------------------+
      ! CFileList_Base.cpp                                                  !
      +--------------------------------------------------------------------*/
      CColumn *get_column_with_subitem( int i_subitem, int i_from ) ;
      enum e_icfl { ICFL_NO_IMAGE, ICFL_SORT_ASC, ICFL_SORT_DESC } ;
      void set_column_image( int i_col_disp, int i_ind_img ) ;
      void set_column_sort_image( int i_col_disp, e_icfl icfl ) ;
      void set_column_sort_image( int i_col_disp = -1 ) ;
      void remove_column_sort_image( int i_col_disp ) ;
      void change_column_sort_image( int i_col_disp ) ;
      int  get_col_num_with_subitem( int i_si ) const ;
      void save_all_col_width() ;
      void init_tb_subitem_col() ;
      void load_col_info() ;
      void transfert_col_info_to_cfg() ;
      void init_col_and_grp_used() ;
      int  delete_all_col() ;
      enum e_cac { CAC_DONT_FORCE_CREATION, CAC_DO_FORCE_CREATION } ;
      bool create_all_col( e_cac cac_force = CAC_DONT_FORCE_CREATION ) ;
      void turn_all_grp_col_off_except( int i_col ) ;
      void turn_all_grp_col_on_off( e_grp grp, bool boo_on ) ;
      void col_auto_size( int i_col ) ;
      void extract_display_state( CFileListDisplayState &state ) ;
      void apply_display_state( const CFileListDisplayState &state ) ;
      void launch_file_explorer() ;
      /*-------------------------------------------------------------------*/
      int  post_loading( CLoadDir &ld ) ;

      /*--------------------------------------------------------------------+
      ! CFileList_On.cpp                                                    !
      +--------------------------------------------------------------------*/
      void OnChar( wxKeyEvent &event ) ;
      void OnKeyDown( wxKeyEvent &event ) ;
      void OnMouseLeftDown( wxMouseEvent &event ) ;
      void OnColClick( wxListEvent &event ) ;
      void OnColRightClick( wxListEvent &event ) ;
      void OnItemRightClick( wxListEvent &event ) ;
      void OnItemActivated( wxListEvent &event ) ;
      void OnItemFocused( wxListEvent &event ) ;
      void OnColBeginDrag( wxListEvent &event ) ;
      void OnBeginDrag( wxListEvent &event ) ;
      /*-------------------------------------------------------------------*/
      virtual wxListItemAttr *OnGetItemAttr( long item ) const ;
      virtual wxString OnGetItemText( long item, long column ) const ;
      virtual int OnGetItemColumnImage( long item, long column ) const ;

      /*--------------------------------------------------------------------+
      ! CFileList_Do.cpp                                                    !
      +--------------------------------------------------------------------*/
      enum e_sfl { SFL_DONT_RESFRESH, SFL_DO_RESFRESH } ;
      void sort_file_list( e_sfl sfl ) ;
      void sort_file_list_after_selection_change() ;
      void exec_hil() ;
      void del_hil() ;
      void manual_rename( long l_item ) ;
      int  rename_one_file( int i_num_item, wxFileName &new_name ) ;
      int  oper_do( e_otfile otfile ) ;
      int  oper_undo() ;
      int  oper_redo() ;
      void info_message_clipboard( const wxString &s_msg,
                                   const wxString &s_data
                                 ) ;
      void selected_list_to_clipboard() ;
      wxString col_selected_list_to_string( int i_si ) ;
      void col_selected_list_to_clipboard( int i_si ) ;
#ifdef __WXMSW__
      int  shell_contextual_menu() ;
#endif // __WXMSW__
      int  file_content_extraction() ;

      /*--------------------------------------------------------------------+
      ! CFileList_Sel.cpp                                                   !
      +--------------------------------------------------------------------*/
      int  sel( int i_item ) ;
      int  unsel( int i_item ) ;
      void sel_all() ;
      void unsel_all() ;
      void inv_sel() ;
      void sel_unsel_hil( bool boo_sel ) ;
      int  sel_fil( const wxString &s_fil, int i_flags ) ;
      void unsel_fil( const wxString &s_fil, int i_flags ) ;
      void hil_fil( const wxString &s_fil, int i_flags ) ;
      void hil_sel() ;

      enum e_foc { FOC_FIRST_OCC, FOC_NOT_FIRST_OCC } ;
      void info_search_not_found() ;
      void search_fil_next( e_foc foc ) ;
      void search_fil_prev( e_foc foc ) ;
      void search_sel_next( e_foc foc ) ;
      void search_sel_prev( e_foc foc ) ;
      void sel_unsel_hil_following_focus() ;
      void sel_name_array( const wxArrayString &as_file ) ;
      void init_view_sel() ;

      /*--------------------------------------------------------------------+
      ! CFileList_Compute.cpp                                               !
      +--------------------------------------------------------------------*/
      static int st_preprocess_exp( const wxString &s_exp, wxString &s_res,
                                    wxArrayString *p_as_inc_stack = NULL
                                  ) ;
      static int st_compute_init( const wxString &s_exp, wxString &s_res ) ;
      void compute_new_names() ;
      void compute_data_and_new_names() ;

   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      DECLARE_EVENT_TABLE()
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CLogFileOper : public CLogGuiGrouped
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CLogFileOper()
                   : CLogGuiGrouped( _( "If a file in error has the focus, "
                                        "a message giving the cause of it\n"
                                        "will be displayed in the status bar"
                                      )
                                   )
      {  ; }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#endif // CFILELIST_H
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CFileList.h                           !
+==========================================================================*/
