/*====================================+=====================================+
! File CConfig.h                      ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifndef CCONFIG_H
#define CCONFIG_H
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#include <wx/config.h>
#include <wx/gdicmn.h>
/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
template < class T >
class CConfigBaseElement
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CConfigBaseElement() : m_boo_to_save( false )
      {  ; }
      CConfigBaseElement( const wxString &s_key ) :
            m_s_key( s_key ), m_boo_to_save( false )
      {  ; }
      /*-------------------------------------------------------------------*/
      virtual ~CConfigBaseElement() { /*save()*/ ; }
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      wxString m_s_key       ;
      bool     m_boo_to_save ; // Value has changed. It has to be saved.
      T        m_val         ;
      T        m_def_val     ; // Default value
   /*----------------------------------------------------------------------*/
   public :
      /*--------------------------------------------------------------------+
      ! Returning values by reference. Could think it might create overhead !
      ! for int and bool but it's not.                                      !
      +--------------------------------------------------------------------*/
      const T &get() const { return( m_val ) ; }
      const T &get_def() const { return( m_def_val ) ; }
      /*-------------------------------------------------------------------*/
      void set( const T &val )
      {  if( m_val != val ) { m_boo_to_save = true ; m_val = val ; } }
      bool get_to_save() const { return( m_boo_to_save ) ; }
      void set_to_save( bool boo_val ) { m_boo_to_save = boo_val ; }
      /*-------------------------------------------------------------------*/
      void save() { if( m_boo_to_save ) { write_val() ; } }
      /*-------------------------------------------------------------------*/
      wxString &get_key() const { return( m_s_key ) ; }
      void set_key( const wxString &s_key ) { m_s_key = s_key ; }
      /*-------------------------------------------------------------------*/
      void init( const wxString &s_key, const T &def_val = T() )
      {  set_key( s_key ) ; m_def_val = def_val ; read_val( m_def_val ) ; }
      /*--------------------------------------------------------------------+
      ! "Virtual" to call the right function from "save" etc.               !
      +--------------------------------------------------------------------*/
      virtual void write_val() = 0 ;
      virtual void read_val( const T &def_val ) = 0 ;
      /*-------------------------------------------------------------------*/
      void del()
      {
         /*----------------------------------------------------------------*/
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }
         /*----------------------------------------------------------------*/
         p_conf->DeleteEntry( m_s_key ) ;
         m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! - Why defining the class CConfigBase and not define the virtual           !
!   functions directly in CCongigBaseElement ?                              !
! * This seems related to a problem between gcc and the "template"          !
!                                                                           !
! - Why using "this->" in CConfigBase for variables defined protected in    !
!   CConfigBaseElement ?                                                    !
! * This seems related to a problem between gcc and the "template" class    !
!   derivation                                                              !
+--------------------------------------------------------------------------*/
template < class T >
class CConfigBase : public CConfigBaseElement< T >
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CConfigBase< T >() : CConfigBaseElement< T >()
      {  ; }
      CConfigBase< T >( const wxString &s_key )
                      : CConfigBaseElement< T >( s_key )
      {  ; }
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void write_val()
      {
         /*----------------------------------------------------------------*/
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }
         /*----------------------------------------------------------------*/
         p_conf->Write( this->m_s_key, this->m_val ) ;
         this->m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      void read_val( const T &def_val )
      {
         /*----------------------------------------------------------------*/
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }
         /*--( Get file data )---------------------------------------------*/
         p_conf->Read( this->m_s_key, &this->m_val, def_val ) ;
         this->m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
typedef CConfigBase< int    > CConfigInt    ;
typedef CConfigBase< bool   > CConfigBool   ;
typedef CConfigBase< double > CConfigDouble ;

/*-------------------------------------------------------------------------*/
class CConfigString : public CConfigBase< wxString >
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CConfigString() : CConfigBase< wxString >()
      {  ; }
      CConfigString( const wxString &s_key )
                   : CConfigBase< wxString >( s_key )
      {  ; }
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      bool empty() const { return( m_val.empty() ) ; }
      sr::wxString_cit begin() { return( m_val.begin() ) ; }
      sr::wxString_cit end() { return( m_val.end() ) ; }
      /*--( Init for strings that can't be empty )-------------------------*/
      void init_mandatory( const wxString &s_key,
                           const wxString &def_val = wxString()
                         )
      {  init( s_key, def_val ) ;
         if( empty() ) { set( get_def() ) ; }
      }

   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! I could have defined the wxRect specialized functions but it doesn't seem !
! to work perfectly under gcc (or maybe it's me!)                           !
+--------------------------------------------------------------------------*/
class CConfigRect : public CConfigBaseElement< wxRect >
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CConfigRect() : CConfigBaseElement< wxRect >()
      {  ; }
      CConfigRect( const wxString &s_key )
                 : CConfigBaseElement< wxRect >( s_key )
      {  ; }
   /*----------------------------------------------------------------------*/
   public :
      /*--( For this type, default value will be "specific" )--------------*/
      void init( const wxString &s_key,
                 const wxRect &def_val
                            = wxRect( wxDefaultPosition, wxDefaultSize )
               )
      {  CConfigBaseElement< wxRect >::init( s_key, def_val ) ; }
      /*-------------------------------------------------------------------*/
      void write_val()
      {
         /*----------------------------------------------------------------*/
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }
         /*----------------------------------------------------------------*/
         p_conf->Write( m_s_key,
                        wxString::Format( "%d %d %d %d",
                                          m_val.x, m_val.y,
                                          m_val.width, m_val.height
                                        )
                      ) ;
         /*----------------------------------------------------------------*/
         m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      void read_val( const wxRect &def_val )
      {
         /*----------------------------------------------------------------*/
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }
         /*--( Get file data )---------------------------------------------*/
         wxString s_val ;
         if( !p_conf->Read( m_s_key, &s_val ) )
         {  m_val = def_val ; }
         else
         {  wxSscanf( s_val, "%d %d %d %d",
                      &m_val.x, &m_val.y, &m_val.width, &m_val.height
                    ) ;
         }
         /*----------------------------------------------------------------*/
         m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CConfigArrayString : public CConfigBaseElement< wxArrayString >
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CConfigArrayString() : CConfigBaseElement< wxArrayString >(),
                             m_i_nb_elmt_loaded( 0 ), m_i_max_nb_elmt( -1 )
      {  ; }
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      int m_i_nb_elmt_loaded ;
      int m_i_max_nb_elmt    ; // < 0 means "no limit"
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void init( const wxString &s_key, int i_max_nb_elmt = -1 )
      {  m_i_max_nb_elmt = i_max_nb_elmt ;
         CConfigBaseElement< wxArrayString >::init( s_key, wxArrayString() );
      }
      /*-------------------------------------------------------------------*/
      bool IsEmpty() const { return( m_val.IsEmpty() ) ; }
      int  Count() const { return( m_val.Count() ) ; }
      int  Index( const wxString &s, bool boo_case_sensitive ) const
      {  return( m_val.Index( s, boo_case_sensitive ) ) ; }
      wxString & operator[]( size_t i_num ) { return( m_val[ i_num ] ) ; }
      /*-------------------------------------------------------------------*/
      void write_val()
      {
         /*----------------------------------------------------------------*/
         int i_nb_elmt_to_save ;
         int i_num ;
         /*----------------------------------------------------------------*/
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }

         /*----------------------------------------------------------------*/
         i_nb_elmt_to_save = ( int )m_val.GetCount() ;
         if( m_i_max_nb_elmt > 0 && i_nb_elmt_to_save > m_i_max_nb_elmt )
         {  i_nb_elmt_to_save = m_i_max_nb_elmt ; }

         /*--( Write what has been asked for )-----------------------------*/
         for( i_num = 0 ; i_num < i_nb_elmt_to_save ; ++i_num )
         {  p_conf->Write( wxString::Format( m_s_key, i_num + 1 ),
                           m_val[ i_num ]
                         ) ;
         }
         /*--( And delete the rest )---------------------------------------*/
         for( ; i_num < m_i_nb_elmt_loaded ; ++i_num )
         {  p_conf->DeleteEntry( wxString::Format( m_s_key, i_num + 1 ) ) ; }
         /*----------------------------------------------------------------*/
         m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      void read_val( const wxArrayString & WXUNUSED( def_val ) )
      {  read_val() ; }
      /*-------------------------------------------------------------------*/
      void read_val()
      {
         /*----------------------------------------------------------------*/
         wxString     s_val ;
         wxConfigBase *p_conf = wxConfigBase::Get() ;
         if( p_conf == NULL ) { return ; }
         /*--( Get file data )---------------------------------------------*/
         for( m_i_nb_elmt_loaded = 0 ;
                 (    m_i_max_nb_elmt < 0
                   || m_i_nb_elmt_loaded < m_i_max_nb_elmt
                 )
              && p_conf->Read( wxString::Format( m_s_key,
                                                 m_i_nb_elmt_loaded + 1
                                               ),
                               &s_val
                             ) ;
              ++m_i_nb_elmt_loaded
            )
         {  m_val.Add( s_val ) ; }
         /*----------------------------------------------------------------*/
         m_boo_to_save = false ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      void init() { read_val() ; }
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
!                                                                           !
! S corresponds to the elementary type used by the type T                   !
! What is the difference between CConfigArrayString and                     !
! CConfigTab< CConfigString, wxString > ?                                   !
!                                                                           !
! The interest of CConfigTab is that each element of the array is a config  !
! base class derived class.                                                 !
! So only the modified elements are save.                                   !
! For CConfigArrayString, as wxArayString is the base element, if any       !
! string is modified the whole array is saved.                              !
!                                                                           !
+--------------------------------------------------------------------------*/
template < class T, class S >
class CConfigTab
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      //CConfigTab() { ; }
      /*-------------------------------------------------------------------*/
      //~CConfigTab() { /*save()*/ ; }
   /*----------------------------------------------------------------------*/
   protected :
      /*-------------------------------------------------------------------*/
      wxString m_s_key         ;
      int      m_i_max_nb_elmt ;
      S        m_val_stop_init ;
      /*--------------------------------------------------------------------+
      ! All the elements have not to be kept. The rest (above this value)   !
      ! has to be deleted.                                                  !
      +--------------------------------------------------------------------*/
      int      m_i_nb_elmt_to_save ;
      /*-------------------------------------------------------------------*/
      typedef  std::vector< T > t_tab  ;
      /*-------------------------------------------------------------------*/
      t_tab    m_tab ;
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      int init_new_val( T &pval )
      {
         /*----------------------------------------------------------------*/
         if( ( int )m_tab.size() == m_i_max_nb_elmt ) { return( -1 ) ; }
         pval.init( wxString::Format( m_s_key, ( int )m_tab.size() + 1 ) ) ;
         /*----------------------------------------------------------------*/
         return( 0 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      void read_val()
      {
         /*----------------------------------------------------------------*/
         T pval ;
         /*----------------------------------------------------------------*/
         clear() ;
         /*--( Once reached the max number, the "init" won't return 0 ... )*/
         while(    init_new_val( pval ) == 0
                && ( m_i_max_nb_elmt >= 0 || pval.get() != m_val_stop_init )
              )
         {  m_tab.push_back( pval ) ; }
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int size() { return( m_tab.size() ) ; }
      /*-------------------------------------------------------------------*/
      T *get_pval( int i_num )
      {  return( i_num < (int)m_tab.size() ? &m_tab[ i_num ] : NULL ) ; }
      /*-------------------------------------------------------------------*/
      S get( int i_num ) const
      {
         /*----------------------------------------------------------------*/
         if( i_num < ( int )m_tab.size() )
         {  return( m_tab[ i_num ].get() ) ; }
         /*----------------------------------------------------------------*/
         return( S() ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int add_val( const S &val )
      {
         /*----------------------------------------------------------------*/
         T   pval  ;
         int i_ret ;
         /*----------------------------------------------------------------*/
         if( ( i_ret = init_new_val( pval ) ) != 0 )
         {  return( i_ret ) ; }
         /*----------------------------------------------------------------*/
         pval.set( val ) ;
         m_tab.push_back( pval ) ;
         /*----------------------------------------------------------------*/
         return( 0 ) ;
         /*----------------------------------------------------------------*/
      }
      /*--( A value can be set at any index )------------------------------*/
      int set( int i_num, const S &val )
      {
         /*----------------------------------------------------------------*/
         if( i_num < ( int )m_tab.size() )
         {  m_tab[ i_num ].set( val ) ; return( 0 ) ; }
         /*----------------------------------------------------------------*/
         return( add_val( val ) ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int get_to_save( int i_num, bool &boo_to_save )
      {
         /*----------------------------------------------------------------*/
         if( i_num < m_tab.size() )
         {  boo_to_save = m_tab[ i_num ].get_to_save() ; return( 0 ) ; }
         /*----------------------------------------------------------------*/
         return( -1 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int set_to_save( int i_num, bool boo_val )
      {
         /*----------------------------------------------------------------*/
         if( i_num < m_tab.size() )
         {  m_tab[ i_num ].set_to_save( boo_val ) ; return( 0 ) ; }
         /*----------------------------------------------------------------*/
         return( -1 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int  size() const { return( m_tab.size() ) ; }
      void clear() { m_tab.clear() ; }
      int  get_max_nb_elmt() const { return( m_i_max_nb_elmt ) ; }
      void set_max_nb_elmt( int i_val ) { m_i_max_nb_elmt = i_val ; }
      /*-------------------------------------------------------------------*/
      int  get_nb_elmt_to_save() const
      {  return( m_i_nb_elmt_to_save ) ; }
      void set_nb_elmt_to_save( int i_val )
      {  m_i_nb_elmt_to_save = i_val ; }
      /*-------------------------------------------------------------------*/
      void init( const wxString &s_key, int i_max_nb_elmt, S val_stop_init )
      {
         /*----------------------------------------------------------------*/
         m_s_key         = s_key         ;
         m_i_max_nb_elmt = i_max_nb_elmt ;
         m_val_stop_init = val_stop_init ;
         /*----------------------------------------------------------------*/
         read_val() ;
         /*--( All is supposed to be saved )-------------------------------*/
         m_i_nb_elmt_to_save = m_tab.size() ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      void save()
      {
         /*----------------------------------------------------------------*/
         int i_num ;
         /*--( Save what has been asked for )------------------------------*/
         for( i_num = 0 ; i_num < m_i_nb_elmt_to_save ; ++i_num )
         {  m_tab[ i_num ].save() ; }
         /*--( And delete the rest if any ... )----------------------------*/
         for( ; i_num < ( int )m_tab.size() ; ++i_num )
         {  m_tab[ i_num ].del() ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int search_val( const S &val ) const
      {
         /*----------------------------------------------------------------*/
         int i_pos ;
         /*-----------------------------------------------------------------+
         ! "std::find" is not used because the stored elements are          !
         ! CConfigXXXX ones.                                                !
         +-----------------------------------------------------------------*/
         for( i_pos = 0 ;
              i_pos < ( int )m_tab.size() && val != get( i_pos ) ;
              ++i_pos
            )
         {  ; }
         /*----------------------------------------------------------------*/
         return( i_pos < ( int )m_tab.size() ? i_pos : -1 ) ;
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
typedef CConfigTab< CConfigString, wxString > CConfigTabString ;
typedef CConfigTab< CConfigInt   , int      > CConfigTabInt    ;

/*-------------------------------------------------------------------------*/
void CConfig_init( const wxString &s_file, const int i_version,
                   bool &boo_conv_from_v2
                 ) ;

/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#endif // CCONFIG_H
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                        End of file CConfig.h                              !
+==========================================================================*/
